/* 
 * All content copyright Terracotta, Inc., unless otherwise indicated. All rights reserved.
 * Copyright IBM Corp. 2024, 2025
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not 
 * use this file except in compliance with the License. You may obtain a copy 
 * of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 *   
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT 
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the 
 * License for the specific language governing permissions and limitations 
 * under the License.
 * 
 */


package org.quartz.impl.jdbcjobstore;

import org.quartz.JobPersistenceException;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Locale;

/**
 * <p>
 * This class contains utility functions for use in all delegate classes.
 * </p>
 * 
 * @author <a href="mailto:jeff@binaryfeed.org">Jeffrey Wescott</a>
 */
public final class Util {

    /**
     * Private constructor because this is a pure utility class.
     */
    private Util() {
    }
    
    /*
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     * 
     * Interface.
     * 
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     */

    /**
     * <p>
     * Replace the table prefix in a query by replacing any occurrences of
     * "{0}" with the table prefix.
     * </p>
     * 
     * @param query
     *          the unsubstituted query
     * @param tablePrefix
     *          the table prefix
     * @return the query, with proper table prefix substituted
     */
    public static String rtp(String query, String tablePrefix, String schedNameLiteral) {
        return MessageFormat.format(query, new Object[]{tablePrefix, schedNameLiteral});
    }

    /**
     * <p>
     * Obtain a unique key for a given job.
     * </p>
     * 
     * @param jobName
     *          the job name
     * @param groupName
     *          the group containing the job
     * @return a unique <code>String</code> key
     */
    static String getJobNameKey(String jobName, String groupName) {
        return (groupName + "_$x$x$_" + jobName).intern();
    }

    /**
     * <p>
     * Obtain a unique key for a given trigger.
     * </p>
     * 
     * @param triggerName
     *          the trigger name
     * @param groupName
     *          the group containing the trigger
     * @return a unique <code>String</code> key
     */
    static String getTriggerNameKey(String triggerName, String groupName) {
        return (groupName + "_$x$x$_" + triggerName).intern();
    }
    
    /**
     * Cleanup helper method that closes the given <code>ResultSet</code>
     * while ignoring any errors.
     */
    public static void closeResultSet(ResultSet rs) {
        if (null != rs) {
            try {
                rs.close();
            } catch (SQLException ignore) {
            }
        }
    }

    /**
     * Cleanup helper method that closes the given <code>Statement</code>
     * while ignoring any errors.
     */
    public static void closeStatement(Statement statement) {
        if (null != statement) {
            try {
                statement.close();
            } catch (SQLException ignore) {
            }
        }
    }
    
    
    public static void setBeanProps(Object obj, String[] propNames, Object[] propValues)  throws JobPersistenceException {
        
        if(propNames == null || propNames.length == 0)
            return;
        if(propNames.length != propValues.length)
            throw new IllegalArgumentException("propNames[].length != propValues[].length");
        
        String name = null;
        
        try {
            BeanInfo bi = Introspector.getBeanInfo(obj.getClass());
            PropertyDescriptor[] propDescs = bi.getPropertyDescriptors();
        
            for(int i=0; i < propNames.length; i++) {
                name = propNames[i];
                String c = name.substring(0, 1).toUpperCase(Locale.US);
                String methName = "set" + c + name.substring(1);
        
                java.lang.reflect.Method setMeth = getSetMethod(methName, propDescs);
        
                if (setMeth == null) {
                    throw new NoSuchMethodException(
                            "No setter for property '" + name + "'");
                }
    
                Class<?>[] params = setMeth.getParameterTypes();
                if (params.length != 1) {
                    throw new NoSuchMethodException(
                        "No 1-argument setter for property '" + name + "'");
                }
                
                setMeth.invoke(obj, new Object[]{ propValues[i] });
            }
        }
        catch(Exception e) {
            throw new JobPersistenceException(
                "Unable to set property named: " + name +" of object of type: " + obj.getClass().getCanonicalName(), 
                e); 
        }
    }

    private static java.lang.reflect.Method getSetMethod(String name, PropertyDescriptor[] props) {
        for (PropertyDescriptor prop : props) {
            java.lang.reflect.Method wMeth = prop.getWriteMethod();

            if (wMeth != null && wMeth.getName().equals(name)) {
                return wMeth;
            }
        }
    
        return null;
    }

    static boolean containsColumnNames(ResultSet rs, String... colNames) throws SQLException {
        ResultSetMetaData rsmd = rs.getMetaData();
        int columnCount = rsmd.getColumnCount();

        for (int i = 1; i <= columnCount; i++ ) {
            String name = rsmd.getColumnName(i);
            for (String colName : colNames) {
                if (colName.equalsIgnoreCase(name)) {
                    return true;
                }
            }
            // Do stuff with name
        }
        return false;
    }

    static String getString(ResultSet resultSet, String... columnNames) throws SQLException {
        for (String columnName : columnNames) {
            if (containsColumnNames(resultSet, columnName)) {
                return resultSet.getString(columnName);
            }
        }
        throw new SQLException("Missing columns in result set: " + Arrays.toString(columnNames));
    }

    static boolean getBoolean(ResultSet resultSet, String... columnNames) throws SQLException {
        for (String columnName : columnNames) {
            if (containsColumnNames(resultSet, columnName)) {
                return resultSet.getBoolean(columnName);
            }
        }
        throw new SQLException("Missing columns in result set: " + Arrays.toString(columnNames));
    }

    /**
     * Checks if all columns are null in the result set.
     * @param resultSet the result set
     * @param columnNames columns names to check
     * @return true if all columns are null, false otherwise
     * @throws SQLException
     */
    static boolean areNull(ResultSet resultSet, String... columnNames) throws SQLException {
        //TODO: check if resultSet is closed?
        for (String columnName : columnNames) {
            if (containsColumnNames(resultSet, columnName)) {
                if (resultSet.getObject(columnName) != null) {
                    return false;
                }
            }
        }
        return true;
    }

}

// EOF
