package org.nuiton.topia.flyway;

/*
 * #%L
 * ToPIA :: Flyway integration service
 * $Id: TopiaFlywayServiceImpl.java 3103 2014-05-14 16:01:51Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-4/topia-service-flyway/src/main/java/org/nuiton/topia/flyway/TopiaFlywayServiceImpl.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import com.googlecode.flyway.core.Flyway;
import com.googlecode.flyway.core.util.Location;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaApplicationContext;
import org.nuiton.topia.persistence.TopiaConfigurationConstants;
import org.nuiton.topia.persistence.TopiaException;

import java.io.IOException;
import java.net.URL;
import java.util.Enumeration;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

/**
 * Implementation for {@link org.nuiton.topia.flyway.TopiaFlywayService}.
 *
 * @since 3.0
 */
public class TopiaFlywayServiceImpl implements TopiaFlywayService {

    private static final Log log = LogFactory.getLog(TopiaFlywayServiceImpl.class);

    /**
     * Value for {@link #FLYWAY_INIT_VERSION} configuration parameter.
     */
    protected String flywayInitVersion = null;

    @Override
    public void setConfiguration(Map<String, String> serviceConfiguration) {
        // set flywayInitVersion is provided
        if (serviceConfiguration.containsKey(FLYWAY_INIT_VERSION)) {
            flywayInitVersion = serviceConfiguration.get(FLYWAY_INIT_VERSION);
            if (StringUtils.isBlank(flywayInitVersion)) {
                throw new IllegalArgumentException("'" + flywayInitVersion + "' must not be empty");
            }
        }
    }

    @Override
    public Class<?>[] getPersistenceClasses() {
        return new Class<?>[0];
    }

    @Override
    public void preInit(TopiaApplicationContext topiaApplicationContext) {

        if (log.isInfoEnabled()) {
            log.info("init flyway service");
        }

        Flyway flyway = new Flyway();

        setDataSource(flyway, topiaApplicationContext);

        setLocations(flyway, topiaApplicationContext);

        doExtraConfiguration(flyway, topiaApplicationContext);

        if (isSchemaEmpty(flyway, topiaApplicationContext)) {

            createSchema(flyway, topiaApplicationContext);

        } else {

            migrateSchema(flyway, topiaApplicationContext);

        }

    }

    /**
     * Define flyway database credentials.
     *
     * This implementation search for parameters given in
     * {@link org.nuiton.topia.persistence.TopiaApplicationContext#getConfiguration()}. We use
     * the same credentials to migrate the database as the one used when we use it.
     */
    protected void setDataSource(Flyway flyway, TopiaApplicationContext topiaApplicationContext) {

        Map<String, String> configuration = topiaApplicationContext.getConfiguration();

        String url = configuration.get(TopiaConfigurationConstants.CONFIG_URL);
        String user = configuration.get(TopiaConfigurationConstants.CONFIG_USER);
        String password = configuration.get(TopiaConfigurationConstants.CONFIG_PASS);

        flyway.setDataSource(url, user, password);

    }

    /**
     * Define where Flyway should look for migrations.
     *
     * This implementation search for *.sql migration files in "db/migration" resources directory
     * and for JDBC migrations in package.to.ApplicationContext<strong>.migration</strong> package.
     */
    protected void setLocations(Flyway flyway, TopiaApplicationContext topiaApplicationContext) {

        String classpathMigrationPackage = topiaApplicationContext.getClass().getPackage().getName() + ".migration";
        ImmutableSet<String> defaultLocations = ImmutableSet.of("db/migration", classpathMigrationPackage);

        // detects migrations and configure flyway locations
        Set<String> locations = new LinkedHashSet<String>();
        for (String defaultLocation : defaultLocations) {
            if (log.isInfoEnabled()) {
                log.info("will search for migration in location " + defaultLocation);
            }
            try {
                Location location = new Location(defaultLocation);
                Enumeration<URL> resources = Thread.currentThread().getContextClassLoader().getResources(location.getPath());
                if (resources.hasMoreElements()) {
                    locations.add(defaultLocation);
                    if (log.isInfoEnabled()) {
                        log.info("migrations found in " + defaultLocation);
                    }
                } else {
                    if (log.isInfoEnabled()) {
                        log.info("no migration found in " + defaultLocation);
                    }
                }
            } catch (IOException e) {
                throw new TopiaException(e);
            }
        }

        String[] locationsArray = locations.toArray(new String[locations.size()]);
        flyway.setLocations(locationsArray);
    }

    /**
     * Opened hook to override in a sub-class.
     */
    protected void doExtraConfiguration(Flyway flyway, TopiaApplicationContext topiaApplicationContext) {

    }

    /**
     * Check if schema exists. If true is returned, we won't use any migrations and just create shema
     * according to model. If false, a schema already exists and an migration should be tried.
     */
    protected boolean isSchemaEmpty(Flyway flyway, TopiaApplicationContext topiaApplicationContext) {

        boolean schemaEmpty = topiaApplicationContext.isSchemaEmpty();

        return schemaEmpty;

    }

    protected void createSchema(Flyway flyway, TopiaApplicationContext topiaApplicationContext) {

        String initVersion = topiaApplicationContext.getModelVersion();

        if (log.isInfoEnabled()) {
            log.info("schema is empty, ignore migrations and let topia create schema");
        }

        topiaApplicationContext.createSchema();

        if (log.isInfoEnabled()) {
            log.info("init flyway to version " + initVersion);
        }

        flyway.setInitVersion(initVersion);
        flyway.setInitDescription("schema creation called on application context by topia flyway service");
        flyway.init();

    }

    protected void migrateSchema(Flyway flyway, TopiaApplicationContext topiaApplicationContext) {

        if (flywayInitVersion == null) {
            if (log.isDebugEnabled()) {
                log.debug("schema exists, no flywayInitVersion found, let suppose flyway is already initialized");
            }
            // if flyway is not initialized, we will get
//            Grave: Exception sending context initialized event to listener instance of class fr.ifremer.wao.web.WaoApplicationListener
//            com.googlecode.flyway.core.api.FlywayException: Found non-empty schema "public" without metadata table! Use init() or set initOnMigrate to true to initialize the metadata table.
//                    at com.googlecode.flyway.core.Flyway$1.execute(Flyway.java:848)
//            at com.googlecode.flyway.core.Flyway$1.execute(Flyway.java:819)
//            at com.googlecode.flyway.core.Flyway.execute(Flyway.java:1200)
//            at com.googlecode.flyway.core.Flyway.migrate(Flyway.java:819)

        } else {
            if (log.isDebugEnabled()) {
                log.debug("schema exists, will ask flyway to init if necessary to version " + flywayInitVersion);
            }
            flyway.setInitOnMigrate(true);
            flyway.setInitVersion(flywayInitVersion);
        }

        String targetVersion = topiaApplicationContext.getModelVersion();

        if (log.isInfoEnabled()) {
            log.info("schema exists, will run flyway migration up to target version " + targetVersion);
        }

        flyway.setTarget(targetVersion);

        if (log.isInfoEnabled()) {
            log.info("run flyway migration");
        }

        flyway.migrate();
    }

    @Override
    public void postInit(TopiaApplicationContext applicationContext) {
        // nothing to do
    }

}
