
package org.nuiton.topia.generator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.i18n.I18n;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;





/**
 * This Template is used to create the skeleton of services for a final
 * application which using Topia. 
 * <div>
 * Generation from interfaces with stereotype &lt;&lt;service&gt;&gt; :
 *  <ul>
 *    <li>Service : interface of the service defined in model.</li>
 *    <li><p>ServiceAbstract : abstract class which contains :</p>
 *        <p>* treateError : abstract method used to catch all exception from a
 *          service method.</p>
 *        <p>* closeTransaction : abstract method used to finally the try/catch
 *          of a service method</p>
 *        <p>* beginTransaction : abstract method used to start the transaction
 *          using rootContext.</p>
 *        <p>* constructor with AppContextImplementor in argument</p>
 *        <p>* for each method : the implementation of the method (skeleton with
 *          try/catch and beginTransaction call to open a new TopiaContext from
 *          AppContextImplementor). Usage of i18n keys for error messages in
 *          exception.</p>
 *        <p>* for each method : an abstract method used to execute the business
 *          code of the method : need to be implemented in subclass.</p>
 *    </li>
 *  </ul>
 * </div>
 * <div>
 * Exemple of ServiceImpl utils method implementation. (The AppException
 * is considered if defined in model tagvalue "exceptionClass") : <br />
 * <pre>
 *   public class ServiceImpl implements ServiceAbstract  {
 *
 *      // properties for Topia configuration
 *      protected Properties properties;
 *      ...
 * 
 *      &#64;Override
 *      public void treateError(TopiaContext transaction, Exception eee,
 *              String message, Object... args) throws AppException {
 *
 *          // Note that the message from service doesn't directly use _() for
 *          // i18 messages but n_(). In this log, the _() is used to translate
 *          // correctly the message. But the message must be translate when
 *          // catching the AppException in UI.
 *          if (log.isErrorEnabled()) {
 *              log.error(_(message, args), eee);
 *          }
 *
 *          // rollback of current transaction
 *          if (transaction != null) {
 *              try {
 *                  transaction.rollbackTransaction();
 *              } catch (TopiaException ex) {
 *                  if (log.isErrorEnabled()) {
 *                      log.error(_("app.error.context.rollback"), ex);
 *                  }
 *              }
 *          }
 *          // wrapping the exception in a AppException with message and
 *          // arguments for i18n translation
 *          throw new AppException(eee, message, args);
 *      }
 *
 *      &#64;Override
 *      public void closeTransaction(TopiaContext transaction) {
 *          if (transaction != null) {
 *              try {
 *                  transaction.closeContext();
 *              } catch (TopiaException eee) {
 *                  if (log.isErrorEnabled()) {
 *                      log.error(_("app.error.context.close"), eee);
 *                  }
 *              }
 *          }
 *      }
 *
 *      &#64;Override
 *      public TopiaContext beginTransaction() throws TopiaException {
 *          TopiaContext rootContext = null;
 *          try {
 *              // You have to manage the properties using ApplicationConfig
 *              // or other lib to have configuration for Topia
 *              rootContext = TopiaContextFactory.getContext(properties);
 *
 *              return getTopiaRootContext().beginTransaction();
 *
 *          // only catch exception for rootContext
 *          } catch (TopiaNotFoundException eee) {
 *              treateError(eee, n_("app.error.context.getTopiaRootContext"));
 *          }
 *          return null;
 *      }
 *
 *      // Implementation of abstract method, the interface method is
 *      // called 'createMyEntity(MyEntity entity)' in this case.
 *      &#64;Override
 *      public void executeCreateMyEntity(TopiaContext transaction, 
 *                      MyEntity entity) throws TopiaException {
 *
 *          MyEntityDAO dao = AppDAOHelper.getMyEntityDAO(transaction);
 *          dao.create(entity);
 *          // That's it, no need to manage errors or transaction, the abstract
 *          // service will do this job.
 *      }
 * }
 * </pre>
 * <div>
 * <h2>TAG_TRANSACTION</h2>
 * <p>Default value : true</p>
 * <p>You can use the tagValue 'transaction=false' to specify that a method
 * doesn't need any TopiaContext, so no need to instantiate a new one.
 * This tagValue can only be put directly in the model and not in properties
 * file (because of multiple methods with same name problem).</p>
 * </div>
 * <div>
 * <h2>TAG_ERROR_ARGS</h2>
 * <p>Default value : false</p>
 * <p>You can use the tagValue 'errorArgs=true' to specify that a method
 * need arguments for error message. This tagValue can only be put directly
 * in the model and not in properties file.</p>
 * </div>
 * <div>
 * <h2>TAG_EXCEPTION_CLASS</h2>
 * <p>Default value : null</p>
 * <p>You can use the tagValue 'exceptionClass=my.exception.full.qualified.Name'
 * to specify that all contract methods will throw this exception.</p>
 * </div>
 * <p>It is smooth, isn't it :p ?</p>
 * <p>TODO : may be refactor to integrate JTA or webservice or may be not in this
 * transformer.</p>
 * <p>TODO : find a good way to change log level</p>
 *
 * Created: 23 mars 2010
 *
 * @author fdesbois
 * @version $Revision: 1871 $
 *
 * Mise a jour: $Date: 2010-04-02 17:44:28 +0200 (ven., 02 avril 2010) $
 * par : $Author$
 */
public class ServiceTransformer extends ObjectModelTransformerToJava {

    protected String modelName;

    protected String defaultPackageName;

    protected String exceptionName;

    private static final String OP_NAME_BEGIN_TRANSACTION = "beginTransaction";

    private static final String OP_NAME_CLOSE_TRANSACTION = "closeTransaction";

    private static final String OP_NAME_TREATE_ERROR = "treateError";

//    protected String getContextInterfaceName() {
//        return StringUtils.capitalize(modelName) + "Context";
//    }
//
//    protected String getContextImplementorInterfaceName() {
//        return getContextInterfaceName() + "Implementor";
//    }
//
//    protected String getExceptionClassName() {
//        return StringUtils.capitalize(modelName) + "Exception";
//    }

    protected String getServiceAbstractClassName(String serviceName) {
        return serviceName + "Abstract";
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        exceptionName = 
                model.getTagValue(TopiaGeneratorUtil.TAG_EXCEPTION_CLASS);
        modelName = model.getName();
    }

    @Override
    public void transformFromInterface(ObjectModelInterface interfacez) {
        if (!interfacez.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_SERVICE)) {
            return;
        }

        ObjectModelInterface serviceContract = 
                createServiceContract(interfacez);

        createServiceAbstract(interfacez, serviceContract);
    }

    /**
     * Create the service contract using {@code source} interface defined
     * in model.
     *
     * @param source interface from model
     * @return the ObjectModelInterface created
     */
    protected ObjectModelInterface createServiceContract(
            ObjectModelInterface source) {

        ObjectModelInterface serviceContract =
                createInterface(source.getName(),
                                     source.getPackageName());

        setDocumentation(serviceContract, source.getDocumentation());
        for (ObjectModelOperation op : source.getOperations()) {
            ObjectModelOperation newOp =
                    addOperation(serviceContract,
                                      op.getName(), op.getReturnType());
            setDocumentation(newOp.getReturnParameter(),
                             op.getReturnParameter().getDocumentation());
            for (ObjectModelParameter param : op.getParameters()) {
                ObjectModelParameter newParam =
                        addParameter(newOp, param.getType(),
                                          param.getName());
                setDocumentation(newParam, param.getDocumentation());
            }
            for (String ex : op.getExceptions()) {
                addException(newOp, ex);
            }
            if (exceptionName != null) {
                addException(newOp, exceptionName);
            }
            setDocumentation(newOp, op.getDocumentation());
        }
        return serviceContract;
    }

    /**
     * Create the service abstract for {@code serviceContract}
     * using {@code source} interface defined
     * in model.
     *
     * @param source interface from model
     * @param serviceContract to implement
     */
    protected void createServiceAbstract(ObjectModelInterface source,
            ObjectModelInterface serviceContract) {

        ObjectModelClass serviceAbstract = createAbstractClass(
                getServiceAbstractClassName(serviceContract.getName()),
                        serviceContract.getPackageName());

        // Implements contract interface
        addInterface(serviceAbstract, serviceContract.getQualifiedName());

        // Add Logger attribute
        // FIXME in EUGene, we want the default value not to be managed
        // for import.
//        addAttribute(service, "log",
//                Log.class,
//                "LogFactory.getLog(" + interfacez.getName() + ".class)",
//                ObjectModelModifier.PRIVATE,
//                ObjectModelModifier.STATIC,
//                ObjectModelModifier.FINAL);
        addAttribute(serviceAbstract, "log",
                Log.class, null,
                ObjectModelModifier.PRIVATE,
                ObjectModelModifier.FINAL);

        addImport(serviceAbstract, Log.class);
        addImport(serviceAbstract, LogFactory.class);

        // Constructor
        ObjectModelOperation constructor =
                addConstructor(serviceAbstract, ObjectModelModifier.PUBLIC);
        setOperationBody(constructor, ""
    +"\n"
+"        // FIXME : must be fixed attribute value in EUGene\n"
+"        this.log = LogFactory.getLog("+serviceAbstract.getName()+".class);\n"
+"    "
        );

        // Create abstract methods
        ObjectModelOperation beginTransaction =
                addOperation(serviceAbstract, OP_NAME_BEGIN_TRANSACTION,
                        TopiaContext.class,
                        ObjectModelModifier.ABSTRACT,
                        ObjectModelModifier.PROTECTED);
        addException(beginTransaction, TopiaException.class);

        ObjectModelOperation treateError1 =
                addOperation(serviceAbstract, OP_NAME_TREATE_ERROR, "void",
                        ObjectModelModifier.ABSTRACT,
                        ObjectModelModifier.PROTECTED);
        addParameter(treateError1, Exception.class, "eee");
        addParameter(treateError1, String.class, "message");
        addParameter(treateError1, "Object...", "args");
        if (exceptionName != null) {
            addException(treateError1, exceptionName);
        }

        ObjectModelOperation treateError2 =
                addOperation(serviceAbstract, OP_NAME_TREATE_ERROR, "void",
                        ObjectModelModifier.ABSTRACT,
                        ObjectModelModifier.PROTECTED);
        addParameter(treateError2, TopiaContext.class, "transaction");
        addParameter(treateError2, Exception.class, "eee");
        addParameter(treateError2, String.class, "message");
        addParameter(treateError2, "Object...", "args");
        if (exceptionName != null) {
            addException(treateError2, exceptionName);
        }

        ObjectModelOperation closeTransaction =
                addOperation(serviceAbstract, OP_NAME_CLOSE_TRANSACTION, "void",
                        ObjectModelModifier.ABSTRACT,
                        ObjectModelModifier.PROTECTED);
        addParameter(closeTransaction, TopiaContext.class, "transaction");

        // Create abstract execute methods
        Map<String, ObjectModelOperation> abstOps =
                new HashMap<String, ObjectModelOperation>();

        // Use source interfacez to have tagvalues from model
        for (ObjectModelOperation op : source.getOperations()) {
            ObjectModelOperation abstOp =
                    createOperationExecuteAbstract(serviceAbstract, op);
            // Keep abstract methods to use them in operationImplementation
            // generation
            abstOps.put(op.getName(), abstOp);
        }

        // Imports for implementations
        addImport(serviceAbstract, TopiaContext.class);
        addImport(serviceAbstract, I18n.class);
        addImport(serviceAbstract, ArrayList.class);
        addImport(serviceAbstract, DurationFormatUtils.class);

        // Create abstract execute methods
        for (ObjectModelOperation op : source.getOperations()) {
            createOperationImplementation(serviceAbstract,
                    abstOps.get(op.getName()), op, source.getName());
        }
    }

    /**
     * Create an operation abstract to execute in contract implementation.
     * You can use tagvalues "errorArgs" (default = false) and "transaction"
     * (default = true) to generate appropriate parameters. This abstract
     * method will throw all exceptions (Exception.class). This is the method
     * which will be implemented by the developper in service implementation
     * class.
     *
     * @param serviceAbstract where the operation will be created
     * @param source ObjectModelOperation from model
     * @return the abstract operation created
     * @see #isErrorArgsNeeded(org.nuiton.eugene.models.object.ObjectModelOperation)
     * @see #isTransactionNeeded(org.nuiton.eugene.models.object.ObjectModelOperation)
     */
    protected ObjectModelOperation createOperationExecuteAbstract(
            ObjectModelClass serviceAbstract, ObjectModelOperation source) {
        String opName = StringUtils.capitalize(source.getName());

        // Abstract operation to execute method content
        ObjectModelOperation abstOp =
                addOperation(serviceAbstract, "execute" + opName,
                        source.getReturnType(),
                        ObjectModelModifier.ABSTRACT,
                        ObjectModelModifier.PROTECTED);

        // Throw all exception from abstract method
        // They will be catched by interface method to use treateError
        addException(abstOp, Exception.class);

        if (isTransactionNeeded(source)) {
            addParameter(abstOp, TopiaContext.class, "transaction");
        }

        if (isErrorArgsNeeded(source)) {
            // Add errorArgs to abstract operation
            addParameter(abstOp,
                    "java.util.List<Object>", "errorArgs");
        }

        // Copy other operation parameters
        for (ObjectModelParameter param : source.getParameters()) {
            addParameter(abstOp, param.getType(), param.getName());
        }
        return abstOp;
    }

    /**
     * Create an operation implementation. This is the skeleton of the operation
     * defined from model. This will put a try/catch block over an abstract
     * method {@code abstOp}. You can use tagvalues "errorArgs" and
     * "transaction" for abstract method parameters to call. If the transaction
     * is needed, this will use the beginTransaction() and closeTransaction()
     * methods defined in {@code serviceAbstract} class.
     *
     * @param serviceAbstract where the operation will be created
     * @param abstOp to execute into the implementation body
     * @param source ObjectModelOperation from model
     * @param serviceContractName where the signature method is defined
     * @see #isErrorArgsNeeded(org.nuiton.eugene.models.object.ObjectModelOperation)
     * @see #isTransactionNeeded(org.nuiton.eugene.models.object.ObjectModelOperation) 
     */
    protected void createOperationImplementation(
            ObjectModelClass serviceAbstract,
            ObjectModelOperation abstOp, ObjectModelOperation source,
            String serviceContractName) {

        // boolean to specify if the method need a transaction or not
        // Default set to true but can be override by a tagvalue on the
        // method
        boolean needTransaction = isTransactionNeeded(source);

        // boolean to specify if the method need error arguments or not
        // Default set to true but can be override by a tagvalue on the
        // method
        boolean needErrorArgs = isErrorArgsNeeded(source);

        // Implementation of interface operation
        ObjectModelOperation implOp =
                addOperation(serviceAbstract, source.getName(),
                                    source.getReturnType(),
                                    ObjectModelModifier.PUBLIC);
        addAnnotation(serviceAbstract, implOp, Override.class.getSimpleName());

        String toStringAppend = "";
        String separatorLog = " : ";
        // Copy operation parameters
        for (ObjectModelParameter param : source.getParameters()) {
            String paramName = param.getName();
            addParameter(implOp, param.getType(), paramName);

            // Prepare Log
            toStringAppend +=
                    "\n\t\t\t.append(\"" + separatorLog + paramName + " = \")" +
                          ".append(" + paramName + ")";
            separatorLog = " _ ";
        }

        // Use buffer for operation body
        StringBuilder buffer = new StringBuilder();

        // Abstract operation parameters
        String abstName = abstOp.getName();
        String abstParams = "";
        String separator = "";
        for (ObjectModelParameter param : abstOp.getParameters()) {
            abstParams += separator + param.getName();
            separator = ", ";
        }
        
        // Abstract operation return managment
        String abstReturnType = "";
        String abstReturn = "";
        String finalReturn = "";
        if (!abstOp.getReturnType().equals("void")) {
            abstReturnType = GeneratorUtil.getSimpleName(abstOp.getReturnType())
                    + " result = ";
            abstReturn = "return result;";
            finalReturn = "return " +
                    getReturnValue(abstOp.getReturnType()) + ";";
        }

        // Error key for i18n
        String contract =
            GeneratorUtil.toLowerCaseFirstLetter(serviceContractName);
        String errorKey = StringUtils.lowerCase(modelName) + ".error." +
                            contract + "." + source.getName();

        String treateErrorParams = "eee, I18n.n_(\"" + errorKey + "\")";

        if (needErrorArgs) {
            // Init errorArgs
            buffer.append(""
    +"\n"
+"        List<Object> errorArgs = new ArrayList<Object>();\n"
+"    "             );
            treateErrorParams += ", errorArgs.toArray()";
        }

        if (needTransaction) {
            // Open the transaction
            buffer.append(""
    +"\n"
+"        TopiaContext transaction = null;\n"
+"        try {\n"
+"            transaction = beginTransaction();\n"
+"    "
                    );
            // Add transaction in treateError parameters
            treateErrorParams = "transaction, " + treateErrorParams;
        } else {
            buffer.append(""
    +"\n"
+"        try {\n"
+"    "
                    );
        }

        String implName = StringUtils.capitalize(implOp.getName());
        String first = modelName.substring(0, 1);

        buffer.append(""
    +"\n"
+"            long startTime = 0;\n"
+"            if (log.isDebugEnabled()) {\n"
+"                log.debug(\""+first+":[ begin "+implName+" ]\");\n"
+"                startTime = System.currentTimeMillis();\n"
+"            }\n"
+"            if (log.isTraceEnabled()) {\n"
+"                String message = new StringBuilder(\"# ARGS >\")"+toStringAppend+".\n"
+"                    toString();\n"
+"                log.trace(message);\n"
+"            }\n"
+"\n"
+"            "+abstReturnType+""+abstName+"("+abstParams+");\n"
+"            if (log.isDebugEnabled()) {\n"
+"                long stopTime = System.currentTimeMillis();\n"
+"                log.debug(\""+first+":[ end "+implName+" ] Time = \" +\n"
+"                        DurationFormatUtils.formatDurationHMS(\n"
+"                                stopTime - startTime));\n"
+"            }\n"
+"            "+abstReturn+" ");

        // Copy exceptions
        for (String ex : source.getExceptions()) {
            addException(implOp, ex);
            // Add catch block for known exceptions we want to throw
            String exName = GeneratorUtil.getSimpleName(ex);
            buffer.append(""
    +"\n"
+"        } catch ("+exName+" eee) {\n"
+"            throw eee; ");
        }

        buffer.append(""
    +"\n"
+"        } catch (Exception eee) {\n"
+"            treateError("+treateErrorParams+"); ");

        if (needTransaction) {
            // Finally block to close transaction
            buffer.append(""
    +"\n"
+"        } finally {\n"
+"            closeTransaction(transaction); ");
        }

        buffer.append(""
    +"\n"
+"        }\n"
+"        "+finalReturn+"\n"
+"    "
                );

        setOperationBody(implOp, buffer.toString());
    }

    /**
     * boolean to specify if the method need a transaction or not.
     * Default set to true but can be override using a tagvalue "transaction"
     * on the method from model.
     *
     * @param op where the tagvalue is set
     * @return true if transaction is needed
     */
    protected boolean isTransactionNeeded(ObjectModelOperation op) {
        boolean needTransaction = true;

        String transactionTag =
                op.getTagValue(TopiaGeneratorUtil.TAG_TRANSACTION);

        if (transactionTag != null) {
            needTransaction = Boolean.parseBoolean(transactionTag);
        }
        return needTransaction;
    }

    /**
     * boolean to specify if the method need error arguments or not
     * Default set to false but can be override using a tagvalue "errorArgs" on
     * the method from model.
     *
     * @param op where the tagvalue is set
     * @return true if errorArgs are needed
     */
    protected boolean isErrorArgsNeeded(ObjectModelOperation op) {
        // 
        boolean needErrorArgs = false;

        String errorArgsTag =
                op.getTagValue(TopiaGeneratorUtil.TAG_ERROR_ARGS);

        if (errorArgsTag != null) {
            needErrorArgs = Boolean.parseBoolean(errorArgsTag);
        }
        return needErrorArgs;
    }

    /**
     * This method give the return string for an operation {@code returnType}.
     * This use {@link Primitive} enum to provide default values for primitive
     * type. For all other object type, this method will return null.
     *
     * @param returnType
     * @return the defaultValue of the returnType
     */
    protected String getReturnValue(String returnType) {
        try {
            Primitive prim =
                Primitive.valueOf(StringUtils.upperCase(returnType));
            return prim.getValue();
        // If not defined in Primitive enum, return null
        } catch (IllegalArgumentException eee) {
            return null;
        }
    }

    protected enum Primitive {
        BYTE("0"),
        SHORT("0"),
        INT("0"),
        LONG("0"),
        FLOAT("0."),
        DOUBLE("0."),
        CHAR("''"),
        BOOLEAN("false");

        private String value;

        Primitive(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }
}
