/*
 * *##% 
 * JRedmine maven plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine.plugin;

import java.text.ParseException;
import java.util.Date;
import org.apache.maven.plugin.MojoExecutionException;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.Version;
import org.nuiton.plugin.PluginHelper;

/**
 * Prepare the next version to be used.
 * </p>
 * If a previous version is given, then move all unclosed issues from the
 * previous version to the new one.
 *<p/>
 * 
 * @goal next-version
 *
 * @author tchemit
 * @since 1.0.0
 */
public class NextVersionMojo extends AbstractRedmineMojo {

    /**
     * The news description to update to redmine server.
     *
     * Note : if not set, no update of the description will be made.
     *
     * @parameter expression="${redmine.versionDescription}"
     * @since 1.0.0
     */
    protected String versionDescription;
    /**
     * The effective date to set on the version.
     * <p/>
     * The format of the date is {@code yyyy-mm-dd}.
     * <p/>
     * If not Set - will use current day date only if {@link #closeVersion}
     * flag is on.
     *
     * @parameter expression="${redmine.effectiveDate}"
     * @since 1.0.0
     */
    private String effectiveDate;
    /**
     * The name of a previous version.
     * <p/>
     * If not Set - will not move any issues to the new version.
     *
     * @parameter expression="${redmine.previousVersionName}"
     * @since 1.0.0
     */
    private String previousVersionName;
    /**
     * A flag to skip the goal.
     *
     * @parameter expression="${redmine.skipNextVersion}" default-value="false"
     * @since 1.0.0
     */
    private boolean skipNextVersion;
    /**
     * A flag to test plugin but send nothing to redmine.
     *
     * @parameter expression="${redmine.dryRun}" default-value="false"
     * @since 1.0.0
     */
    protected boolean dryRun;
    /**
     * A flag to restirct only one run in a build (for multi-module context).
     *
     * @parameter expression="${redmine.runOnce}" default-value="true"
     * @since 1.0.0
     */
    private boolean runOnce;
    /**
     * effective date to set
     */
    private Date date;
//    private boolean runOnceDone;

    public NextVersionMojo() {
        super(true, false, true);
    }

    @Override
    protected boolean isGoalSkip() {
        return skipNextVersion;
    }

    @Override
    protected boolean isRunOnce() {
        return runOnce;
    }

    @Override
    protected boolean checkRunOnceDone() {
        return isRunOnce() && !isExecutionRoot();
    }

    @Override
    protected boolean init() throws Exception {

        if (versionId == null || versionId.trim().isEmpty()) {
            throw new MojoExecutionException("required a versionId parameter");
        }

        versionId = PluginHelper.removeSnapshotSuffix(versionId);

        runOnceDone = false;

        if (isRunOnce()) {
            runOnceDone = checkRunOnceDone();
            if (runOnceDone) {
                return true;
            }
        }

        if (effectiveDate != null && !effectiveDate.trim().isEmpty()) {
            try {
                date = dateFormat.parse(effectiveDate);
            } catch (ParseException e) {
                throw new MojoExecutionException("could not parse effectivate date " + effectiveDate + " for reason " + e.getMessage(), e);
            }
        }
        return super.init();
    }

    @Override
    protected void doAction() throws Exception {
        if (isRunOnceDone()) {
            getLog().info("skip goal, runOnce flag is on, and was already executed.");
            return;
        }

        if (dryRun) {
            getLog().info("\n  dryRun flag is on, no data will be send!\n");
        }

        // get version

        Version[] versions = service.getVersions(projectId);
        Version v = ModelHelper.byVersionName(versionId, versions);

        boolean usePreviousVersion = false;
        if (previousVersionName != null && !previousVersionName.trim().isEmpty()) {
            // retrieve previous version
            Version previousVersion = ModelHelper.byVersionName(previousVersionName, versions);
            if (previousVersion == null) {
                getLog().warn("can not find the previous version " + previousVersionName);
            } else {
                usePreviousVersion = true;
            }
        }

        if (v == null) {
            // version must be created
            v = new Version();
            v.setName(versionId);
        }

        if (versionDescription != null && !versionDescription.trim().isEmpty()) {
            v.setDescription(versionDescription.trim());
        }

        if (date != null) {
            v.setEffectiveDate(date);
        }

        this.releaseVersion = v;

        if (!dryRun) {

            // create version
            getLog().info("next version " + releaseVersion.getName() + (usePreviousVersion ? " from previous version " + previousVersionName : ""));

            service.nextVersion(projectId, usePreviousVersion ? previousVersionName : null, releaseVersion);
        }


    }
}
