package jaxx.runtime.swing.editor;

/*
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.BeanUtil;

import javax.swing.JFormattedTextField;
import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;

/**
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.6
 */
public class SimpleTimeEditorHandler {

    /**
     * Logger.
     */
    private static final Log log =
            LogFactory.getLog(SimpleTimeEditorHandler.class);

    private static final KeyAdapter MOVE_TO_MINUTES_KEY_LISTENER = new KeyAdapter() {

        @Override
        public void keyReleased(KeyEvent e) {
            JFormattedTextField source = (JFormattedTextField) e.getSource();
            String text = source.getText();

            // if the user typed 2 digits or if he typed an hour which cannot be the tens digit,
            // then transfer the focus to the minute editor (cf #3833)
            if (text.length() >= 2 || !text.equals("0") && !text.equals("1") && !text.equals("2")) {
                source.transferFocus();
            }
        }

    };

    private static final KeyAdapter MOVE_TO_SECONDS_KEY_LISTENER = new KeyAdapter() {

        @Override
        public void keyReleased(KeyEvent e) {
            JFormattedTextField source = (JFormattedTextField) e.getSource();
            String text = source.getText();

            // if the user typed 2 digits or if he typed an minute which cannot be the tens digit,
            // then transfer the focus to the second editor (cf #3833)
            if (text.length() >= 2 || !text.equals("0") && !text.equals("1") && !text.equals("2") && !text.equals("3") && !text.equals("4") && !text.equals("5")) {
                source.transferFocus();
            }
        }

    };

    private final PropertyChangeListener propertyDateChanged;

    private final PropertyChangeListener propertyTimeChanged;

    private final SimpleTimeEditor editor;

    private final SimpleTimeEditorModel model;

    /**
     * the mutator method on the property of boxed bean in the editor
     */
    protected Method mutator;

    protected final Calendar calendarSecond;

    protected final Calendar calendarMinute;

    protected final Calendar calendarHour;

    protected final Calendar calendarDate;

    public SimpleTimeEditorHandler(SimpleTimeEditor editor) {
        this.editor = editor;
        this.model = editor.getModel();
        this.calendarSecond = Calendar.getInstance();
        this.calendarMinute = Calendar.getInstance();
        this.calendarHour = Calendar.getInstance();
        this.calendarDate = Calendar.getInstance();
        this.propertyDateChanged = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Date date = (Date) evt.getNewValue();

                if (date != null) {
                    calendarDate.setTime(date);
                    int hours = calendarDate.get(Calendar.HOUR_OF_DAY);
                    int minutes = calendarDate.get(Calendar.MINUTE);
                    int seconds = calendarDate.get(Calendar.SECOND);
                    if (log.isDebugEnabled()) {
                        log.debug("date changed : new value " + hours + ":" + minutes + ":" + seconds);
                    }

                    if(model.isSecondEditable()){
                        model.setTimeModel((hours * 3600) + (minutes * 60) + seconds);
                    }else{
                        model.setTimeModel((hours * 3600) + (minutes * 60) + 0);
                    }


                } else {
                    model.setTimeModel(null);
                }

            }
        };
        this.propertyTimeChanged = new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Integer time = (Integer) evt.getNewValue();
                int hours = time / 3600;
                int minutes = (time % 3600) / 60;
                int seconds = (time % 3600) % 60;

                if(!model.isSecondEditable()){
                    seconds = 0;
                }

                calendarDate.set(Calendar.HOUR_OF_DAY, hours);
                calendarDate.set(Calendar.MINUTE, minutes);
                calendarDate.set(Calendar.SECOND, seconds);

                // push it back into the bean

                Date newValue = calendarDate.getTime();

                if (log.isDebugEnabled()) {
                    log.debug(model.getProperty() + " on " + model.getBean().getClass() + " :: " + newValue);
                }

                try {
                    getMutator().invoke(model.getBean(), newValue);

                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        };
    }

    public void init() {

        if (model.getBean() == null) {
            throw new NullPointerException("can not have a null bean in ui " + editor);
        }

        editor.getSecond().setEditor(new JSpinner.DateEditor(editor.getSecond(), "ss"));

        editor.getMinute().setEditor(new JSpinner.DateEditor(editor.getMinute(), "mm"));
        //editor.getHour().setEditor(new JSpinner.DateEditor(editor.getHour(), "HH"));

        // listen to the typed text and automatically focus the minute when the user typed the hour
        JSpinner.DefaultEditor hourEditor = (JSpinner.DefaultEditor) editor.getHour().getEditor();
        hourEditor.getTextField().addKeyListener(MOVE_TO_MINUTES_KEY_LISTENER);

        JSpinner.DefaultEditor minuteEditor = (JSpinner.DefaultEditor) editor.getMinute().getEditor();
        minuteEditor.getTextField().addKeyListener(MOVE_TO_SECONDS_KEY_LISTENER);

        JSpinner.DefaultEditor secondEditor = (JSpinner.DefaultEditor) editor.getSecond().getEditor();
        secondEditor.getTextField().addKeyListener(MOVE_TO_SECONDS_KEY_LISTENER);

//        TuttiUIUtil.autoSelectOnFocus(minuteEditor.getTextField());
//        JSpinner.NumberEditor hourEditor = (JSpinner.NumberEditor) editor.getHour().getEditor();
//        TuttiUIUtil.autoSelectOnFocus(hourEditor.getTextField());

        // listen when date changes (should come from outside)

        model.addPropertyChangeListener(SimpleTimeEditorModel.PROPERTY_DATE, propertyDateChanged);

        // When time model change, let's push it back in bean
        model.addPropertyChangeListener(SimpleTimeEditorModel.PROPERTY_TIME_MODEL, propertyTimeChanged);
    }

    public SimpleTimeEditor getEditor() {
        return editor;
    }

    protected Date setMinuteModel(Date incomingDate) {
        if (incomingDate == null) {
            incomingDate = new Date();
            calendarMinute.setTime(incomingDate);
            calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
            calendarMinute.set(Calendar.MINUTE, 0);
            calendarMinute.set(Calendar.SECOND, 0);
        } else {
            calendarMinute.setTime(incomingDate);
            calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
            calendarMinute.set(Calendar.SECOND, 0);
        }
        incomingDate = calendarMinute.getTime();
        return incomingDate;
    }

    protected Date setSecondModel(Date incomingDate) {
        if (incomingDate == null) {
            incomingDate = new Date();
            calendarMinute.setTime(incomingDate);
            calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
            calendarMinute.set(Calendar.MINUTE, 0);
            calendarMinute.set(Calendar.SECOND, 0);
        } else {
            calendarMinute.setTime(incomingDate);
            calendarMinute.set(Calendar.HOUR_OF_DAY, 0);
            calendarMinute.set(Calendar.MINUTE, 0);
        }
        incomingDate = calendarMinute.getTime();
        return incomingDate;
    }

    public void updateTimeModelFromHour(Integer hour) {
        if(model.isSecondEditable()){
            model.setTimeModel((hour * 3600) + (model.getMinute() * 60) + model.getSecond());
        }else{
            model.setTimeModel((hour * 3600) + (model.getMinute() * 60) + 0);
        }

    }

    public void updateTimeModelFromMinute(Date minuteDate) {
        calendarMinute.setTime(minuteDate);
        int newHour = calendarMinute.get(Calendar.HOUR_OF_DAY);
        int newMinute = calendarMinute.get(Calendar.MINUTE);

        int oldHour = model.getHour();
        int oldMinute = model.getMinute();
        int oldSecond = model.getSecond();

        if (oldHour == newHour && oldMinute == newMinute) {

            // do nothing, same data
            if (log.isDebugEnabled()) {
                log.debug("Do not update time model , stay on same time = " + oldHour + ":" + oldMinute);
            }
            return;
        }

        // by default stay on same hour
        int hour = oldHour;

        // by default, use the new minute data

        if (log.isDebugEnabled()) {
            log.debug("hh:mm (old from dateModel)   = " + oldHour + ":" + oldMinute);
            log.debug("hh:mm (new from minuteModel) = " + newHour + ":" + newMinute);
        }

        SpinnerDateModel minuteModel = editor.getMinuteModel();

        if (newMinute == 0) {

            // minute pass to zero (check if a new hour is required)
            if (newHour == 1) {

                if (oldHour == 23) {

                    // can't pass from 23:59 to 0:00, stay on 23:59
                    if (log.isDebugEnabled()) {
                        log.debug("Do not update time model , stay on hh:mm = " + oldHour + ":" + oldMinute);
                    }
                    minuteModel.setValue(minuteModel.getPreviousValue());
                    return;
                }
                hour = (oldHour + 1) % 24;
            }
        } else if (newMinute == 59) {

            // minute pass to 59 (check if a new hour is required)

            if (newHour == 23) {

                if (oldHour == 0) {

                    // can't pass from 0:00 to 23:59, stay on 0:00
                    if (log.isDebugEnabled()) {
                        log.debug("Do not update time model , stay on hh:mm = " + oldHour + ":" + oldMinute);
                    }
                    minuteModel.setValue(minuteModel.getNextValue());
                    return;
                }

                // decrease hour
                hour = (oldHour - 1) % 24;
            }
        }

        // date has changed
        if (log.isDebugEnabled()) {
            log.debug("Update time model to hh:mm = " + hour + ":" + newMinute);
        }

        if(model.isSecondEditable()){
            model.setTimeModel(hour * 3600 + newMinute * 60 + oldSecond);
        }else{
            model.setTimeModel(hour * 3600 + newMinute * 60 + 0);
        }

    }

    public void updateTimeModelFromSecond(Date secondDate) {

        calendarSecond.setTime(secondDate);
        int newHour = calendarSecond.get(Calendar.HOUR_OF_DAY);
        int newMinute = calendarSecond.get(Calendar.MINUTE);
        int newSecond = calendarSecond.get(Calendar.SECOND);

        int oldHour = model.getHour();
        int oldMinute = model.getMinute();
        int oldSecond = model.getSecond();




        SpinnerDateModel secondModel = editor.getSecondModel();

        if(model.isSecondEditable()){


            // second pass to zero
            if (newMinute >= 1) {
                if (oldHour == 23 && oldMinute == 59) {
                    // can't pass from 23:59:59 to 0:00:00, stay on 23:59:59
                    if (log.isDebugEnabled()) {
                        log.info("Do not update time model , stay on hh:mm:ss = " + oldHour + ":" + oldMinute + ":" + oldSecond);
                    }
                    secondModel.setValue(59);
                    return;
                }
            // second pass to 59
            } else if (newHour == 23 && newSecond == 59) {
                if (oldHour == 0 && oldMinute == 0) {
                    // can't pass from 0:00:00 to 23:59:59, stay on 0:00:00
                    if (log.isDebugEnabled()) {
                        log.info("Do not update time model , stay on hh:mm:ss = " + oldHour + ":" + oldMinute + ":" + oldSecond);
                    }
                    secondModel.setValue(secondModel.getNextValue());
                    return;
                }
            }

            int hours = oldHour+newHour;
            int minutes = oldMinute+newMinute;
            int newTime = (hours * 3600) + (minutes * 60) + newSecond;

            model.setTimeModel(newTime);

            // date has changed
            if (log.isDebugEnabled()) {
                log.debug("Update time model to hh:mm:ss = " + hours + ":" + minutes + ":" + newSecond);
            }

        }else{
            model.setTimeModel((oldHour * 3600) + (oldMinute * 60) + 0);
        }
    }

    protected void setDate(Date oldValue, Date newValue) {
        if (model.getBean() != null) {

            if (log.isDebugEnabled()) {
                log.debug(model.getProperty() + " on " + model.getBean().getClass() + " :: " + oldValue + " to " + newValue);
            }

            try {
                getMutator().invoke(model.getBean(), newValue);

            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    protected Method getMutator() {
        if (mutator == null) {
            Object bean = model.getBean();
            if (bean == null) {
                throw new NullPointerException("could not find bean in " + editor);
            }
            String property = model.getProperty();
            if (property == null) {
                throw new NullPointerException("could not find property in " + editor);
            }

            mutator = BeanUtil.getMutator(bean, property);
        }
        return mutator;
    }
}

