/*
 * *##% 
 * JAXX Action
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
/**
 * ##% Copyright (C) 2008 Code Lutin, Tony Chemit
 * This program is free software; you
 * can redistribute it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details. You
 * should have received a copy of the GNU General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 59 Temple Place
 * - Suite 330, Boston, MA 02111-1307, USA. 
 * ##%
 */


package org.nuiton.jaxx.action;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.LoaderClassPath;
import javassist.NotFoundException;
import org.nuiton.util.SortedProperties;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedOptions;
import javax.annotation.processing.SupportedSourceVersion;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.AnnotationValue;
import javax.lang.model.element.AnnotationValueVisitor;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.SimpleAnnotationValueVisitor6;
import javax.tools.FileObject;
import javax.tools.JavaFileObject;
import javax.tools.StandardLocation;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;


@SupportedAnnotationTypes(value = {"org.nuiton.jaxx.action.*"})
@SupportedSourceVersion(SourceVersion.RELEASE_6)
@SupportedOptions({"jaxx.verbose"})
/**
 * Annotation processor to compute actions mapping.
 *
 * @author chemit */
public class ActionAnnotationProcessing extends AbstractProcessor {

    /** the {@link ActionProvider} service declaration relative path */
    protected String providerDeclarationLocation = "META-INF/services/" + ActionProvider.class.getName();

    /** the relative path where to store actions mapping, will be complete with the name of base action to use */
    protected String actionsFileLocation = ActionProviderFromProperties.actionsFileLocation;

    /** verbose flag (can be activated by passing an annotation parameter to compiler via <code>-Ai18n.verbose</code>) */
    protected boolean verbose;

    /** the list of class processed by the processor */
    protected java.util.List<String> processedClass;

    /** the map of actions processed, keys are the action commaned and values are fqn of implementations */
    protected Properties actions;

    /** Extractor of values of annotations found */
    protected AnnotationValueVisitor<Object, Void> annotationValueExtractor;

    /** the type element of the base action to be used by {@link ActionProvider} */
    protected TypeElement baseActionElement;

    /** the fqn of the action provider to generate */
    protected String providerFQN;

    /** the fqn of the base action class to be used */
    protected String baseFQN;

    @Override
    public synchronized void init(ProcessingEnvironment processingEnv) {
        super.init(processingEnv);
        parseOptions();
        printDebug("verbose      : " + verbose);
        printDebug("FileLocation : " + actionsFileLocation);
        processedClass = new ArrayList<String>();
        actions = new SortedProperties();
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        for (TypeElement annotation : annotations) {

            Set<? extends Element> annotatedWith = roundEnv.getElementsAnnotatedWith(annotation);

            if (annotation.getQualifiedName().toString().equals(ActionProviderAnnotation.class.getName())) {
                // init provider
                if (annotatedWith.size() != 1) {
                    throw new IllegalStateException("can have only one provider defined by the annotation " + ActionProviderAnnotation.class);
                }

                baseActionElement = (TypeElement) annotatedWith.iterator().next();

                //fixme it is not possible to know if baseActionElement is assigned from MyAbstractAction, since we
                // can NOT garanted at this stage thaht the class was compiled...

                baseFQN = baseActionElement.asType().toString();
                int index = baseFQN.lastIndexOf(".") + 1;
                String baseSimpleName = baseFQN.substring(index);
                String packageName = baseFQN.substring(0, index);

                providerFQN = packageName + baseSimpleName + "Provider";
                printDebug("providerFQN " + providerFQN);
                actionsFileLocation = String.format(actionsFileLocation, baseSimpleName);
                continue;
            }

            for (Element e : annotatedWith) {
                String className = e.toString();

                if (processedClass.contains(className)) {
                    printWarning("class already processed " + className);
                    // do not process class twice
                    continue;
                }

                boolean wasTreated = registerActionsForClass(annotation.asType(), e);
                if (wasTreated) {
                    printDebug("process class " + className);
                    processedClass.add(className);
                } else {
                    printDebug("class was not processed " + e);
                }
            }
        }

        if ((roundEnv.processingOver())) {
            printDebug("round is over " + roundEnv);
            try {
                if (baseActionElement != null) {
                    // found the base action class to be compiled, so we have to write provider things...
                    writeProviderClass();
                    writeProviderServiceDeclaration();
                } else {

                    // baseActionClass was not compiled at this time, must find it back
                    // this means they should have an already mapping file written

                    actionsFileLocation = findMappingFile();

                    printInfo("reused actionFilesLocation " + actionsFileLocation);
                }

                writeActionMapping();

            } catch (Exception e) {
                throw new RuntimeException(e);
            } finally {
                processedClass.clear();
                actions.clear();
            }
        }

        return true;
    }

    protected String findMappingFile() throws IOException {
        String path = String.format(actionsFileLocation, "dummy_" + System.nanoTime());


        FileObject oldFo = processingEnv.getFiler().getResource(StandardLocation.SOURCE_OUTPUT, "", path);
        File dummyFile = new File(oldFo.toUri().toString()).getParentFile();
        File[] files = dummyFile.listFiles(new FilenameFilter() {

            public boolean accept(File dir, String name) {
                return name.startsWith("jaxx-") && name.endsWith("-actions.properties");
            }
        });
        if (files.length < 1) {
            // this is not normal, should have exactly one file here
            throw new IllegalStateException("no provider name found, you must add on baseaction the annotation " + ActionProviderAnnotation.class);
        }
        File f = files[0];
        int index = f.getAbsolutePath().indexOf("META-INF");

        return f.getAbsolutePath().substring(index);
    }

    protected boolean registerActionsForClass(TypeMirror annotationType, Element e) {
        boolean doTreate = false;
        for (AnnotationMirror mirror : e.getAnnotationMirrors()) {
            if (!mirror.getAnnotationType().equals(annotationType)) {
                // do not treate other annotations
                continue;
            }
            doTreate = true;
            printDebug("found a annotation to treate : " + mirror + " for action : " + e.toString());
            for (String name : getActionNames(mirror)) {
                actions.put("action." + name, e.toString());
                printDebug("registerActionForClass " + name + " : " + e.toString());
            }
        }
        return doTreate;
    }

    protected void parseOptions() {
        java.util.Map options = processingEnv.getOptions();
        verbose = options.containsKey("jaxx.verbose");
    }

    protected void writeProviderClass() throws IOException, NotFoundException, CannotCompileException, ClassNotFoundException {
        OutputStream outputStream = null;
        try {

            ClassPool pool = ClassPool.getDefault();

            pool.appendClassPath(new LoaderClassPath(ActionProviderFromProperties.class.getClassLoader()));

            CtClass superClass = pool.get(ActionProviderFromProperties.class.getName());
            CtClass clazz = pool.makeClass(providerFQN);
            // define the base action class in javassist pool to make possible compilation
            pool.makeClass(baseFQN);
            clazz.setSuperclass(superClass);
            // add constructor
            CtConstructor constructor = new CtConstructor(null, clazz);
            constructor.setBody("super( " + baseFQN + ".class);");
            clazz.addConstructor(constructor);
            byte[] byteCode = clazz.toBytecode();

            JavaFileObject fo = processingEnv.getFiler().createClassFile(providerFQN);
            printInfo("writing " + fo.toUri());
            outputStream = fo.openOutputStream();
            outputStream.write(byteCode);

        } finally {
            if (outputStream != null) {
                outputStream.close();
            }
        }
    }

    protected void writeProviderServiceDeclaration() throws IOException {
        BufferedWriter w = null;
        try {
            FileObject fo = processingEnv.getFiler().createResource(StandardLocation.SOURCE_OUTPUT, "", providerDeclarationLocation);
            printInfo("writing " + fo.toUri());
            w = new BufferedWriter(fo.openWriter());
            w.append("# generated by ").append(getClass().getName()).append("\n").toString();
            w.append("#").append(new java.util.Date().toString()).append("\n").toString();
            w.append(providerFQN);
        } finally {
            if (w != null) {
                w.close();
            }
        }
    }

    protected void writeActionMapping() throws IOException {
        if (actions.isEmpty()) {
            // nothing to write or overwrite
            return;
        }

        BufferedWriter w = null;
        try {
            Properties oldProps = loadOldActionMapping();
            if (oldProps != null) {
                oldProps.putAll(actions);
                actions = oldProps;
            }
            // ecriture de toutes les actions trouvees
            FileObject fo = processingEnv.getFiler().createResource(StandardLocation.SOURCE_OUTPUT, "", actionsFileLocation);
            printInfo("writing " + fo.toUri());
            w = new BufferedWriter(fo.openWriter());
            actions.store(w, "generated by " + getClass().getName());
        } finally {
            if (w != null) {
                w.close();
            }
        }
    }

    protected Properties loadOldActionMapping() throws IOException {
        // reprise sur une ancienne compilation
        FileObject oldFo = processingEnv.getFiler().getResource(StandardLocation.SOURCE_OUTPUT, "", actionsFileLocation);
        if (!new File(oldFo.toUri().toString()).exists()) {
            return null;
        }
        Properties oldProps = new SortedProperties();
        InputStream inputStream = null;
        try {
            inputStream = oldFo.openInputStream();
            if (inputStream != null) {
                oldProps.load(inputStream);
            }
            oldFo.delete();
        } finally {
            if (inputStream != null) {
                inputStream.close();
            }
        }

        return oldProps;
    }

    /**
     * Obtain the array of names to be used by the annotation
     *
     * @param element the dictonnary of values found in a annotation
     * @return thee array of names detected in the annotation
     */
    @SuppressWarnings({"unchecked"})
    protected String[] getActionNames(AnnotationMirror element) {
        String[] result = null;
        for (Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : element.getElementValues().entrySet()) {
            ExecutableElement type = entry.getKey();
            String name = type.getSimpleName().toString();

            if ("actionCommands".equals(name)) {
                List<String> stringList = (List<String>) entry.getValue().accept(getAnnotationValueExtractor(), null);
                result = stringList.toArray(new String[stringList.size()]);
                // a actionCommands field means
                break;
            }
            if ("actionCommandProvider".equals(name)) {
                TypeMirror t = (TypeMirror) entry.getValue().accept(getAnnotationValueExtractor(), null);
                String classname = t.toString();
                printDebug("actionCommandProvider = " + classname);
                if (classname.equals(ActionNameProvider.class.getName())) {
                    continue;
                }

                // means there is a runtime names provider
                result = new String[]{":" + classname};
                break;
            }
            if ("actionCommand".equals(name)) {
                result = new String[]{(String) entry.getValue().accept(getAnnotationValueExtractor(), null)};
            }
        }

        return result;
    }

    protected AnnotationValueVisitor<Object, Void> getAnnotationValueExtractor() {
        if (annotationValueExtractor == null) {
            annotationValueExtractor = new SimpleAnnotationValueVisitor6<Object, Void>() {

                @Override
                protected Object defaultAction(Object o, Void aVoid) {
                    return o;
                }

                @Override
                public Object visitArray(List<? extends AnnotationValue> vals, Void aVoid) {
                    List<Object> realVals = new java.util.ArrayList<Object>();
                    for (AnnotationValue val : vals) {
                        realVals.add(val.accept(this, aVoid));
                    }
                    return realVals;
                }

                public Object visitType(TypeMirror t, Void aVoid) {
                    return t;
                }
            };
        }
        return annotationValueExtractor;
    }

    protected void printWarning(String msg) {
        System.out.println("[WARN] " + getClass().getName() + " : " + msg);
    }

    protected void printInfo(String msg) {
        System.out.println("[INFO] " + getClass().getName() + " : " + msg);
    }

    protected void printDebug(String msg) {
        if (verbose) {
            System.out.println("[DEBUG] " + getClass().getName() + " : " + msg);
        }
    }
}	
