/*
 * #%L
 * JAXX :: Runtime
 * 
 * $Id: BeanValidatorDetector.java 1847 2010-04-16 12:27:48Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2/jaxx-runtime/src/main/java/jaxx/runtime/validator/BeanValidatorDetector.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.validator;

import org.apache.commons.beanutils.ConstructorUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.FilenameFilter;
import java.lang.reflect.Constructor;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Un detecteur de validateurs pour un liste de classes données et un répertoire
 * où chercher les fichiers de validation.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.6.0
 */
public class BeanValidatorDetector {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log =
            LogFactory.getLog(BeanValidatorDetector.class);

    public SortedSet<BeanValidator<?>> detect(File sourceRoot,
                                              Class<?>... types) {
        SortedSet<BeanValidator<?>> result =
                detect(BeanValidator.class, sourceRoot, null, types);
        return result;
    }

    public SortedSet<BeanValidator<?>> detect(Class<?> validatorClass,
                                              File sourceRoot,
                                              Pattern contextFilter,
                                              Class<?>... types) {
        SortedSet<BeanValidator<?>> result = detect(validatorClass,
                                                    sourceRoot,
                                                    contextFilter,
                                                    null,
                                                    types);
        return result;
    }

    public SortedSet<BeanValidator<?>> detect(Class<?> validatorClass,
                                              File sourceRoot,
                                              Pattern contextFilter,
                                              BeanValidatorScope[] scopes,
                                              Class<?>... types) {

        SortedSet<BeanValidator<?>> result =
                new TreeSet<BeanValidator<?>>(new BeanValidatorComparator());

        for (Class<?> c : types) {
            File dir = getClassDir(sourceRoot, c);
            if (!dir.exists()) {
                // pas de repertoire adequate
                log.debug("skip non existing directory " + dir);
                continue;
            }
            String[] contexts = getContexts(c, dir);
            if (log.isDebugEnabled()) {
                log.debug("contexts : " + Arrays.toString(contexts));
            }

            if (contexts.length > 0) {
                String[] realContexts = getContextsWithoutScopes(contexts);

                if (log.isDebugEnabled()) {
                    log.debug("realContexts : " +
                              Arrays.toString(realContexts));
                }

                if (contextFilter != null) {
                    // filter contexts
                    realContexts = getFilterContexts(contextFilter,
                                                     realContexts
                    );
                    if (log.isDebugEnabled()) {
                        log.debug("filterContexts : " +
                                  Arrays.toString(realContexts));
                    }
                }

                for (String context : realContexts) {

                    // on cherche le validateur
                    BeanValidator<?> validator = getValidator(
                            validatorClass,
                            c,
                            context.isEmpty() ? null : context,
                            scopes
                    );
                    if (validator != null) {
                        // on enregistre le validateur
                        result.add(validator);
                    }
                }
            }
        }
        return result;
    }

    /**
     * Pour un context et un type d'entité donné, instancie un validateur et
     * test si ce validateur est utilisable (i.e qu'il admet des champs à
     * valider).
     * <p/>
     * Si aucun champ n'est trouvé dans le validateur, alors on retourne null.
     *
     * @param <B>            le type du bean
     * @param validatorClass le type de validateur a instancie
     * @param klass          le type du bean
     * @param context        le context du validateur
     * @param scopes         les scopes a utiliser (si {@code null} alors pas de
     *                       filtre sur les scopes)
     * @return le validateur initialisé, ou <code>null</code> si aucun scope
     *         détecté dans le validateur.
     */
    protected <B> BeanValidator<B> getValidator(Class<?> validatorClass,
                                                Class<B> klass,
                                                String context,
                                                BeanValidatorScope... scopes) {

        BeanValidator<B> valitator;
        valitator = createValidator(validatorClass, klass, context, scopes);

        Set<BeanValidatorScope> resultScopes = valitator.getScopes();
        if (resultScopes.isEmpty()) {
            valitator = null;
            if (log.isDebugEnabled()) {
                log.debug(klass + " : validator skip (no scopes detected)");
            }
        } else {
            if (log.isDebugEnabled()) {
                log.debug(klass + " : keep validator " + valitator);
            }
        }
        return valitator;
    }

    protected <B> BeanValidator<B> createValidator(
            Class<?> validatorClass,
            Class<B> klass,
            String context,
            BeanValidatorScope[] scopes) {
        BeanValidator<B> valitator;
        Constructor<?> con;
        try {
            con = ConstructorUtils.getAccessibleConstructor(
                    validatorClass,
                    new Class<?>[]{
                            Class.class,
                            String.class,
                            BeanValidatorScope[].class
                    }
            );
            if (con != null) {

                valitator = (BeanValidator<B>) con.newInstance(
                        klass,
                        context, scopes
                );

            } else {
                con = ConstructorUtils.getAccessibleConstructor(
                        validatorClass,
                        new Class<?>[]{
                                Class.class,
                                String.class,
                                BeanValidatorScope[].class
                        }
                );

                if (con == null) {
                    throw new IllegalStateException(
                            "could not find a constructor with " +
                            "(Class.class, String) or " +
                            "(Class,String BeanValidatorScope[])");
                }

                valitator = (BeanValidator<B>) con.newInstance(
                        klass,
                        context
                );
                if (scopes != null && scopes.length > 0) {
                    valitator.setFilterScopes(scopes);
                }
            }

        } catch (Exception ex) {
            throw new RuntimeException(
                    "could not instanciate validator " + validatorClass +
                    " for reason " + ex.getMessage(), ex);
        }
        return valitator;
    }

    protected File getClassDir(File sourceRoot, Class<?> clazz) {
        String path = clazz.getPackage().getName();
        path = path.replaceAll("\\.", File.separator);
        File dir = new File(sourceRoot, path);
        return dir;
    }

    protected String[] getContexts(Class<?> clazz, File dir) {
        Set<String> result = new TreeSet<String>();
        ValidatorFilenameFilter filter = new ValidatorFilenameFilter(clazz);
        if (log.isDebugEnabled()) {
            log.debug("dir : " + dir);
        }
        String[] files = dir.list(filter);
        for (String file : files) {
            if (log.isDebugEnabled()) {
                log.debug("file " + file);
            }
            String context = file.substring(
                    filter.prefix.length(),
                    file.length() - ValidatorFilenameFilter.SUFFIX.length()
            );
            if (log.isDebugEnabled()) {
                log.debug("detect " + clazz.getSimpleName() +
                          " context [" + context + "]");
            }
            result.add(context);
        }
        return result.toArray(new String[result.size()]);
    }

    protected String[] getContextsWithoutScopes(String[] contexts) {
        Set<String> result = new TreeSet<String>();
        BeanValidatorScope[] scopes = BeanValidatorScope.values();
        for (String context : contexts) {
            for (BeanValidatorScope scope : scopes) {
                String scopeName = scope.name().toLowerCase();
                if (!context.endsWith(scopeName)) {
                    // pas concerne par ce scope
                    continue;
                }
                if (log.isDebugEnabled()) {
                    log.debug("detect context : " + context);
                }
                String realContext = context.substring(
                        0,
                        context.length() - scopeName.length()
                );
                if (realContext.endsWith("-")) {
                    realContext = realContext.substring(
                            0,
                            realContext.length() - 1
                    );
                }
                result.add(realContext);
            }
        }
        return result.toArray(new String[result.size()]);
    }

    protected String[] getFilterContexts(Pattern contextFilter,
                                         String[] realContexts) {
        List<String> result = new ArrayList<String>();
        for (String c : realContexts) {
            Matcher m = contextFilter.matcher(c);
            if (m.matches()) {
                result.add(c);
            }
        }
        return result.toArray(new String[result.size()]);
    }

    protected static class ValidatorFilenameFilter implements FilenameFilter {

        protected static final String SUFFIX = "-validation.xml";

        protected Class<?> clazz;

        protected String prefix;

        public ValidatorFilenameFilter(Class<?> clazz) {
            this.clazz = clazz;
            prefix = clazz.getSimpleName() + "-";
        }

        @Override
        public boolean accept(File dir, String name) {
            boolean result = name.endsWith(SUFFIX);
            if (result) {
                result = name.startsWith(prefix);
            }
            return result;
        }
    }

    protected static class BeanValidatorComparator implements Comparator<BeanValidator<?>> {

        @Override
        public int compare(BeanValidator<?> o1, BeanValidator<?> o2) {
            String contextName1 =
                    o1.getBeanClass().getSimpleName() + "-" +
                    (o1.getContextName() == null ? "" : o1.getContextName());
            String contextName2 =
                    o2.getBeanClass().getSimpleName() + "-" +
                    (o2.getContextName() == null ? "" : o2.getContextName());
            return contextName1.compareTo(contextName2);
        }
    }
}
