package org.nuiton.eugene.plugin;

import org.nuiton.plugin.PluginIOContext;
import java.io.File;
import java.util.Arrays;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Settings;
import org.nuiton.plugin.AbstractPlugin;

/**
 * La classe de base pour definir des mojos dans eugene.
 * 
 * @author chemit
 * @since 1.0.0-rc-8
 */
public abstract class EugeneAbstractMojo extends AbstractPlugin {

    /**
     * Maven project.
     *
     * @description Dépendance du projet.
     * @parameter default-value="${project}"
     * @readonly
     * @since 1.0.0-rc-8
     */
    protected MavenProject project;
    /**
     * @description le settings (pour obtenir le mode offline)
     * @parameter default-value="${settings}"
     * @readonly
     * @since 1.0.0-rc-4
     */
    protected Settings settings;
    /**
     * Ecrase les fichiers générés.
     *
     * @parameter expression="${eugene.overwrite}" default-value="false"
     * @since 0.50
     */
    protected boolean overwrite;
    /**
     * Pour activer le mode verbeux.
     *
     * @parameter expression="${eugene.verbose}" default-value="${maven.verbose}"
     * @since 1.0.0-rc-8
     */
    protected boolean verbose;
    /**
     * Encoding to be used for generation of files.
     *
     * @parameter expression="${eugene.encoding}" default-value="${project.build.sourceEncoding}"
     * @since 0.60
     */
    protected String encoding;
    /**
     * A flag to mark the mojo to be used in a test phase. This will permits to add generated sources in test compile roots.
     *
     * @parameter expression="${eugene.testPhase}" default-value="false"
     * @since 0.64
     */
    protected boolean testPhase;

    /**
     *
     * Note : l'objet peut etre {@code null} si la configuration ne definit
     * pas la propriété associées dans l'implentation.
     * <p/>
     *
     * Dans tous les cas, une fois la methode {@link #initResources()} appele,
     * l'objet devrait ne plus être null!
     * 
     * @return l'io du plugin ou {@code null} si non initialisé
     *
     */
    protected abstract PluginIOContext getResources();

    /**
     * Initialiser les io du plugins.
     *
     * @return l'io intialisé (avec les valeurs par défaut, là où rien n'a
     * été renseigné).
     */
    protected abstract PluginIOContext initResources();

    /**
     * Build the {@link #getResources()} data.
     *
     * @param defaultIn the default input directory
     * @param defaultOut the default output directory
     * @param defaultTestIn the default test input directory
     * @param defaultTestOut the default test output directory
     * @return the new {@link PluginIOContext} to use in the goal
     */
    protected PluginIOContext initResources(File defaultIn, File defaultOut, File defaultTestIn, File defaultTestOut) {

        PluginIOContext resources = getResources();
        if (resources == null) {
            resources = new PluginIOContext();
        }

        // adding default inputs only if not consumed
        if (resources.getInputs() == null) {
            if (testPhase) {
                resources.setInput(defaultTestIn);
            } else {
                resources.setInput(defaultIn);
            }
            getLog().info(" using default in : " + Arrays.asList(resources.getInputs()));
        } else {
            getLog().info(" in               : " + Arrays.asList(resources.getInputs()));
        }

        if (resources.getOutput() == null) {
            if (testPhase) {
                resources.setOutput(defaultTestOut);
            } else {
                resources.setOutput(defaultOut);
            }
            getLog().info(" using default out : " + resources.getOutput());
        } else {
            getLog().info(" out               : " + resources.getOutput());
        }

        return resources;
    }

    @Override
    protected boolean init() throws Exception {

        // init goal io context
        PluginIOContext p = initResources();

        if ((p.getInputs() == null || p.getInputs().length == 0)) {
            throw new MojoExecutionException("no input defined");
        }
        if (p.getOutput() == null) {
            throw new MojoExecutionException("no output defined");
        }

        p.getOutput().mkdirs();
        return true;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }
}
