/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.thread;

import com.oracle.svm.core.SubstrateOptions;
import com.oracle.svm.core.annotate.RestrictHeapAccess;
import com.oracle.svm.core.annotate.Uninterruptible;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.option.HostedOptionKey;
import com.oracle.svm.core.option.SubstrateOptionsParser;
import com.oracle.svm.core.thread.Safepoint;
import com.oracle.svm.core.thread.VMThreads;
import com.oracle.svm.core.threadlocal.FastThreadLocalFactory;
import com.oracle.svm.core.threadlocal.FastThreadLocalInt;
import com.oracle.svm.core.threadlocal.FastThreadLocalObject;
import com.oracle.svm.core.util.UserError;
import java.util.concurrent.TimeUnit;
import org.graalvm.compiler.api.replacements.Fold;
import org.graalvm.compiler.options.Option;
import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.Threading;
import org.graalvm.nativeimage.impl.ThreadingSupport;

public class ThreadingSupportImpl
implements ThreadingSupport {
    private static final FastThreadLocalObject<RecurringCallbackTimer> activeTimer = FastThreadLocalFactory.createObject(RecurringCallbackTimer.class);
    private static final FastThreadLocalInt currentPauseDepth = FastThreadLocalFactory.createInt();
    private static final String enableSupportOption = SubstrateOptionsParser.commandArgument(Options.SupportRecurringCallback, "+");

    static void initialize() {
        ImageSingletons.add(ThreadingSupport.class, (Object)new ThreadingSupportImpl());
    }

    public void registerRecurringCallback(long interval, TimeUnit unit, Threading.RecurringCallback callback) {
        if (callback != null) {
            UserError.guarantee(Options.SupportRecurringCallback.getValue(), "Recurring callbacks must be enabled during image build with option %s", enableSupportOption);
            UserError.guarantee(SubstrateOptions.MultiThreaded.getValue(), "Recurring callbacks are only supported in multi-threaded mode.", new Object[0]);
            long intervalNanos = unit.toNanos(interval);
            if (intervalNanos < 1L) {
                throw new IllegalArgumentException("intervalNanos");
            }
            RecurringCallbackTimer timer = new RecurringCallbackTimer(intervalNanos, callback);
            activeTimer.set(timer);
            Safepoint.setSafepointRequested(timer.requestedChecks);
        } else {
            activeTimer.set(null);
        }
    }

    @Uninterruptible(reason="Must not contain safepoint checks.")
    static void onSafepointCheckSlowpath() {
        RecurringCallbackTimer timer;
        assert (VMThreads.StatusSupport.isStatusJava()) : "must only be executed when the thread is in Java state";
        RecurringCallbackTimer recurringCallbackTimer = timer = ThreadingSupportImpl.isRecurringCallbackSupported() ? activeTimer.get() : null;
        if (timer != null) {
            timer.evaluate();
        } else {
            Safepoint.setSafepointRequested(Integer.MAX_VALUE);
        }
    }

    @Uninterruptible(reason="Called by uninterruptible code.", mayBeInlined=true)
    static boolean isRecurringCallbackRegistered(IsolateThread thread) {
        return ThreadingSupportImpl.isRecurringCallbackSupported() && activeTimer.get(thread) != null;
    }

    static boolean needsNativeToJavaSlowpath() {
        return VMThreads.ActionOnTransitionToJavaSupport.isActionPending() || ThreadingSupportImpl.isRecurringCallbackSupported() && Options.CheckRecurringCallbackOnNativeToJavaTransition.getValue() != false && activeTimer.get() != null;
    }

    @Uninterruptible(reason="Must not contain safepoint checks.")
    public static void pauseRecurringCallback(String reason) {
        if (!ThreadingSupportImpl.isRecurringCallbackSupported()) {
            return;
        }
        assert (currentPauseDepth.get() >= 0);
        currentPauseDepth.set(currentPauseDepth.get() + 1);
    }

    @Uninterruptible(reason="Must not contain safepoint checks.")
    public static void resumeRecurringCallbackAtNextSafepoint() {
        if (ThreadingSupportImpl.resumeCallbackExecution()) {
            RecurringCallbackTimer timer = activeTimer.get();
            assert (timer != null);
            timer.updateStatistics();
            timer.setSafepointRequested(1);
        }
    }

    public static void resumeRecurringCallback() {
        if (ThreadingSupportImpl.resumeCallbackExecution()) {
            try {
                ThreadingSupportImpl.onSafepointCheckSlowpath();
            }
            catch (Safepoint.SafepointException e) {
                ThreadingSupportImpl.throwUnchecked(e.inner);
            }
        }
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    private static boolean resumeCallbackExecution() {
        if (!ThreadingSupportImpl.isRecurringCallbackSupported()) {
            return false;
        }
        assert (currentPauseDepth.get() > 0);
        currentPauseDepth.set(currentPauseDepth.get() - 1);
        return !ThreadingSupportImpl.isRecurringCallbackPaused() && ThreadingSupportImpl.isRecurringCallbackRegistered(CurrentIsolate.getCurrentThread());
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public static boolean isRecurringCallbackPaused() {
        if (!ThreadingSupportImpl.isRecurringCallbackSupported()) {
            return false;
        }
        return currentPauseDepth.get() != 0;
    }

    @Fold
    public static boolean isRecurringCallbackSupported() {
        return Options.SupportRecurringCallback.getValue() != false && SubstrateOptions.MultiThreaded.getValue() != false;
    }

    @Uninterruptible(reason="Called by uninterruptible code.")
    private static <T extends Throwable> void throwUnchecked(Throwable exception) throws T {
        throw exception;
    }

    private static class RecurringCallbackTimer {
        private static final Threading.RecurringCallbackAccess CALLBACK_ACCESS = new Threading.RecurringCallbackAccess(){

            public void throwException(Throwable t) {
                throw new Safepoint.SafepointException(t);
            }
        };
        private static final double EWMA_LAMBDA = 0.3;
        private static final double TARGET_INTERVAL_FLEXIBILITY = 0.95;
        private static final int INITIAL_CHECKS = 100;
        private static final long MINIMUM_INTERVAL_NANOS = 1000L;
        private final long targetIntervalNanos;
        private final long flexibleTargetIntervalNanos;
        private final Threading.RecurringCallback callback;
        private int requestedChecks;
        private double ewmaChecksPerNano;
        private long lastCapture;
        private long lastCallbackExecution;
        private volatile boolean isExecuting = false;

        RecurringCallbackTimer(long targetIntervalNanos, Threading.RecurringCallback callback) {
            long now;
            this.targetIntervalNanos = Math.max(1000L, targetIntervalNanos);
            this.flexibleTargetIntervalNanos = (long)((double)targetIntervalNanos * 0.95);
            this.callback = callback;
            this.lastCapture = now = System.nanoTime();
            this.lastCallbackExecution = now;
            this.requestedChecks = 100;
        }

        @Uninterruptible(reason="Must not contain safepoint checks.")
        public void evaluate() {
            this.updateStatistics();
            try {
                this.executeCallback();
            }
            finally {
                this.updateSafepointRequested();
            }
        }

        @Uninterruptible(reason="Must be uninterruptible to avoid races with the safepoint code.")
        public void updateStatistics() {
            long now = System.nanoTime();
            long elapsedNanos = now - this.lastCapture;
            int skippedChecks = RecurringCallbackTimer.getSkippedChecks(CurrentIsolate.getCurrentThread());
            int executedChecks = this.requestedChecks - skippedChecks;
            assert (executedChecks >= 0);
            if (elapsedNanos > 0L && executedChecks > 0) {
                double checksPerNano = (double)executedChecks / (double)elapsedNanos;
                this.ewmaChecksPerNano = this.ewmaChecksPerNano == 0.0 ? checksPerNano : 0.3 * checksPerNano + 0.7 * this.ewmaChecksPerNano;
                this.lastCapture = now;
            }
        }

        @Uninterruptible(reason="Must be uninterruptible to avoid races with the safepoint code.")
        private static int getSkippedChecks(IsolateThread thread) {
            int rawValue = Safepoint.getSafepointRequested(thread);
            return rawValue >= 0 ? rawValue : -rawValue;
        }

        @Uninterruptible(reason="Called by uninterruptible code.")
        private void executeCallback() {
            block7: {
                if (this.isCallbackDisabled()) {
                    return;
                }
                this.isExecuting = true;
                try {
                    if (System.nanoTime() < this.lastCallbackExecution + this.flexibleTargetIntervalNanos) break block7;
                    this.setSafepointRequested(Integer.MAX_VALUE);
                    try {
                        this.invokeCallback();
                    }
                    finally {
                        this.lastCallbackExecution = System.nanoTime();
                        this.updateStatistics();
                    }
                }
                finally {
                    this.isExecuting = false;
                }
            }
        }

        @Uninterruptible(reason="Called by uninterruptible code.")
        private void updateSafepointRequested() {
            long nextDeadline = this.lastCallbackExecution + this.targetIntervalNanos;
            long remainingNanos = nextDeadline - System.nanoTime();
            if (remainingNanos < 0L && this.isCallbackDisabled()) {
                this.setSafepointRequested(Integer.MAX_VALUE);
            } else {
                double checks = this.ewmaChecksPerNano * (double)(remainingNanos = remainingNanos < 1000L ? 1000L : remainingNanos);
                this.setSafepointRequested(checks > 2.147483647E9 ? Integer.MAX_VALUE : (checks < 1.0 ? 1 : (int)checks));
            }
        }

        @Uninterruptible(reason="Called by uninterruptible code.")
        public void setSafepointRequested(int value) {
            this.requestedChecks = value;
            Safepoint.setSafepointRequested(value);
        }

        @Uninterruptible(reason="Called by uninterruptible code.")
        private boolean isCallbackDisabled() {
            return this.isExecuting || ThreadingSupportImpl.isRecurringCallbackPaused();
        }

        @Uninterruptible(reason="Required by caller, but does not apply to callee.", calleeMustBe=false)
        @RestrictHeapAccess(reason="Callee may allocate", access=RestrictHeapAccess.Access.UNRESTRICTED, overridesCallers=true)
        private void invokeCallback() {
            try {
                this.callback.run(CALLBACK_ACCESS);
            }
            catch (Safepoint.SafepointException se) {
                throw se;
            }
            catch (Throwable t) {
                Log.log().string("Exception caught in recurring callback (ignored): ").object(t).newline();
            }
        }
    }

    public static class Options {
        @Option(help={"Support a per-thread timer that is called at a specific interval."})
        public static final HostedOptionKey<Boolean> SupportRecurringCallback = new HostedOptionKey<Boolean>(true);
        @Option(help={"Test whether a thread's recurring callback is pending on each transition from native code to Java."})
        public static final HostedOptionKey<Boolean> CheckRecurringCallbackOnNativeToJavaTransition = new HostedOptionKey<Boolean>(false);
    }
}

