/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: PollenActionSupport.java 3738 2012-11-06 20:09:20Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.5/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/PollenActionSupport.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.inject.Inject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.struts2.interceptor.ParameterAware;
import org.apache.struts2.interceptor.ServletRequestAware;
import org.chorem.pollen.PollenApplicationContext;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.PollenUserSecurityContext;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.Polls;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.services.DefaultPollenServiceContext;
import org.chorem.pollen.services.PollenServiceContext;
import org.chorem.pollen.services.PollenServiceFactory;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.impl.FavoriteService;
import org.chorem.pollen.services.impl.PollCommentService;
import org.chorem.pollen.services.impl.PollExportService;
import org.chorem.pollen.services.impl.PollFeedService;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.services.impl.PollUrlService;
import org.chorem.pollen.services.impl.PollVoteCountingService;
import org.chorem.pollen.services.impl.PreventRuleService;
import org.chorem.pollen.services.impl.SecurityService;
import org.chorem.pollen.services.impl.UserService;
import org.chorem.pollen.services.impl.VoteService;
import org.chorem.pollen.ui.PollenSession;
import org.chorem.pollen.ui.interceptors.PollenSecurityInterceptor;
import org.chorem.pollen.ui.security.AbstractPollenAuthorization;
import org.chorem.pollen.votecounting.VoteCounting;
import org.chorem.pollen.votecounting.VoteCountingFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.web.filter.TopiaTransactionFilter;
import org.nuiton.web.struts2.BaseAction;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import java.net.URL;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Pollen action support.
 * <p/>
 * <b>Note :</b> All actions must implement this class to have a better i18n
 * support, when a i18n key is not translated it has a empty translation and
 * we do NOT want this behaviour in gui, prefer to return the marked
 * untranslated key.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenActionSupport extends BaseAction implements ServletRequestAware {

    private static final long serialVersionUID = 1L;

    public static PollenApplicationContext getPollenApplicationContext() {
        PollenApplicationContext applicationContext =
                PollenApplicationContext.get(ActionContext.getContext());
        return applicationContext;
    }

    public PollenConfiguration getConfiguration() {
        return getPollenApplicationContext().getConfiguration();
    }

    /** Pollen User session. */
    private transient PollenSession pollenSession;

    /**
     * Provides a way to get a service.
     * <p/>
     * Actions may <strong>not</strong> call it directly by use
     * {@link #getService(Class)} instead.
     */
    @Inject
    protected transient PollenServiceFactory serviceFactory;

    protected transient PollenServiceContext serviceContext;

    protected transient TopiaContext transaction;

    private transient Map<Class<? extends PollenServiceSupport>, PollenServiceSupport> services;

    /**
     * The incoming request (some stuff are store in it from security filters).
     *
     * @since 1.4.5
     */
    private transient HttpServletRequest request;

    /**
     * All the parameters send by request used to build back choices of the
     * poll.
     * <p/>
     * This will be setted by the {@link ParameterAware#setParameters(Map)} as soon
     * as your action implements {@link ParameterAware}.
     *
     * @since 1.4.5
     */
    private Map<String, String[]> parameters;

    /**
     * The user security context for this request.
     * <p/>
     * This security contains the {@code poll}, {@code userAccount} and {@code userRoles}.
     * <p/>
     * It will be injected by the {@link PollenSecurityInterceptor} for all
     * action which implements {@link PollenUserSecurityAware}.
     * <p/>
     * <strong>Import Note:</strong> Security context is load by shiro filter
     * via method {@link AbstractPollenAuthorization#createSecurityContext(ServletRequest)}.
     *
     * @since 1.4.5
     */
    private PollenUserSecurityContext securityContext;

    private final PageSkin skin;

    public PollenActionSupport() {
        // by default use the index skin
        this(PageSkin.INDEX);
    }

    public PollenActionSupport(PageSkin skin) {
        this.skin = skin;
    }

    public final PageSkin getSkin() {
        return skin;
    }

    public final String getPageLogo() {
        return getSkin().getCssSuffix();
    }

    public final String getJqueryTheme() {
        return getSkin().getTheme();
    }

    public PollenSession getPollenSession() {
        if (pollenSession == null) {
            pollenSession = PollenSession.get(ActionContext.getContext());
        }
        return pollenSession;
    }

    public UserAccount getPollenUserAccount() {
        return getPollenSession().getUserAccount();
    }

    public boolean isUserLoggued() {
        UserAccount userAccount = getPollenUserAccount();
        return userAccount != null;
    }

    public boolean isUserAdmin() {
        UserAccount userAccount = getPollenUserAccount();
        return userAccount != null && userAccount.isAdministrator();
    }

    public static String getApplicationVersion() {
        return getPollenApplicationContext().getConfiguration().getVersion().toString();
    }

    public URL getApplicationUrl() {
        URL url = getServiceContext().getApplicationURL();
        return url;
    }

    /**
     * Keep in this action the incoming request.
     *
     * @param request the http request to set (can not be null)
     * @see ServletRequestAware
     * @since 1.4.5
     */
    @Override
    public final void setServletRequest(HttpServletRequest request) {
        Preconditions.checkNotNull(request);
        this.request = request;
    }

    /**
     * Keep in this action the incoming request parameters.
     * <p/>
     * <strong>Note:</strong> This method will be invoked only if your action
     * implements the {@link ParameterAware} contract.
     *
     * @param parameters paramters of the request (can not be null)
     * @see ParameterAware
     * @since 1.4.5
     */
    public final void setParameters(Map<String, String[]> parameters) {
        Preconditions.checkNotNull(parameters);
        this.parameters = Maps.newHashMap(parameters);
    }

    /**
     * Keep in this action the incoming security context.
     * <p/>
     * <strong>Note:</strong> This method will be invoked only if your action
     * implements the {@link PollenUserSecurityAware} contract.
     *
     * @param pollenUserSecurityContext pollen security context to set (can not be null)
     * @see PollenUserSecurityAware
     * @since 1.4.5
     */
    public final void setPollenUserSecurityContext(PollenUserSecurityContext pollenUserSecurityContext) {
        Preconditions.checkNotNull(pollenUserSecurityContext);
        this.securityContext = pollenUserSecurityContext;
    }

    /**
     * Gets the {@link PollUri} from the incoming request.
     * <p/>
     * If the action is {@link PollenUserSecurityAware}, then will take it
     * from {@link PollenUserSecurityContext#getPollUri()}, otherwise will
     * return {@code null} (this means you must override this method in your action.
     *
     * @return the pollUri register in this action or null.
     * @since 1.4.5
     */
    public PollUri getPollUri() {
        PollUri result;
        if (this instanceof PollenUserSecurityAware) {
            // use the security context to ge the pollUri
            result = getUserSecurityContext().getPollUri();
        } else {
            // No information from here, must be override in your action
            result = null;
        }
        return result;
    }

    public VoteCountingFactory getVoteCountingFactory() {
        return getPollenApplicationContext().getVoteCountingFactory();
    }

    public TopiaContext getTransaction() {
        if (transaction == null) {
            transaction = TopiaTransactionFilter.getTransaction(request);
        }
        return transaction;
    }

    public static final List<String> SAFE_LANGUAGES =
            Lists.newArrayList("en", "fr");

    public String getSafeLanguage() {
        String language = getLocale().getLanguage();
        if (!SAFE_LANGUAGES.contains(language)) {
            language = SAFE_LANGUAGES.get(0);
        }
        return language;
    }

    public String getCurrentUrl() {
        String result = request.getRequestURL().toString();
        return result;
    }

    public void addFlashMessage(String message) {
        getPollenSession().addMessage(PollenSession.SESSION_TOKEN_MESSAGES, message);
    }

    public void addFlashError(String message) {
        getPollenSession().addMessage(PollenSession.SESSION_TOKEN_ERRORS, message);
    }

    public void addFlashWarning(String message) {
        getPollenSession().addMessage(PollenSession.SESSION_TOKEN_WARNINGS, message);
    }

    public Collection<String> getFlashMessages() {
        Collection<String> result = getPollenSession().consumeDynamicSetData(PollenSession.SESSION_TOKEN_MESSAGES);
        return result;
    }

    public Collection<String> getFlashErrors() {
        Collection<String> result = getPollenSession().consumeDynamicSetData(PollenSession.SESSION_TOKEN_ERRORS);
        return result;
    }

    public Collection<String> getFlashWarnings() {
        Collection<String> result = getPollenSession().consumeDynamicSetData(PollenSession.SESSION_TOKEN_WARNINGS);
        return result;
    }

    public boolean hasFlashMessages() {
        Collection<String> result = getPollenSession().getDynamicSetData(PollenSession.SESSION_TOKEN_MESSAGES);
        return CollectionUtils.isNotEmpty(result);
    }

    public boolean hasFlashErrors() {
        Collection<String> result = getPollenSession().getDynamicSetData(PollenSession.SESSION_TOKEN_ERRORS);
        return CollectionUtils.isNotEmpty(result);
    }

    public boolean hasFlashWarnings() {
        Collection<String> result = getPollenSession().getDynamicSetData(PollenSession.SESSION_TOKEN_WARNINGS);
        return CollectionUtils.isNotEmpty(result);
    }

    public void clearFlashMessages() {
        Collection<String> result = getPollenSession().getDynamicSetData(PollenSession.SESSION_TOKEN_MESSAGES);
        if (result != null) {
            result.clear();
        }
    }

    public void clearFlashErrors() {
        Collection<String> result = getPollenSession().getDynamicSetData(PollenSession.SESSION_TOKEN_ERRORS);
        if (result != null) {
            result.clear();
        }
    }

    public boolean hasAnyErrors() {
        boolean result = super.hasErrors();
        return result || hasFlashErrors();
    }

    protected final PollService getPollService() {
        return getService(PollService.class);
    }

    protected final PollUrlService getPollUrlService() {
        return getService(PollUrlService.class);
    }

    protected final PollExportService getPollExportService() {
        return getService(PollExportService.class);
    }

    protected final PollVoteCountingService getPollVoteCountingService() {
        return getService(PollVoteCountingService.class);
    }

    protected final VoteService getVoteService() {
        return getService(VoteService.class);
    }

    protected final PollCommentService getPollCommentService() {
        return getService(PollCommentService.class);
    }

    protected final PollFeedService getPollFeedService() {
        return getService(PollFeedService.class);
    }

    protected final UserService getUserService() {
        return getService(UserService.class);
    }

    protected final FavoriteService getFavoriteService() {
        return getService(FavoriteService.class);
    }

    protected final PreventRuleService getPreventRuleService() {
        return getService(PreventRuleService.class);
    }

    protected final SecurityService getSecurityService() {
        return getService(SecurityService.class);
    }

    /**
     * Get a instance of a service for the given {@code serviceType}.
     *
     * @param serviceType type of the required service
     * @param <S>         type of the required service
     * @return instance of the service
     * @since 1.4.5
     */
    protected final <S extends PollenServiceSupport> S getService(Class<S> serviceType) {
        if (services == null) {
            services = Maps.newHashMap();
        }
        PollenServiceSupport result = services.get(serviceType);
        if (result == null) {
            result = serviceFactory.newService(serviceType, getServiceContext());
            services.put(serviceType, result);
        }
        return (S) result;
    }

    /**
     * Tests if the incoming http request is a get method.
     * <p/>
     * <strong>Note:</strong> {@link #request} must not be null.
     *
     * @return {@code true} if incoming request uses {@code GET} method.
     * @since 1.4.5
     */
    protected boolean isGetMethod() {
        Preconditions.checkNotNull(request);
        return "GET".equalsIgnoreCase(request.getMethod());
    }

    /**
     * Obtain the injected security context.
     * <p/>
     * <strong>Note:</strong> {@link #securityContext} must not be null
     * (concrete action should implements {@link PollenUserSecurityAware}).
     *
     * @return the injected security context
     * @see PollenUserSecurityContext
     * @see PollenUserSecurityAware
     * @since 1.4.5
     */
    protected final PollenUserSecurityContext getUserSecurityContext() {
        Preconditions.checkNotNull(securityContext, "To use the pollenUserSecurityContext, make your action implements PollenUserSecurityAware.");
        return securityContext;
    }

    /**
     * Obtain the injected request parameters.
     * <p/>
     * <strong>Note:</strong> {@link #parameters} must not be null
     * (concrete action should implements {@link ParameterAware}).
     *
     * @return the injected request parameters
     * @see ParameterAware
     * @since 1.4.5
     */
    protected final Map<String, String[]> getParameters() {
        Preconditions.checkNotNull(parameters);
        return parameters;
    }

    protected VoteCounting getVoteCounting(Poll poll) {
        return Polls.getVoteCounting(getVoteCountingFactory(), poll);
    }

    /**
     * Fabrique pour récupérer le ServiceContext tel qu'il devrait être fourni
     * à la fabrication d'un service.
     *
     * @return service context
     */
    protected PollenServiceContext getServiceContext() {
        if (serviceContext == null) {
            serviceContext = DefaultPollenServiceContext.newContext(
                    getLocale(),
                    getTransaction(),
                    getConfiguration(),
                    serviceFactory,
                    getVoteCountingFactory()
            );
        }
        return serviceContext;
    }

}
