/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: ResultForPoll.java 3611 2012-08-13 14:11:15Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/ResultForPoll.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.opensymphony.xwork2.inject.Inject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.views.util.UrlHelper;
import org.chorem.pollen.bean.PollResult;
import org.chorem.pollen.bean.PollResultList;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.services.impl.PollResultsService;
import org.chorem.pollen.ui.PollenUIUtils;
import org.chorem.pollen.ui.actions.PageSkin;
import org.chorem.pollen.ui.actions.PollUriAware;
import org.chorem.pollen.ui.actions.PollenActionSupport;
import org.chorem.pollen.ui.actions.PollenUserSecurityAware;
import org.chorem.pollen.ui.converters.DateConverter;
import org.chorem.pollen.votecounting.strategy.VoteCountingStrategy;
import org.nuiton.util.StringUtil;

import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Display results of a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class ResultForPoll extends PollenActionSupport implements PollenUserSecurityAware, PollUriAware {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(ResultForPoll.class);

    /**
     * Is result by group ?
     *
     * @since 1.3
     */
    private Boolean byGroup;

    /**
     * Is user is poll's creator ?
     *
     * @since 1.4
     */
    private boolean creatorUser;

    /**
     * Is feed exists for this poll ?
     *
     * @since 1.4
     */
    private boolean feedFileExisting;

    /**
     * Résultats du sondage.
     * <p/>
     * La map est indexée par les noms de choix.
     *
     * @since 1.3
     */
    private transient Multimap<String, String> choicesResults;

    /**
     * Résultats du sondage.
     *
     * @since 1.3
     */
    private transient Multimap<String, String> subtitles;

    /**
     * Loaded poll.
     *
     * @since 1.3
     */
    private Poll poll;

    /**
     * Poll results.
     *
     * @since 1.3
     */
    private List<PollResult> results;

    /**
     * Top ranking results.
     *
     * @since 1.3
     */
    private List<PollResult> topRanking;

    /**
     * Url helper (to build new url)
     *
     * @since 1.3.1
     */
    @Inject(required = true)
    private UrlHelper urlHelper;

    public ResultForPoll() {
        super(PageSkin.RESULT);
    }

    public Poll getPoll() {
        return poll;
    }

    public boolean isCreatorOrAdmin() {
        return creatorUser || isUserAdmin();
    }

    public boolean isFeedFileExisting() {
        return feedFileExisting;
    }

    public String getVoteCountingTypeName() {
        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());
        String result = strategy.getStrategyName(getLocale());
        return result;
    }

    public String getVoteCountingTypeHelp() {
        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());
        String result = strategy.getStrategyHelp(getLocale());
        return result;
    }

    public String getSummaryUrl() {
        PollUrl url = getPollUrlService().getPollSummaryUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        return url.getUrl();
    }

    public String getResultUrl(boolean byGroup) {
        PollUrl url = getPollUrlService().getPollResultUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        String result = url.getUrl();
        if (!byGroup) {
            result += "?byGroup=false";
        }
        return result;
    }

    public String getVoteUrl() {
        PollUrl url = getPollUrlService().getPollVoteUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        if (poll.isPollFree()) {

            // can removed accountId only for free poll
            //FIXME Should found out in ohter case the accountId (if exists for the connected id) if no accountId is given
            getUserSecurityContext().removeAccountIdWhenConnected(url);
        }
        return url.getUrl();
    }

    public List<PollResult> getResults() {
        return results;
    }

    public String getPollCreatorName() {
        PollAccount creator = poll.getCreator();
        String result = creator.getVotingId();
        if (StringUtils.isBlank(result)) {
            result = creator.getEmail();
        }
        if (StringUtils.isBlank(result)) {
            result = _("pollen.common.undefined");
        }
        return result;
    }

    public String getPollBeginDate() {
        Date date = poll.getBeginDate();
        String result = date == null ? _("pollen.common.undefined") :
                        getPollService().decorateDate(date);
        return result;
    }

    public String getPollEndDate() {
        Date date = poll.getEndDate();
        String result = date == null ? _("pollen.common.undefined") :
                        getPollService().decorateDate(date);
        return result;
    }

    public Date getResultAsDate(PollResult result) {
        Date date = new Date(Long.valueOf(result.getName()));
        return date;
    }

    public Multimap<String, String> getChoicesResults() {
        return choicesResults;
    }

    public Multimap<String, String> getSubtitles() {
        return subtitles;
    }

    public String getVoteSizeMessage() {
        String result = PollenUIUtils.getVoteSizeMessage(getPoll(),
                                                         getLocale());
        return result;
    }

    public String getVictoryMessage() {

        String victoryMessage;

        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());
        boolean displayResultsByChoice = strategy.isDisplayResultsByChoice();
        //FIXME-tchemit-2012-06-24, why doing this ??? Should remove this...
//        if (isNumberVoteCounting()) {
        if (displayResultsByChoice) {
            victoryMessage = null;
        } else if (CollectionUtils.isEmpty(topRanking)) {
            victoryMessage = "";
        } else if (topRanking.size() == 1) {
            victoryMessage = _("pollen.common.victory");
        } else {
            victoryMessage = _("pollen.common.victories");
        }
        return victoryMessage;
    }

    /**
     * @param result la résultat à utiliser.
     * @return la valeur du résultat courant sans le .0 final
     */
    public String getTrimValue(PollResult result) {

        String value = result.getValue();

        // le résultat peut-être un double : 1,0 -> 1 et 1,2 -> 1,2
        if (value.endsWith(".0")) {
            value = value.substring(0, value.indexOf('.'));
        }
        return value;
    }

    @Override
    public String execute() throws Exception {
        poll = getUserSecurityContext().getPoll();

        feedFileExisting = getPollFeedService().isFeedExists(poll);

        if (byGroup == null) {

            // default byGroup value
            byGroup = isGroupPoll();
        }

        creatorUser = getUserSecurityContext().isCreator();

        if (poll.isRunning(serviceContext.getCurrentTime())) {
            addFlashWarning(_("pollen.information.pollRunning"));
        }

        // load results

        PollResultsService service = getPollResultsService();

        PollResultList resultListDTO = service.getResults(poll);

        results = resultListDTO.getPollResults();

        if (log.isDebugEnabled()) {
            for (PollResult res : results) {
                log.debug(res
                          + ", (voteCounting=" + res.getVoteCountingType()
                          + ", byGroup=" + res.isByGroup() + ")");
            }
        }

        VoteCountingStrategy strategy = getVoteCountingStrategy(poll);
        if (strategy.isDisplayResultsByChoice()) {

            // resultats des choix cachés
            choicesResults = LinkedHashMultimap.create();

            // sous-titres des vrais choix (les choix pas cachés)
            subtitles = ArrayListMultimap.create();

            results = service.createNumberVoteCountingResult(
                    resultListDTO,
                    choicesResults,
                    subtitles);
        }

        topRanking = service.getTopRanking(results);

        if (log.isDebugEnabled()) {
            log.debug("topRanking = " + topRanking);
        }

        return SUCCESS;
    }

    public List<PollResult> getTopRanking() {
        return topRanking;
    }

    public boolean isFreePoll() {
        return poll.isPollFree();
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isByGroup() {
        return byGroup;
    }

    public void setByGroup(boolean byGroup) {
        this.byGroup = byGroup;
    }

    /** @return Récupération de la liste des résultats pour le diagramme. */
    public List<String> getChoices() {

        List<String> choiceValues = Lists.newArrayList();

        for (PollResult result : results) {
            String name = result.getName();
            String value = result.getValue();

            if (isDateType()) {
                // mise en forme de la date
                Date date = new Date(Long.parseLong(name));
                name = DateConverter.convertToString(date);
            }

            choiceValues.add(name);
            choiceValues.add(value);
        }
        return choiceValues;
    }

    public String getNormalPollResultChartUrl(int width, int height) {
        URL applicationUrl = getApplicationUrl();
        StringBuilder url = new StringBuilder(applicationUrl.toString());

        url.append("/io/generateChart");

        Map<String, Object> params = Maps.newHashMap();
        params.put("width", width);
        params.put("height", height);
        params.put("title", "");
        params.put("values", getChoices());

        urlHelper.buildParametersString(params, url, "&");

        String chartUrl = url.toString();
        return chartUrl;
    }

    public Set<String> getChoicesResultNames() {
        return choicesResults.keySet();
    }

    public String getChoicesResultCharUrl(int width,
                                          int height,
                                          String choiceName) {

        URL applicationUrl = getApplicationUrl();
        StringBuilder url = new StringBuilder(applicationUrl.toString());

        url.append("/io/generateChart");

        Map<String, Object> params = Maps.newHashMap();
        params.put("width", width);
        params.put("height", height);
        params.put("title", "");
        params.put("values", choicesResults.get(choiceName));
        params.put("subtitles", subtitles.get(choiceName));

        urlHelper.buildParametersString(params, url, "&");

        String chartUrl = url.toString();
        return chartUrl;

    }

    public String getChoicesAsString() {
        List<String> choices = getChoices();
        return StringUtil.join(choices, ",", true);
    }

}
