/*
 * #%L
 * Pollen :: UI (strust2)
 * 
 * $Id: AbstractPollForm.java 3395 2012-05-28 15:34:46Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/AbstractPollForm.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.I18nAble;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.services.PollenServiceFunctions;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.ui.actions.PageSkin;
import org.chorem.pollen.ui.actions.PollenActionSupport;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Created: 04/04/12
 *
 * @author fdesbois <desbois@codelutin.com>
 *         $Id: AbstractPollForm.java 3395 2012-05-28 15:34:46Z tchemit $
 */
public abstract class AbstractPollForm extends PollenActionSupport {

    private static final long serialVersionUID = 1L;

    protected Poll poll;

    private Map<String, String> pollTypes;

    private Map<String, String> voteCountingTypes;

    private Map<String, String> choiceTypes;

    /** Text choices. */
    private List<Choice> textChoices;

    /** Image choices. */
    private List<Choice> imageChoices;

    /** Date choices. */
    private List<Choice> dateChoices;

    /** restricted Voting lists. */
    private List<VotingList> restrictedVotingList;

    /** Group voting lists. */
    private List<VotingList> groupVotingList;

    private boolean notification;

    private boolean reminder;

    private boolean limitChoice;

    private int reminderHourCountdown = 2;

    private transient PollService pollService;

    /** To create a new personToList. */
    private transient Function<PersonToList, PersonToList> persontoListCreator;

    /** To create a new votingList. */
    private transient Function<VotingList, VotingList> votingListCreator;

    /** To create a new text choice. */
    private transient Function<Choice, Choice> textChoiceCreator;

    /** To create a new date choice. */
    private transient Function<Choice, Choice> dateChoiceCreator;

    /** To create a new image choice. */
    private transient Function<Choice, Choice> imageChoiceCreator;

    @Override
    public PageSkin getSkin() {
        return PageSkin.EDITION;
    }

    protected PollService getPollService() {
        if (pollService == null) {
            pollService = newService(PollService.class);
        }
        return pollService;
    }

    public Poll getPoll() {
        return poll;
    }

    public boolean isInformationsError() {
        return false;
    }

    public boolean isOptionsError() {
        return false;
    }

    public Map<String, String> getPollTypes() {
        return pollTypes;
    }

    public Map<String, String> getVoteCountingTypes() {
        return voteCountingTypes;
    }

    public Map<String, String> getChoiceTypes() {
        return choiceTypes;
    }

    public List<Choice> getTextChoices() {
        return textChoices;
    }

    public List<Choice> getImageChoices() {
        return imageChoices;
    }

    public List<Choice> getDateChoices() {
        return dateChoices;
    }

    public List<VotingList> getRestrictedVotingList() {
        return restrictedVotingList;
    }

    public List<VotingList> getGroupVotingList() {
        return groupVotingList;
    }

    public boolean isNotification() {
        return notification;
    }

    public void setNotification(boolean notification) {
        this.notification = notification;
    }

    public boolean isReminder() {
        return reminder;
    }

    public void setReminder(boolean reminder) {
        this.reminder = reminder;
    }

    public boolean isLimitChoice() {
        return limitChoice;
    }

    public void setLimitChoice(boolean limitChoice) {
        this.limitChoice = limitChoice;
    }

    public int getReminderHourCountdown() {
        return reminderHourCountdown;
    }

    public void setReminderHourCountdown(int reminderHourCountdown) {
        this.reminderHourCountdown = reminderHourCountdown;
    }

    public boolean isFreePoll() {
        PollType pollType = poll.getPollType();
        return pollType == PollType.FREE;
    }

    public boolean isGroupPoll() {
        PollType pollType = poll.getPollType();
        return pollType == PollType.GROUP;
    }

    public String getPollVoteUrl() {
        return getPollService().getPollVoteUrl(poll, false).getUrl();
    }

    public String getPollModerateUrl() {
        return getPollService().getPollVoteUrl(poll, true).getUrl();
    }

    public String getPollEditUrl() {
        return getPollService().getPollEditUrl(poll).getUrl();
    }

    public int getSelectedTab() {
        int result = 0;
        return result;
    }

    public String getActionLabel() {
        return isEdit() ? _("pollen.action.editPoll") :
               _("pollen.action.createPoll");
    }

    public String getPageTitle() {
        return isEdit() ? getPoll().getTitle() :
               _("pollen.title.createPoll");
    }

    public boolean isVoteStarted() {
        return isEdit() && poll.sizeVote() > 0;
    }

//    public boolean isPersonEditable(PersonToList personToList) {
//        boolean result;
//        // If vote is started, the person is editable if he hasn't vote yet
//        if (isVoteStarted()) {
//            result = !personToList.isHasVoted();
//
//        } else {
//            result = true;
//        }
//        return result;
//    }

    public boolean isCreatorUserAccountDefined() {
        PollAccount creator = poll.getCreator();
        return creator.getUserAccount() != null;
    }

    public boolean isEdit() {
        return StringUtils.isNotEmpty(getPoll().getTopiaId());
    }

    public void prepareFormPage() throws Exception {

        pollTypes = decorateToName(PollType.values());
        choiceTypes = decorateToName(ChoiceType.values());
        voteCountingTypes = decorateToName(VoteCountingType.values());

        textChoices = Lists.newArrayList();
        imageChoices = Lists.newArrayList();
        dateChoices = Lists.newArrayList();

        restrictedVotingList = Lists.newArrayList();
        groupVotingList = Lists.newArrayList();
    }

    public Function<PersonToList, PersonToList> getPersontoListCreator() {
        if (persontoListCreator == null) {
            persontoListCreator = PollenServiceFunctions.newPersonToListCreator();
        }
        return persontoListCreator;
    }

    public Function<VotingList, VotingList> getVotingListCreator() {
        if (votingListCreator == null) {
            votingListCreator = PollenServiceFunctions.newVotingListCreator(getPersontoListCreator());
        }
        return votingListCreator;
    }

    public Function<Choice, Choice> getTextChoiceCreator() {
        if (textChoiceCreator == null) {
            textChoiceCreator = PollenServiceFunctions.newTextChoiceCreator();
        }
        return textChoiceCreator;
    }

    public Function<Choice, Choice> getDateChoiceCreator() {
        if (dateChoiceCreator == null) {
            dateChoiceCreator = PollenServiceFunctions.newDateChoiceCreator();
        }
        return dateChoiceCreator;
    }

    public Function<Choice, Choice> getImageChoiceCreator() {
        if (imageChoiceCreator == null) {
            imageChoiceCreator = PollenServiceFunctions.newImageChoiceCreator();
        }
        return imageChoiceCreator;
    }

    private <E extends Enum<E> & I18nAble> Map<String, String> decorateToName(E... values) {
        Map<String, String> result = Maps.newLinkedHashMap();
        for (E value : values) {
            result.put(value.name(), getText(value.getI18nKey()));
        }
        return result;
    }

    protected void loadChoicesAndvotingLists(Poll poll,
                                             Collection<Choice> pollChoices,
                                             Collection<VotingList> pollVotinLists,
                                             boolean fillLists) {

        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(poll.getChoiceType());
        Preconditions.checkNotNull(poll.getPollType());


        if (CollectionUtils.isNotEmpty(pollChoices)) {

            //push back choices

            switch (poll.getChoiceType()) {

                case TEXT:
                    getTextChoices().addAll(pollChoices);
                    break;
                case DATE:
                    getDateChoices().addAll(pollChoices);
                    break;
                case IMAGE:
                    getImageChoices().addAll(pollChoices);
                    break;
            }
        }

        if (CollectionUtils.isNotEmpty(pollVotinLists)) {

            // push back voting lists

            switch (poll.getPollType()) {

                case FREE:
                    // not voting lists
                    break;
                case RESTRICTED:
                    getRestrictedVotingList().addAll(pollVotinLists);
                    break;
                case GROUP:
                    getGroupVotingList().addAll(pollVotinLists);
                    break;
            }
        }

        if (fillLists) {

            // refill lists

            int defaultMaxChoices = 5;
            int defaultMaxVoting = 5;

            fillLists(defaultMaxChoices, defaultMaxVoting);
        }
    }

    protected void fillLists(int defaultMaxChoices, int defaultMaxVoting) {

        PollenServiceFunctions.fillChoiceList(textChoices,
                                              defaultMaxChoices,
                                              getTextChoiceCreator());

        PollenServiceFunctions.fillChoiceList(dateChoices,
                                              defaultMaxChoices,
                                              getDateChoiceCreator());

        PollenServiceFunctions.fillChoiceList(imageChoices,
                                              defaultMaxChoices,
                                              getImageChoiceCreator());

        if (CollectionUtils.isEmpty(restrictedVotingList)) {
            restrictedVotingList.add(getVotingListCreator().apply(null));
        }
        for (VotingList votingList : restrictedVotingList) {
            PollenServiceFunctions.fillVotingList(votingList,
                                                  getPersontoListCreator(),
                                                  defaultMaxVoting);
        }
        if (CollectionUtils.isEmpty(groupVotingList)) {
            groupVotingList.add(getVotingListCreator().apply(null));
        }
        for (VotingList votingList : groupVotingList) {
            PollenServiceFunctions.fillVotingList(votingList,
                                                  getPersontoListCreator(),
                                                  defaultMaxVoting);
        }

    }
}
