/**
 * *##% Callao LogServiceImpl
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.LogFactory;
import org.chorem.callao.entity.CallaoDAOHelper;
import org.chorem.callao.entity.Log;
import org.chorem.callao.entity.LogDAO;
import org.chorem.callao.service.convertObject.ConvertLog;
import org.chorem.callao.service.dto.LogDTO;
import org.chorem.callao.service.utils.ContextCallao;
import org.chorem.callao.service.utils.ServiceHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Permet d'ajouter des logs au systeme, ATTENTION on ne peut supprimer
 * un log.
 * @author Rémi Chapelet
 */

public class LogServiceImpl { //implements LogService {

    /** log */
    private static final org.apache.commons.logging.Log log = LogFactory.getLog(LogServiceImpl.class);

    private TopiaContext rootContext = ContextCallao.getInstance().getContext();

    private ConvertLog convertLog = new ConvertLog();

    /**
     * Création d'un log dans le systeme
     * @param logDate date de creation du log
     * @param transDate date de la transaction
     * @param type modification (add, mod, tmod ou del)
     * @param voucherRef reference du document justificatif
     * @param transDesc description de la transaction
     * @param entryDesc description de l'entry
     * @param amount montant de l'entry
     * @param debit debit (vrai) ou credit (faux)
     * @param lettering letterage de l'entry
     * @return
     */
    //@Override
	public String addLog(Date logDate, Date transDate, String type,
			String voucherRef, String transDesc, String entryDesc,
			String amount, boolean debit, String lettering) {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            LogDAO logDAO = CallaoDAOHelper.getLogDAO(topiaContext);
            // Creation du log
			Log newlog = logDAO.create();
            newlog.setLogDate(logDate);
            newlog.setTransDate(transDate);
            newlog.setVoucherRef(voucherRef);
            newlog.setType(type);
            newlog.setTransDesc(transDesc);
            newlog.setEntryDesc(entryDesc);
            newlog.setAmount(amount);
            newlog.setDebit(debit);
            newlog.setLettering(lettering);    
            // Création BDD
            topiaContext.commitTransaction();
            // Fermeture BDD
			topiaContext.closeContext();
            // Affichage pour les logs
            String topiaId = newlog.getTopiaId();
            if (log.isInfoEnabled()) {
                log.info("Log created: "+ topiaId);
            }
            result =  ServiceHelper.RESPOND_SUCCESS;
        }catch (TopiaException e) {
			log.error(e);
		}
        return result;
	}


    /**
     * Création d'un log à partir d'un log DTO.
     * @param logDTO log en format DTO qu'on souhaite créer
     * @return
     */
    public String addLog (LogDTO logDTO)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Vérifie si logDTO existe
        if (logDTO != null)
        {
            result = addLog(logDTO.getLogDate(),logDTO.getTransDate(),logDTO.getType(),
                    logDTO.getVoucherRef(),logDTO.getTransDesc(),logDTO.getEntryDesc(),
                    logDTO.getAmount(),logDTO.isDebit(),logDTO.getLettering());
        }
        return result;
    }


    /**
     * Recherche tous les logs dans la base de donnees
     * @return la liste des logs
     */
    public List<Log> listeAllLog ()
    {
        List<Log> listelog = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            LogDAO logDAO = CallaoDAOHelper.getLogDAO(topiaContext);
            // recherche des log
			listelog = logDAO.findAll();
            // Fermeture BDD
			topiaContext.closeContext();
            if (log.isInfoEnabled()) {
                log.info("Search all logs");
            }
        }catch (TopiaException e) {
			log.error(e);
		}
		return listelog;
    }


    /**
     * Permet de renvoyer tous les logs au format DTO.
     * @return
     */
    public List<LogDTO> listeAllLogDTO ()
    {
        List<LogDTO> listLogDTO = new ArrayList<LogDTO>();
        // On recherche tous les logs.
        List<Log> listLog = listeAllLog();
        // Pour chaque log, on le converti en logDTO et on l'ajoute à la nouvelle liste de DTO
        for (Log log : listLog)
        {
            // conerti de l'entity en DTO
            LogDTO logDTO = convertLog.logEntityToDto(log);
            // Ajoute à la liste des logs DTO
            listLogDTO.add(logDTO);
        }
        return listLogDTO;
    }


    /**
     * Renvoie une liste de tous les logs suivant la date de creation
     * @param datecreate date de creation des logs recherchés
     * @return une liste des logs trouves
     */
    public List<Log> searchLogWithDateCreate (Date dateCreate)
    {
        List<Log> listelog = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            LogDAO logDAO = CallaoDAOHelper.getLogDAO(topiaContext);
            // recherche des log
			listelog = logDAO.findAllByLogDate(dateCreate);
            // Fermeture BDD
			topiaContext.closeContext();
            if (log.isInfoEnabled()) {
                log.info("Log(s) find : " + listelog.size() + "log(s)");
            }
        }catch (TopiaException e) {
			log.error(e);
		}
        return listelog;
    }


    /**
     * Permet de rechercher tous les logs suivant la date de création, c'est à
     * dire de l'opération effectuée sur une entry à un jour précis, et qui a
     * déclenchée automatiquement une création de log.
     * La méthode renvoie sous forme de liste, en convertissant les objets
     * entitys en DTO.
     * @param dateCreate date de création des logs recherchés
     * @return
     */
    public List<LogDTO> searchLogDTOWithDateCreate (Date dateCreate)
    {
        List<LogDTO> listLogDTO = new ArrayList<LogDTO>();
        // On recherche tous les logs suivant la date de création.
        List<Log> listLog = searchLogWithDateCreate(dateCreate);
        // Pour chaque log, on le converti en logDTO et on l'ajoute à la nouvelle liste de DTO
        for (Log log : listLog)
        {
            // conerti de l'entity en DTO
            LogDTO logDTO = convertLog.logEntityToDto(log);
            // Ajoute à la liste des logs DTO
            listLogDTO.add(logDTO);
        }
        return listLogDTO;
    }


    /**
     * Recherche un log avec son identifiant topiaId.
     * @param topiaId identifiant d'un log
     * @return
     */
    public Log searchLogWithTopiaId (String topiaId)
    {
        Log logSearch = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            LogDAO logDAO = CallaoDAOHelper.getLogDAO(topiaContext);
            // Creation du journal
            logSearch = logDAO.findByTopiaId(topiaId);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return logSearch;
    }


    /**
     * Efface un log : uniquement utilisé pour supprimer les logs dans les tests
     * unitaires.
     * @param topiaId identifiant du log à supprimer
     * @return
     */
    public String removeLog (String topiaId)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        Log logDelete = searchLogWithTopiaId(topiaId);
        // Si le log n'existe pas
        if (logDelete == null)
        {
            if (log.isWarnEnabled()) {
                log.warn("Le log "+topiaId+" n'existe pas !");
            }
        }else // Sinon on efface le log
        {
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                LogDAO logDAO = CallaoDAOHelper.getLogDAO(topiaContext);
                // Supprime le log
                logDAO.delete(logDelete);
                // Création BDD
                topiaContext.commitTransaction();
                // Fermeture BDD
                topiaContext.closeContext();
                result = ServiceHelper.RESPOND_SUCCESS;
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }


    /**
     * Permet de supprimer un log à partir de son DTO.
     * @param logDTO log à supprimer
     * @return
     */
    public String removeLog (LogDTO logDTO)
    {
        String result = removeLog(logDTO.getId());
        return result;
    }


    //@Override
	public String[] getMethods() {
		// TODO Auto-generated method stub
		return null;
	}

    //@Override
	public void destroy() {
		// TODO Auto-generated method stub

	}

    //@Override
	public void init(TopiaContext arg0) {
		// TODO Auto-generated method stub

	}

}
