/**
 * *##% Callao JournalServiceImpl
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.callao.entity.CallaoDAOHelper;
import org.chorem.callao.entity.Journal;
import org.chorem.callao.entity.JournalDAO;
import org.chorem.callao.entity.Transaction;
import org.chorem.callao.entity.TransactionDAO;
import org.chorem.callao.service.convertObject.ConvertJournal;
import org.chorem.callao.service.dto.JournalDTO;
import org.chorem.callao.service.utils.ContextCallao;
import org.chorem.callao.service.utils.ServiceHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémente la fonction multi-journal. Il est possible de créer ici le
 * journal des ventes, le journal des achats, etc.
 *
 * @author Rémi Chapelet
 */
public class JournalServiceImpl {

    private static final Log log = LogFactory.getLog(JournalServiceImpl.class);

	private TopiaContext rootContext = ContextCallao.getInstance().getContext();

    private ConvertJournal convertJournal = new ConvertJournal();

    /**
     * Création d'un journal. Il peut être unique sur son label (au choix)
     * @param label libellé du journal
     * @param prefix préfix du journal (peut être utilisé dans l'UI)
     * @return
     */
    public String createJournal (String label, String prefix, String description)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            JournalDAO journalDAO = CallaoDAOHelper.getJournalDAO(topiaContext);
            /**
             * Si on souhaite avoir un journal unique avec son label
             */
            
            Journal journal = searchJournalWithLabel(label);
            // Si un journal existe déjà avec ce label
            if ( journal != null )
            {
                if (log.isWarnEnabled()) {
                    log.warn("Un journal avec le label "+label+" existe déjà !");
                }
                result = ServiceHelper.JOURNAL_DOUBLE;
            } else { 
                // Creation du journal
                Journal newJournal = journalDAO.create();
                newJournal.setLabel(label);
                newJournal.setPrefix(prefix);
                newJournal.setDescription(description);
                // Création BDD
                topiaContext.commitTransaction();
                result = ServiceHelper.RESPOND_SUCCESS;
            }
            // Fermeture BDD
            topiaContext.closeContext();            
        }catch (TopiaException e) {
            log.error(e);
        }
        return result;
    }


    /**
     * Création d'un journal à partir de son DTO.
     * @param journalDTO Journal au format DTO.
     * @return
     */
    public String createJournal (JournalDTO journalDTO)
    {
        String result;
        result = createJournal(journalDTO.getLabel(),journalDTO.getPrefix(),journalDTO.getDescription());
        return result;
    }


    public List<JournalDTO> getAllJournal ()
    {
        List<JournalDTO> listJournalDTO = new ArrayList<JournalDTO>();;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            JournalDAO journalDAO = CallaoDAOHelper.getJournalDAO(topiaContext);
            // Creation du journal
            List<Journal> listJournal = journalDAO.findAll();
            // Pour chaque journal, on le transforme en DTO
            for (Journal journal : listJournal)
            {
                // Converti en DTO
                JournalDTO journalDTO = convertJournal.journalEntityToDto(journal);
                // Ajoute à la liste
                listJournalDTO.add(journalDTO);
            }
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return listJournalDTO;
    }


    /**
     * Permet de rechercher un journal existant à partir de son label.
     * ATTENTION : il est possible que plusieurs journaux existent avec ce
     * même label. La méthode renvoie le premier trouvé.
     * @param prefix préfixe du journal recherché
     * @return
     */
    public Journal searchJournalWithLabel (String label)
    {
        Journal journalResult = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            JournalDAO journalDAO = CallaoDAOHelper.getJournalDAO(topiaContext);
            // Creation du journal
            journalResult = journalDAO.findByLabel(label);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return journalResult;
    }


    /**
     * Permet de rechercher un journal existant à partir du label.
     * Le journal recherché est converti automatiquement en format DTO.
     * @param prefix préfixe du journal recherché
     * @return
     */
    public JournalDTO searchJournalDTOWithLabel (String label)
    {
        Journal journal = searchJournalWithLabel(label);
        JournalDTO journalDTO = convertJournal.journalEntityToDto(journal);
        return journalDTO;
    }
    

    /**
     * Permet de rechercher un journal unique identifé par son topiaId.
     * @param topiaId identifiant du journal recherché
     * @return
     */
    public Journal searchJournalWithTopiaId (String topiaId)
    {
        Journal journalResult = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            JournalDAO journalDAO = CallaoDAOHelper.getJournalDAO(topiaContext);
            // Creation du journal
            journalResult = journalDAO.findByTopiaId(topiaId);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return journalResult;
    }


    /**
     * Permet de rechercher un journal existant à partir du topiaId.
     * Le journal recherché est converti automatiquement en format DTO.
     * @param prefix préfixe du journal recherché
     * @return
     */
    public JournalDTO searchJournalDTOWithTopiaId (String topiaId)
    {
        Journal journal = searchJournalWithTopiaId(topiaId);
        JournalDTO journalDTO = convertJournal.journalEntityToDto(journal);
        return journalDTO;
    }


    /**
     * Permet de modifier un journal.
     * @param label
     * @param prefix
     * @return
     */
    public String modifyJournal (String topiaId, String label, String prefix, String description)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        Journal journalModify = searchJournalWithTopiaId(topiaId);
        // Si le journal n'existe pas
        if (journalModify == null)
        {
            if (log.isWarnEnabled()) {
                log.warn("Le journal "+label+" n'existe pas !");
            }
            result = ServiceHelper.JOURNAL_NOT_EXIST;
        }else // Sinon on modifie le journal
        {           
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                JournalDAO journalDAO = CallaoDAOHelper.getJournalDAO(topiaContext);
                // Modifie le journal
                journalModify.setPrefix(prefix);
                journalModify.setDescription(description);
                journalDAO.update(journalModify);
                // Création BDD
                topiaContext.commitTransaction();
                // Fermeture BDD
                topiaContext.closeContext();
                result = ServiceHelper.RESPOND_SUCCESS;
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }


    /**
     * Permet d'effacer un journal dans la base de données.
     * ATTENTION : si un journal est associé avec des transactions, il est alors
     * impossible de supprimer celui-ci.
     * @param prefix préfixe du journal à supprimer
     * @return
     */
    public String removeJournal (String topiaId)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        Journal journalDelete = searchJournalWithTopiaId(topiaId);
        // Si le journal n'existe pas
        if (journalDelete == null)
        {
            if (log.isWarnEnabled()) {
                log.warn("Le journal "+topiaId+" n'existe pas !");
            }
            result = ServiceHelper.JOURNAL_NOT_EXIST;
        }else // Sinon on efface le journal
        {
            /**
             * Vérifie si une transaction n'appartient pas à ce journal.
             */
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
                // Recherche au moins une transaction avec ce journal.
                Transaction transaction = transactionDAO.findByJournal(journalDelete);
                // Il existe au moins une transaction
                if ( transaction != null )
                {
                    if (log.isWarnEnabled()) {
                        log.warn("Le journal "+topiaId+" possède des transactions comptables !");
                    }
                    result = ServiceHelper.JOURNAL_WITH_TRANSACTIONS;
                } else {                    
                    /**
                     * Efface le journal
                     */
                    // Chargement du DAO
                    JournalDAO journalDAO = CallaoDAOHelper.getJournalDAO(topiaContext);
                    // Supprime le journal
                    journalDAO.delete(journalDelete);
                    // Création BDD
                    topiaContext.commitTransaction();
                    // Fermeture BDD
                    topiaContext.closeContext();
                    result = ServiceHelper.RESPOND_SUCCESS;
                }
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }

    /**
     * Permet d'effacer un journal dans la base de données.
     * ATTENTION : si un journal est associé avec des transactions, il est alors
     * impossible de supprimer celui-ci.
     * @param journalDTO journal au format DTO à supprimer
     * @return
     */
    public String removeJournal (JournalDTO journalDTO)
    {
        String result = removeJournal(journalDTO.getId());
        return result;
    }

	
}
