/**
 * *##% Callao AccountServiceImpl
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.callao.entity.CallaoDAOHelper;
import org.chorem.callao.entity.Account;
import org.chorem.callao.entity.AccountDAO;
import org.chorem.callao.entity.Entry;
import org.chorem.callao.entity.EntryDAO;
import org.chorem.callao.service.convertObject.ConvertAccount;
import org.chorem.callao.service.dto.AccountDTO;
import org.chorem.callao.service.utils.ContextCallao;
import org.chorem.callao.service.utils.ServiceHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;


/**
 * Permet d'implémenter le Plan Comptable Général.
 * Un compte ne peut être supprimé si il contient des écritures comptables.
 * Un compte peut devenir père et avoir des comptes fils. Chaque compte créé doit
 * renseigné si il appartient à un compte père avec le numéro de compte père.
 *
 * @author Rémi Chapelet
 */
public class AccountServiceImpl  {

    private static final Log log = LogFactory.getLog(AccountServiceImpl.class);

	private TopiaContext rootContext = ContextCallao.getInstance().getContext();

    private ConvertAccount convertAccount = new ConvertAccount();
    

    /**
     * Permet de créer un nouveau compte dans le PCG de l'application.
     * Il ne peut exister deux numéros identiques. Pour chaque nouveau compte,
     * il est vérifié si il n'existe pas un compte avec le numéro donné. Dans ce
     * cas présent, le compte n'est pas créé.
     * ATTENTION : le compte masterAccount, c'est à dire le compte père, doit
     * exister, sinon le compte n'est pas créé. Si le compte n'a pas de père, alors
     * mettre masterAccount à null.
     * @param label label du compte
     * @param number numéro du compte
     * @param maserAccount compte père, si il en a pas, alors mettre à null.
     * @param type type du compte (actif,passif,etc.)
     * @return
     */
    public String createAccount (String accountNumber, String label,Account masterAccount,String type)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Détermine si le compte existe déjà ou non
        boolean existAccount = existAccount(accountNumber);
        // Si le numéro de compte existe
        if (existAccount)
        {
            if (log.isWarnEnabled()) {
                log.warn("Le compte numéro "+accountNumber+" existe deja !");
            }
            result = ServiceHelper.ACCOUNT_DOUBLE;
        } else {
            // Création du compte
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);
                /**
                 * Le compte a un père
                 */
                if ( masterAccount != null )
                {
                    // Vérification que le compte père existe dans la base de données
                    boolean existMasterAccount = existAccount(masterAccount.getAccountNumber());
                    // Si le compte père n'existe pas
                    if (!existMasterAccount)
                    {
                        if (log.isWarnEnabled()) {
                            log.warn("Le compte père numéro "+accountNumber+" n'existe pas !");
                        }
                        result = ServiceHelper.ACCOUNT_NOT_MASTER;
                    } else {
                        // Creation du compte
                        Account newAccount = accountDAO.create();
                        newAccount.setAccountNumber(accountNumber);
                        newAccount.setLabel(label);
                        newAccount.setMasterAccount(masterAccount);
                        newAccount.setType(type);
                        // Création BDD
                        topiaContext.commitTransaction();
                        result = ServiceHelper.RESPOND_SUCCESS;
                    }
                } else {
                /**
                 * Le compte n'a pas de père
                 */
                    // Creation du compte
                    Account newAccount = accountDAO.create();
                    newAccount.setAccountNumber(accountNumber);
                    newAccount.setLabel(label);
                    newAccount.setType(type);
                    // Création BDD
                    topiaContext.commitTransaction();
                    result = ServiceHelper.RESPOND_SUCCESS;
                }
                // Fermeture BDD
                topiaContext.closeContext();                
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }

    /**
     * Permet de créer un compte à partir du numéro de compte père. Il appelle
     * ensuite createAccount avec le compte père.
     * @param accountNumber numéro du compte à créer
     * @param label label pour le compte
     * @param maserAccountNumber numéro du compte père, si il en a pas, mettre 0.
     * @return
     */
    public String createAccount (String accountNumber, String label,String masterAccountNumber,String type)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Recherche le compte Master
        Account masterAccount = searchAccount(masterAccountNumber);
        // Si le compte père n'existe pas
        if ( masterAccount == null && !masterAccountNumber.equals("0") )
        {
            if (log.isWarnEnabled()) {
                log.warn("Le compte père numéro "+masterAccountNumber+" n'existe pas !");
            }
            result = ServiceHelper.ACCOUNT_NOT_MASTER;
        }else{
            // Création du compte
            result = createAccount(accountNumber,label,masterAccount,type);
        }
        return result;
    }

    /**
     * Permet de creer un compte à partir d'un objet DTO
     * @param accountDTO compte au format DTO
     * @return
     */
    public String createAccount (AccountDTO accountDTO)
    {
        String result;
        String result_final;
        boolean error = false; // Si il y a eu une erreur pour créer les enfants
        /**
         * Création du compte père
         */
        result = createAccount(accountDTO.getAccountNumber(),accountDTO.getLabel(),accountDTO.getMasterAccount(),accountDTO.getType());
        /**
         * Création des comptes enfants
         */
        // Pour chaque enfant
        for (AccountDTO accountChildDTO : accountDTO.getAccountChildDTO())
        {
            // Création de l'enfant
            result = createAccount(accountChildDTO);
            // Si il y a eu une erreur
            if ( !result.equals(ServiceHelper.RESPOND_SUCCESS))
            {
                if (log.isWarnEnabled()) {
                    log.warn("Le compte numéro "+accountChildDTO.getLabel()+" n'a pu être créé !");
                }
                error = true;
                result_final = result;
            }
        }
        // Si il y a eu une erreur lors de la création d'un enfant, il est envoyé une erreur.
        if (error)
        {
            result = ServiceHelper.RESPOND_ERROR;
        }
        return result;
    }


    /**
     * Recherche un compte DTO. Il est recherché dans la base de données et est
     * converti avec ConvertAccount. ConvertAccount recherche dans la liste de
     * ses enfants pour les convertir également, et ainsi de suite.
     * @param accountNumber numéro du compte qu'on souhait rechercher
     * @return
     */
    public AccountDTO searchAccountDTO (String accountNumber)
    {
        AccountDTO accountDTO = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);
            // Recherche du compte
            Account account = accountDAO.findByAccountNumber(accountNumber);         
            // Converti entity en DTO
            convertAccount.setTransaction(topiaContext);
            accountDTO = convertAccount.accountEntityToDto(account, null);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }        
        return accountDTO;
    }


    /**
     * Permet de rechercher un compte à partir d'un numéro de compte.
     * @param accountNumber numéro du compte à rechercher
     * @return
     */
    public Account searchAccount (String accountNumber)
    {
        Account accountResult = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);
            // Recherche du compte
            accountResult = accountDAO.findByAccountNumber(accountNumber);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return accountResult;
    }

    /**
     * Permet de renvoyer tous les enfants du compte.
     * Cette fonction permet de renvoyer les enfants, mais PAS les enfants des
     * enfants (et ainsi de suite). Cette méthode descend donc uniquement d'un
     * niveau sur la hiérarchie.
     * @param account compte dont on souhaite obtenir ses enfants.
     * @return
     */
    public List<Account> searchListChildAccount (Account account)
    {
        List<Account> ListAccount = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);
            // Recherche des comptes enfants
            ListAccount = accountDAO.findAllByMasterAccount(account);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return ListAccount;
    }

    
    /**
     * Permet de renvoyer tous les enfants du compte. Elle se base sur la fonction
     * searchListChildAccount(Account account).
     * Cette fonction permet de renvoyer les enfants, mais PAS les enfants des
     * enfants (et ainsi de suite). Cette méthode descend donc uniquement d'un
     * niveau sur la hiérarchie.
     * @param account numéro de compte dont on souhaite obtenir ses enfants.
     * @return
     */
    public List<Account> searchListChildAccount (String accountNumber)
    {
        List<Account> ListAccount = null;
        // Recherche le compte
        Account account = searchAccount(accountNumber);
        // Si le compte n'existe pas
        if ( account == null )
        {
            if (log.isWarnEnabled()) {
                log.warn("Le compte numéro "+accountNumber+" n'existe pas !");
            }
        }else{
            // Recherche des comptes enfants
            ListAccount = searchListChildAccount(account);
        }
        return ListAccount;
    }


    /**
     * Permet de convertir tous les comptes en DTO. Il recherche dans un premier
     * temps tous les comptes "master", et les converti en DTO avec leur descendance.
     * @return
     */
    public List<AccountDTO> getAllAccount ()
    {
        ArrayList<AccountDTO> listAccountDTO = new ArrayList<AccountDTO>();
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);
            // Recherche des comptes enfants
            List<Account> listAccount = accountDAO.findAll();
            // Pour chaque compte
            // Converti entity en DTO
            convertAccount.setTransaction(topiaContext);
            for (Account account : listAccount)
            {
                // Pour les comptes sans père
                if (account.getMasterAccount() == null)
                {
                    AccountDTO accountDTO = convertAccount.accountEntityToDto(account, null);
                    listAccountDTO.add(accountDTO);
                }
            }
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return listAccountDTO;
    }


    /**
     * Renvoie vrai si le compte avec son numéro existe déjà dans la base de
     * données.
     * @param accountNumber numéro du compte recherché
     * @return
     */
    public boolean existAccount (String accountNumber)
    {
        // Recherche du compte
        Account accountSearch =  searchAccount(accountNumber);
        boolean result = false;
        // Si le compte est trouvé
        if ( accountSearch != null )
        {
            result = true;
        }
        return result;
    }

    /**
     * Permet d'effacer un compte dans la base de données. Il est vérifié dans
     * un premier temps si le compte existe bien.
     * ATTENTION : si il existe une entrée comptable associée au numéro de
     * compte, il est alors impossible de supprimer le compte.
     * @param accountNumber
     * @return
     */
    public String removeAccount (String accountNumber)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        Account deleteAccount = searchAccount(accountNumber);
        // Si le compte n'existe pas
        if (deleteAccount == null)
        {
            if (log.isWarnEnabled()) {
                log.warn("Le compte numéro "+accountNumber+" n'existe pas !");
            }
            result = ServiceHelper.ACCOUNT_NOT_EXIST;
        }else // Sinon on efface le compte
        {
            /**
             * Vérifie si une entrée ne possède pas ce numéro de compte.
             */
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
                // Recherche au moins une entry avec ce compte.
                Entry entry = entryDAO.findByAccount(deleteAccount);
                // Il existe au moins une entrée
                if ( entry != null )
                {
                    if (log.isWarnEnabled()) {
                        log.warn("Le compte numéro "+accountNumber+" possède des entrées comptable !");
                    }
                    result = ServiceHelper.ACCOUNT_WITH_ENTRIES;
                } else {
                    // On efface tous les comptes enfants
                    List<Account> listAccountChild =  searchListChildAccount(deleteAccount);
                    boolean ErrorRemoveChild = false;
                    for (Account accountChild : listAccountChild)
                    {
                        // Si on souhaite supprimer les comptes enfants
                        result = removeAccount(accountChild.getAccountNumber());
                        if ( result.equals(ServiceHelper.RESPOND_ERROR) )
                        {
                            ErrorRemoveChild = true;
                            if (log.isErrorEnabled()) {
                                log.error("Le compte fils numéro "
                                        +accountChild.getAccountNumber()
                                        +" n'a pas été supprimé !");
                            }
                        }
                        /**
                         * Si on ne souhaite pas supprimer les comptes enfants, tous les
                         * comptes enfants se retrouvent sans père. Commenter removeAccountChildTest
                         * dans le fichier AccountServiceImplTest dans ce cas.
                         */
                        //modifyAccount(accountChild.getAccountNumber(),accountChild.getLabel(),null);
                    }
                    // Si il n'a jamais eu d'erreur pour supprimer ses fils
                    if ( !ErrorRemoveChild )
                    {
                        /**
                         * Efface le compte père
                         */
                        // Chargement du DAO
                        AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);
                        // Supprime le compte
                        accountDAO.delete(deleteAccount);
                        // Création BDD
                        topiaContext.commitTransaction();
                        // Fermeture BDD
                        topiaContext.closeContext();
                        result = ServiceHelper.RESPOND_SUCCESS;
                        if (log.isInfoEnabled()) {
                            log.info("Le compte numéro "+accountNumber+" a ete supprimé avec succès.");
                        }
                    }
                }
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }


    /**
     * Permet d'effacer un compte à partir d'un compte DTO. Il appelle la
     * méthode removeAccount, qui effacera également les enfants (et ainsi de
     * suite dans la hiérarchie du compte).
     * @param accountDTO Compte au format DTO qu'on souhaite supprimer.
     * @return
     */
    public String removeAccount (AccountDTO accountDTO)
    {
        String result;
        result = removeAccount(accountDTO.getAccountNumber());
        return result;
    }

    
    /**
     * Permet de modifier un compte sur son label et son compte père.
     * Il n'est pas possible de modifier un numéro de compte.
     * Si le compte n'existe pas, il envoie alors un message d'avertissement.
     * @param accountNumber numéro de compte à modifier
     * @param label label à modifier
     * @param masterAccountNumber compte père à modifier
     * @return
     */
    public String modifyAccount (String accountNumber, String label,String type,Account masterAccount)
    {
        String result = ServiceHelper.RESPOND_ERROR;   
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);            
            // Recherche du compte
            Account modifyAccount = accountDAO.findByAccountNumber(accountNumber);
            // Si le compte n'existe pas
            if (modifyAccount == null)
            {
                if (log.isWarnEnabled()) {
                    log.warn("Le compte numéro "+accountNumber+" n'existe pas !");
                }
                result = ServiceHelper.ACCOUNT_NOT_EXIST;
            } else {
                /**
                 * Modifie le compte
                 */
                modifyAccount.setLabel(label);
                modifyAccount.setType(type);
                // Recherche du compte père
                // Si il possède un compte père
                if (masterAccount != null)
                {
                    modifyAccount.setMasterAccount(masterAccount);
                }
                modifyAccount.update();
                // Création BDD
                topiaContext.commitTransaction();
                // Fermeture BDD
                topiaContext.closeContext();
                result = ServiceHelper.RESPOND_SUCCESS;
            }
        }catch (TopiaException e) {
            log.error(e);
        }      
        return result;
    }

    /**
     * Permet de modifier un compte sur son label et son compte père.
     * Il recherche le compte père avec le numéro de compte fourni. Appel ensuite
     * la méthode modifyAccount avec pour paramètre le compte père trouvé.
     * @param accountNumber numéro de compte à modifier
     * @param label label à modifier
     * @param masterAccountNumber le numéro du compte père
     * @return
     */
    public String modifyAccount (String accountNumber, String label,String type, String masterAccountNumber)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);
            /**
             * Modifie le compte
             */
            // Recherche du compte père
            Account masterAccount = accountDAO.findByAccountNumber(masterAccountNumber);
            result = modifyAccount(accountNumber,label,type,masterAccount);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return result;
    }


    /**
     * Permet de modifier un compte à partir d'un compte DTO.
     * @param accountDTO compte au format DTO qu'on souhaire modifier.
     * @return
     */
    public String modifyAccount (AccountDTO accountDTO)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            AccountDAO accountDAO = CallaoDAOHelper.getAccountDAO(topiaContext);   
            // Recherche du compte père
            Account masterAccount = accountDAO.findByAccountNumber(accountDTO.getMasterAccount());
            result = modifyAccount(accountDTO.getAccountNumber(),accountDTO.getLabel(),accountDTO.getType(),masterAccount);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return result;
    }


    /**
     * Permet de mettre à jour le compte DTO ainsi que tous ses fils.
     * Cette méthode permet d'ajouter, modifier, et supprimer un compte et
     * ses enfants (et ainsi de suite).
     * @param accountDTO
     * @return
     */
    public String updateDTO (AccountDTO accountDTO)
    {
        String result;
        // Si le compte DTO père existe
        if (existAccount(accountDTO.getAccountNumber()))
        {
            // Modification du compte père
            result = modifyAccount(accountDTO);
        } else {
            // Création du compte père
            result = createAccount (accountDTO);
        }
        // Pour chaque enfant on lance l'update
        for (AccountDTO accountChildDTO : accountDTO.getAccountChildDTO())
        {
            updateDTO(accountChildDTO);
        }
        /**
         * On recherche les comptes DTO effacés
         */
        List<Account> listAccountChild = searchListChildAccount(accountDTO.getAccountNumber());
        for (Account account : listAccountChild)
        {
            // Si le compte n'est pas dans la liste, alors on efface
            if (!accountDTO.existAccountChild(account.getAccountNumber()))
            {
                result = removeAccount(account.getAccountNumber());
            }
        }       
        return result;
    }
	
}