/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapred;

import java.io.IOException;
import java.util.Set;
import java.util.TreeSet;
import javax.security.auth.login.LoginException;
import junit.framework.TestCase;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.examples.SleepJob;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.mapred.JobClient;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.JobTracker;
import org.apache.hadoop.mapred.MiniMRCluster;
import org.apache.hadoop.mapred.QueueManager;
import org.apache.hadoop.mapred.RunningJob;
import org.apache.hadoop.mapred.UtilsForTests;
import org.apache.hadoop.security.UnixUserGroupInformation;

public class TestQueueManager
extends TestCase {
    private static final Log LOG = LogFactory.getLog(TestQueueManager.class);
    private MiniDFSCluster miniDFSCluster;
    private MiniMRCluster miniMRCluster;

    public void testDefaultQueueConfiguration() {
        JobConf conf = new JobConf();
        QueueManager qMgr = new QueueManager((Configuration)conf);
        TreeSet<String> expQueues = new TreeSet<String>();
        expQueues.add("default");
        this.verifyQueues(expQueues, qMgr.getQueues());
        TestQueueManager.assertFalse((boolean)conf.getBoolean("mapred.acls.enabled", true));
    }

    public void testMultipleQueues() {
        JobConf conf = new JobConf();
        conf.set("mapred.queue.names", "q1,q2,Q3");
        QueueManager qMgr = new QueueManager((Configuration)conf);
        TreeSet<String> expQueues = new TreeSet<String>();
        expQueues.add("q1");
        expQueues.add("q2");
        expQueues.add("Q3");
        this.verifyQueues(expQueues, qMgr.getQueues());
    }

    public void testSchedulerInfo() {
        JobConf conf = new JobConf();
        conf.set("mapred.queue.names", "qq1,qq2");
        QueueManager qMgr = new QueueManager((Configuration)conf);
        qMgr.setSchedulerInfo("qq1", (Object)"queueInfoForqq1");
        qMgr.setSchedulerInfo("qq2", (Object)"queueInfoForqq2");
        TestQueueManager.assertEquals((Object)qMgr.getSchedulerInfo("qq2"), (Object)"queueInfoForqq2");
        TestQueueManager.assertEquals((Object)qMgr.getSchedulerInfo("qq1"), (Object)"queueInfoForqq1");
    }

    public void testAllEnabledACLForJobSubmission() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-submit-job", "*");
        this.verifyJobSubmission(conf, true);
    }

    public void testAllDisabledACLForJobSubmission() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-submit-job", "");
        this.verifyJobSubmission(conf, false);
    }

    public void testUserDisabledACLForJobSubmission() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-submit-job", "3698-non-existent-user");
        this.verifyJobSubmission(conf, false);
    }

    public void testDisabledACLForNonDefaultQueue() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-submit-job", "*");
        conf.set("mapred.queue.names", "default,q1");
        conf.set("mapred.queue.q1.acl-submit-job", "dummy-user");
        this.verifyJobSubmission(conf, false, "q1");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testSubmissionToInvalidQueue() throws IOException {
        JobConf conf = new JobConf();
        conf.set("mapred.queue.names", "default");
        this.setUpCluster(conf);
        String queueName = "q1";
        try {
            RunningJob rjob = this.submitSleepJob(1, 1, 100L, 100L, true, null, queueName);
        }
        catch (IOException ioe) {
            TestQueueManager.assertTrue((boolean)ioe.getMessage().contains("Queue \"" + queueName + "\" does not exist"));
            return;
        }
        finally {
            this.tearDownCluster();
        }
        TestQueueManager.fail((String)"Job submission to invalid queue job shouldnot complete , it should fail with proper exception ");
    }

    public void testEnabledACLForNonDefaultQueue() throws IOException, LoginException {
        UnixUserGroupInformation ugi = UnixUserGroupInformation.login();
        String userName = ugi.getUserName();
        JobConf conf = this.setupConf("mapred.queue.default.acl-submit-job", "*");
        conf.set("mapred.queue.names", "default,q2");
        conf.set("mapred.queue.q2.acl-submit-job", userName);
        this.verifyJobSubmission(conf, true, "q2");
    }

    public void testUserEnabledACLForJobSubmission() throws IOException, LoginException {
        UnixUserGroupInformation ugi = UnixUserGroupInformation.login();
        String userName = ugi.getUserName();
        JobConf conf = this.setupConf("mapred.queue.default.acl-submit-job", "3698-junk-user," + userName + " 3698-junk-group1,3698-junk-group2");
        this.verifyJobSubmission(conf, true);
    }

    public void testGroupsEnabledACLForJobSubmission() throws IOException, LoginException {
        UnixUserGroupInformation ugi = UnixUserGroupInformation.login();
        String[] groups = ugi.getGroupNames();
        TestQueueManager.assertTrue((groups.length > 0 ? 1 : 0) != 0);
        JobConf conf = this.setupConf("mapred.queue.default.acl-submit-job", "3698-junk-user1,3698-junk-user2 " + groups[groups.length - 1] + ",3698-junk-group");
        this.verifyJobSubmission(conf, true);
    }

    public void testAllEnabledACLForJobKill() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-administer-jobs", "*");
        this.verifyJobKill(conf, true);
    }

    public void testAllDisabledACLForJobKill() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-administer-jobs", "");
        this.verifyJobKillAsOtherUser(conf, false, "dummy-user,dummy-user-group");
    }

    public void testOwnerAllowedForJobKill() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-administer-jobs", "junk-user");
        this.verifyJobKill(conf, true);
    }

    public void testUserDisabledACLForJobKill() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-administer-jobs", "dummy-user");
        this.verifyJobKillAsOtherUser(conf, false, "dummy-user,dummy-user-group");
    }

    public void testUserEnabledACLForJobKill() throws IOException, LoginException {
        UnixUserGroupInformation ugi = UnixUserGroupInformation.login();
        String userName = ugi.getUserName();
        JobConf conf = this.setupConf("mapred.queue.default.acl-administer-jobs", "dummy-user," + userName);
        this.verifyJobKillAsOtherUser(conf, true, "dummy-user,dummy-user-group");
    }

    public void testUserDisabledForJobPriorityChange() throws IOException {
        JobConf conf = this.setupConf("mapred.queue.default.acl-administer-jobs", "junk-user");
        this.verifyJobPriorityChangeAsOtherUser(conf, false, "junk-user,junk-user-group");
    }

    private JobConf setupConf(String aclName, String aclValue) {
        JobConf conf = new JobConf();
        conf.setBoolean("mapred.acls.enabled", true);
        conf.set(aclName, aclValue);
        return conf;
    }

    private void verifyQueues(Set<String> expectedQueues, Set<String> actualQueues) {
        TestQueueManager.assertEquals((int)expectedQueues.size(), (int)actualQueues.size());
        for (String queue : expectedQueues) {
            TestQueueManager.assertTrue((boolean)actualQueues.contains(queue));
        }
    }

    private void verifyJobSubmission(JobConf conf, boolean shouldSucceed) throws IOException {
        this.verifyJobSubmission(conf, shouldSucceed, "default");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyJobSubmission(JobConf conf, boolean shouldSucceed, String queue) throws IOException {
        this.setUpCluster(conf);
        try {
            RunningJob rjob = this.submitSleepJob(1, 1, 100L, 100L, true, null, queue);
            if (shouldSucceed) {
                TestQueueManager.assertTrue((boolean)rjob.isSuccessful());
            } else {
                TestQueueManager.fail((String)"Job submission should have failed.");
            }
        }
        catch (IOException ioe) {
            if (shouldSucceed) {
                throw ioe;
            }
            LOG.info((Object)("exception while submitting job: " + ioe.getMessage()));
            TestQueueManager.assertTrue((boolean)ioe.getMessage().contains("cannot perform operation SUBMIT_JOB on queue " + queue));
            JobTracker jobtracker = this.miniMRCluster.getJobTrackerRunner().getJobTracker();
            Path sysDir = new Path(jobtracker.getSystemDir());
            FileSystem fs = sysDir.getFileSystem((Configuration)conf);
            int size = fs.listStatus(sysDir).length;
            while (size > 1) {
                System.out.println("Waiting for the job files in sys directory to be cleaned up");
                UtilsForTests.waitFor(100L);
                size = fs.listStatus(sysDir).length;
            }
        }
        finally {
            this.tearDownCluster();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyJobKill(JobConf conf, boolean shouldSucceed) throws IOException {
        this.setUpCluster(conf);
        try {
            RunningJob rjob = this.submitSleepJob(1, 1, 1000L, 1000L, false);
            TestQueueManager.assertFalse((boolean)rjob.isComplete());
            while (rjob.mapProgress() == 0.0f) {
                try {
                    Thread.sleep(10L);
                }
                catch (InterruptedException ie) {
                    // empty catch block
                    break;
                }
            }
            rjob.killJob();
            while (rjob.cleanupProgress() == 0.0f) {
                try {
                    Thread.sleep(10L);
                }
                catch (InterruptedException ie) {
                    // empty catch block
                    break;
                }
            }
            if (shouldSucceed) {
                TestQueueManager.assertTrue((boolean)rjob.isComplete());
            } else {
                TestQueueManager.fail((String)"Job kill should have failed.");
            }
        }
        catch (IOException ioe) {
            if (shouldSucceed) {
                throw ioe;
            }
            LOG.info((Object)("exception while submitting job: " + ioe.getMessage()));
            TestQueueManager.assertTrue((boolean)ioe.getMessage().contains("cannot perform operation ADMINISTER_JOBS on queue default"));
        }
        finally {
            this.tearDownCluster();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyJobKillAsOtherUser(JobConf conf, boolean shouldSucceed, String otherUserInfo) throws IOException {
        this.setUpCluster(conf);
        try {
            String userInfo = otherUserInfo;
            RunningJob rjob = this.submitSleepJob(1, 1, 1000L, 1000L, false, userInfo);
            TestQueueManager.assertFalse((boolean)rjob.isComplete());
            try {
                rjob.killJob();
                if (!shouldSucceed) {
                    TestQueueManager.fail((String)"should fail kill operation");
                }
            }
            catch (IOException ioe) {
                if (shouldSucceed) {
                    throw ioe;
                }
                LOG.info((Object)("exception while submitting job: " + ioe.getMessage()));
                TestQueueManager.assertTrue((boolean)ioe.getMessage().contains("cannot perform operation ADMINISTER_JOBS on queue default"));
            }
            while (!rjob.isComplete()) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException ie) {
                    // empty catch block
                    break;
                }
            }
        }
        finally {
            this.tearDownCluster();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyJobPriorityChangeAsOtherUser(JobConf conf, boolean shouldSucceed, String otherUserInfo) throws IOException {
        this.setUpCluster(conf);
        try {
            String userInfo = otherUserInfo;
            RunningJob rjob = this.submitSleepJob(1, 1, 1000L, 1000L, false, userInfo);
            TestQueueManager.assertFalse((boolean)rjob.isComplete());
            try {
                rjob.setJobPriority("VERY_LOW");
                if (!shouldSucceed) {
                    TestQueueManager.fail((String)"changing priority should fail.");
                }
            }
            catch (IOException ioe) {
                LOG.info((Object)("exception while submitting job: " + ioe.getMessage()));
                TestQueueManager.assertTrue((boolean)ioe.getMessage().contains("cannot perform operation ADMINISTER_JOBS on queue default"));
            }
            while (!rjob.isComplete()) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException ie) {
                    // empty catch block
                    break;
                }
            }
        }
        finally {
            this.tearDownCluster();
        }
    }

    private void setUpCluster(JobConf conf) throws IOException {
        this.miniDFSCluster = new MiniDFSCluster((Configuration)conf, 1, true, null);
        FileSystem fileSys = this.miniDFSCluster.getFileSystem();
        String namenode = fileSys.getUri().toString();
        this.miniMRCluster = new MiniMRCluster(1, namenode, 3, null, null, conf);
    }

    private void tearDownCluster() throws IOException {
        if (this.miniMRCluster != null) {
            this.miniMRCluster.shutdown();
        }
        if (this.miniDFSCluster != null) {
            this.miniDFSCluster.shutdown();
        }
    }

    private RunningJob submitSleepJob(int numMappers, int numReducers, long mapSleepTime, long reduceSleepTime, boolean shouldComplete) throws IOException {
        return this.submitSleepJob(numMappers, numReducers, mapSleepTime, reduceSleepTime, shouldComplete, null);
    }

    private RunningJob submitSleepJob(int numMappers, int numReducers, long mapSleepTime, long reduceSleepTime, boolean shouldComplete, String userInfo) throws IOException {
        return this.submitSleepJob(numMappers, numReducers, mapSleepTime, reduceSleepTime, shouldComplete, userInfo, null);
    }

    private RunningJob submitSleepJob(int numMappers, int numReducers, long mapSleepTime, long reduceSleepTime, boolean shouldComplete, String userInfo, String queueName) throws IOException {
        JobConf clientConf = new JobConf();
        clientConf.set("mapred.job.tracker", "localhost:" + this.miniMRCluster.getJobTrackerPort());
        SleepJob job = new SleepJob();
        job.setConf((Configuration)clientConf);
        clientConf = job.setupJobConf(numMappers, numReducers, mapSleepTime, (int)mapSleepTime / 100, reduceSleepTime, (int)reduceSleepTime / 100);
        if (queueName != null) {
            clientConf.setQueueName(queueName);
        }
        RunningJob rJob = null;
        if (shouldComplete) {
            rJob = JobClient.runJob((JobConf)clientConf);
        } else {
            JobConf jc = new JobConf((Configuration)clientConf);
            if (userInfo != null) {
                jc.set("hadoop.job.ugi", userInfo);
            }
            rJob = new JobClient(clientConf).submitJob(jc);
        }
        return rJob;
    }
}

