/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2011 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.ObservationType;
import fr.ifremer.wao.entity.Boat;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.io.ImportResults;
import fr.ifremer.wao.service.ServiceContact;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.components.SamplingFilterComponent;
import fr.ifremer.wao.ui.data.ErrorReport;
import fr.ifremer.wao.ui.data.ExportStreamResponse;
import fr.ifremer.wao.ui.data.ImportEngine;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.data.SampleRowDataSource;
import fr.ifremer.wao.ui.data.WaoActivationContext;
import fr.ifremer.wao.ui.services.SampleRowModelFactory;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.BeanModelSource;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Set;

@RequiresAuthentication(allowedPrograms = {ObsProgram.OBSDEB})
public class ObsDebSamplingPlan {

    //// Tapestry stuffs

    @Inject
    private Logger log;

    @InjectComponent
    private Layout layout;

    @Inject
    private BeanModelSource beanModelSource;

    @Inject
    private ComponentResources resources;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceContact serviceContact;

    @Inject
    private Messages messages;

    @Inject
    private WaoManager manager;

    private DateFormat dateFormat;

    public DateFormat getDateFormat() {
        if (dateFormat == null) {
            dateFormat = new SimpleDateFormat("dd/MM/YYYY");
        }
        return dateFormat;
    }

    //// session

    @SessionState
    @Property
    private ConnectedUser connectedUser;

    private WaoActivationContext waoActivationContext;

    void onActivate(EventContext ec) {
        waoActivationContext = WaoActivationContext.newContextFromTapestry(ec);
    }

    //// actions of the sub-menu

    @Persist
    private Boolean showImport;

    @InjectComponent
    private Zone importZone;

    @InjectComponent
    private SamplingFilterComponent samplingFilter;

    public ServiceSampling getServiceSampling() {
        return serviceSampling;
    }

    @Log
    public boolean getShowImport() {
        if (showImport == null) {
            showImport = false;
        }
        return showImport;
    }

    @Log
    public Object onActionFromShowFilters() {
        samplingFilter.switchFiltersVisible();
        return samplingFilter;
    }

    public Block onSuccessFromImportForm() {
        showImport = false;
        return importZone.getBody();
    }

    public ImportEngine getSamplingPlanImportEngine() {
        return new ImportEngine() {
            @Override
            public ImportResults execute(InputStream input) throws WaoBusinessException {
                ImportResults results = serviceSampling.importSamplingPlanCsv(input, connectedUser);
                return results;
            }
        };
    }

    StreamResponse onActionFromExportSamplingPlan() {
        return new ExportStreamResponse("wao-echantillonnage-obsdeb") {
            @Override
            public InputStream getStream() throws IOException {
                InputStream csv;
                try {
                    csv = serviceSampling.exportSamplingPlanCsv(connectedUser, samplingFilter.getFilter());
                } catch (WaoException eee) {
                    throw new IOException(eee);
                }
                return csv;
            }
        };
    }

    @Log
    void onImported(ErrorReport report) {
        for (String info : report.getInfos()) {
            layout.addInfo(info);
        }
        for (String error : report.getErrors()) {
            layout.addError(error);
        }
    }

    // FIXME 20110502 bleny clicking on 'export' make the 'import' link still
    // active but content is not shown/hidden

    @Log
    public Object onActionFromShowImport() {
        showImport = ! getShowImport();
        return importZone.getBody();
    }

    protected void setObservationType(ObservationType observationType) {
        samplingFilter.resetFilter();
        samplingFilter.getFilter().setExpectedObservationType(observationType);
    }

    protected ObservationType getObservationType() {
        ObservationType observationType = samplingFilter.getFilter().getExpectedObservationType();
        if (observationType == null) {
            if (waoActivationContext.getObservationType() == null) {
                observationType = ObservationType.FIELD_WORK_OBSERVATION;
            } else {
                observationType = waoActivationContext.getObservationType();
            }
            setObservationType(observationType);
        }
        return observationType;
    }

    public String[] getContextForNewSampleRow() {
        WaoActivationContext contextForNewSampleRow =
                                WaoActivationContext.newEmptyContext();
        contextForNewSampleRow.setObservationType(getObservationType());
        return contextForNewSampleRow.toStrings();
    }

    public boolean showFieldWorkSamplingPlan() {
        return getObservationType() == ObservationType.FIELD_WORK_OBSERVATION;
    }

    public void onActionFromSwitchObservationType() {
        if (showFieldWorkSamplingPlan()) {
            setObservationType(ObservationType.PHONE_CALL);
        } else {
            setObservationType(ObservationType.FIELD_WORK_OBSERVATION);
        }
    }

    //// grid data

    private SampleRowDataSource sampleRows;

    @Property
    private SampleRow sampleRow;

    public SampleRowDataSource getSampleRows() {
        if (sampleRows == null) {
            getObservationType();
            samplingFilter.getFilter().setOrderBy(SampleRow.PROPERTY_EXPECTED_DATE);
            sampleRows = new SampleRowDataSource(samplingFilter.getFilter(), serviceSampling);
        }
        return sampleRows;
    }

    private BeanModel<SampleRow> obsdebSamplingPlanModel;

    public BeanModel<SampleRow> getObsdebSamplingPlanModel() {
        if (obsdebSamplingPlanModel == null) {
            SampleRowModelFactory sampleRowModelFactory =
                    new SampleRowModelFactory(beanModelSource, resources, connectedUser,
                                              messages, serviceContact);
            obsdebSamplingPlanModel = sampleRowModelFactory.newModel(getObservationType());
        }
        return obsdebSamplingPlanModel;
    }

    public String getBoatDescription() {
        return manager.getTooltipBoat(sampleRow.getBoat());
    }

    @Property
    private Boat substitute;

    public String getSubstituteDescription() {
        return manager.getTooltipBoat(substitute);
    }

    public Set<Boat> getSubstitutes() {
        return serviceContact.getSubstitutesForBoat(sampleRow.getBoat());
    }

    public boolean isBoatSampled() {
        return serviceContact.isBoatSampled(sampleRow.getBoat(), sampleRow);
    }

    protected String[] getContextForBoatOrSubstitute(Boat boat) {
        WaoActivationContext contextForBoat =
                WaoActivationContext.newEmptyContext();
        contextForBoat.setSampleRowCode(sampleRow.getCode());
        contextForBoat.setBoatImmatriculation(boat.getImmatriculation());
        return contextForBoat.toStrings();
    }

    /** Context for boats page for links on the name of a boat */
    public String[] getContextForBoat() {
        return getContextForBoatOrSubstitute(sampleRow.getBoat());
    }

    /** Context for boats page for links to boats page in 'actions' */
    public String[] getContextForBoats() {
        WaoActivationContext contextForBoats =
                                    WaoActivationContext.newEmptyContext();
        contextForBoats.setSampleRowCode(sampleRow.getCode());
        return contextForBoats.toStrings();
    }

    /** Context for boats page for links on the name of a substitute boat */
    public String[] getContextForSubstitute() {
        return getContextForBoatOrSubstitute(substitute);
    }

    //// grid actions

    public boolean canEditSampleRow() {
        boolean canEditSampleRow = connectedUser.isAdmin() || connectedUser.isCoordinator();
        return canEditSampleRow;
    }

    @Log
    public void onActionFromDeleteSampleRow(String sampleRowId) throws WaoBusinessException {
        SampleRow sampleRow = serviceSampling.getSampleRow(sampleRowId);
        if (sampleRow == null) {
            throw new IllegalStateException();
        }
        serviceSampling.deleteSampleRow(sampleRow);
    }

    public String[] getContextForEditingSampleRow() {
        WaoActivationContext waoActivationContext = WaoActivationContext.newEmptyContext();
        waoActivationContext.setSampleRowId(sampleRow.getTopiaId());
        waoActivationContext.setObservationType(sampleRow.getObservationType());
        return waoActivationContext.toStrings();
    }

}
