/*
 * *##% 
 * Wao :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
// Copyright 2007 The Apache Software Foundation
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package fr.ifremer.wao.ui.components;

import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.annotations.Environmental;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.corelib.base.AbstractField;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.FormSupport;
import org.apache.tapestry5.services.Heartbeat;
import org.apache.tapestry5.services.Request;

/**
  * Corresponds to &lt;input type="submit"&gt;, a client-side element that can force the
  * enclosing form to submit. The submit responsible for the form submission will post a
  * notification that allows the application to know that it was the responsible entity. The
  * notification is named "selected" and has a String context.
  */
public final class SubmitContext extends AbstractField
{
     static final String SELECTED_EVENT = "selected";

     /**
      * If true, then any notification sent by the component will be deferred until the end of
      * the form submission (this is usually desirable).
      */
     @Parameter
     private boolean defer = true;

     @Parameter
     private String context;

     @Environmental
     private FormSupport formSupport;

     @Environmental
     private Heartbeat heartbeat;

     @Inject
     private ComponentResources resources;

     @Inject
     private Request request;

     public SubmitContext() {
     }

     SubmitContext(Request request) {
         this.request = request;
     }

     void beginRender(MarkupWriter writer) {
         // write a hidden input for the context
         //String elementName = getElementName();
         String elementName = getControlName(); //Modified to work with 5.0.12
         writer.element("input", "type", "hidden", "name", elementName + "X", "value", context);
         writer.end();

         // now the submit
         writer.element("input", "type", "submit", "name", elementName, "id", getClientId());
         resources.renderInformalParameters(writer);
     }

     void afterRender(MarkupWriter writer) {
         writer.end();
     }

     @Override
     protected void processSubmission(String elementName) {
         String value = request.getParameter(elementName);
         final String cxt = request.getParameter(elementName + "X");

         if (value == null) {
             return;
         }

         Runnable sendNotification = new Runnable() {
            @Override
             public void run() {
                 resources.triggerEvent(SELECTED_EVENT, new Object[] {cxt}, null);
             }
         };

         // When not deferred, don't wait, fire the event now (actually, at the end of the current
         // heartbeat). This is most likely because the Submit is inside a Loop and some contextual
         // information will change if we defer. Another option might be to wait until the next
         // heartbeak?

         if (defer) {
             formSupport.defer(sendNotification);
         } else {
             heartbeat.defer(sendNotification);
         }
     }

     // For testing:

     void setDefer(boolean defer) {
         this.defer = defer;
     }

     void setup(ComponentResources resources, FormSupport support, Heartbeat heartbeat) {
         this.resources = resources;
         formSupport = support;
         this.heartbeat = heartbeat;
     }
}
