package fr.ifremer.tutti.ui.swing.content.cruise;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditCruiseUIHandler.java 138 2013-01-02 10:07:34Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/cruise/EditCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import jaxx.runtime.validator.swing.SwingValidatorMessageTableRenderer;
import jaxx.runtime.validator.swing.SwingValidatorUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.List;

/**
 * Handler of UI {@link EditCruiseUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EditCruiseUIHandler extends AbstractTuttiUIHandler<EditCruiseUIModel> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(EditCruiseUIHandler.class);

    /**
     * UI.
     *
     * @since 0.1
     */
    private final EditCruiseUI ui;

    /**
     * Persistence service.
     *
     * @since 0.1
     */
    private final PersistenceService persistenceService;

    public EditCruiseUIHandler(TuttiUIContext context, EditCruiseUI ui) {
        super(context);
        this.ui = ui;
        persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    public void beforeInitUI() {

        EditCruiseUIModel model = new EditCruiseUIModel();

        String cruiseId = context.getCruiseId();
        if (cruiseId == null) {

            // create new cruise
            Program program =
                    persistenceService.getProgram(context.getProgramId());
            model.setProgram(program);
            model.setPoche(1);

            if (log.isDebugEnabled()) {
                log.debug("Will create a new cruise from program: " + program);
            }
        } else {

            // load existing program
            Cruise cruise =
                    persistenceService.getCruise(cruiseId);

            model.fromBean(cruise);
            if (log.isDebugEnabled()) {
                log.debug("Will edit cruise: " + cruiseId);
            }
        }
        listModelIsModify(model);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Person> users = persistenceService.getAllPerson();

        EditCruiseUIModel model = getModel();

        initBeanComboBox(ui.getSurveyComboBox(),
                         persistenceService.getAllProgram(),
                         model.getProgram());

        initBeanComboBox(ui.getCountryComboBox(),
                         persistenceService.getAllCountry(),
                         model.getCountry());

        initBeanList(ui.getVesselList(),
                     persistenceService.getAllVessel(),
                     model.getVessel());

        initBeanList(ui.getGearList(),
                     persistenceService.getAllGear(),
                     model.getGear());

        initBeanList(ui.getHeadOfMissionList(),
                     users,
                     model.getHeadOfMission());

        initBeanList(ui.getHeadOfSortRoomList(),
                     users,
                     model.getHeadOfSortRoom());

        SwingValidatorUtil.installUI(ui.getErrorTable(),
                                     new SwingValidatorMessageTableRenderer());

        listenValidatorValid(ui.getValidator(), model);

        // if new fishingOperation can already cancel his creation
        model.setModify(model.isCreate());
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }

    @Override
    protected EditCruiseUIModel getModel() {
        return ui.getModel();
    }

    public void cancel() {

        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public void save() {

        EditCruiseUIModel model = getModel();

        Cruise bean = model.toBean();

        Cruise saved;
        if (TuttiEntities.isNew(bean)) {

            saved = persistenceService.createCruise(bean);
        } else {
            saved = persistenceService.saveCruise(bean);
        }

        context.setProgramId(saved.getProgram().getId());
        context.setCruiseId(saved.getId());

        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public static final String cruiseNameFormat = "%1$s_%2$s";

    public void generateCampaignName() {

        EditCruiseUIModel model = getModel();
        Program program = model.getProgram();
        Integer year = model.getYear();
        String name = String.format(cruiseNameFormat, program.getName(), year);

        String surveyPart = model.getSurveyPart();
        if (StringUtils.isNotEmpty(surveyPart)) {
            name += "_" + surveyPart;
        }
        model.setName(name);
    }

}
