/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseActionSupport.java 355 2012-03-12 08:29:59Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-0.4/echobase-ui/src/main/java/fr/ifremer/echobase/ui/actions/EchoBaseActionSupport.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui.actions;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.opensymphony.xwork2.ActionContext;
import fr.ifremer.echobase.EchoBaseConfiguration;
import fr.ifremer.echobase.I18nAble;
import fr.ifremer.echobase.InputFile;
import fr.ifremer.echobase.services.DecoratorService;
import fr.ifremer.echobase.services.DefaultEchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseService;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceFactory;
import fr.ifremer.echobase.ui.EchoBaseApplicationContext;
import fr.ifremer.echobase.ui.EchoBaseSession;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.StrutsStatics;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.framework.TopiaTransactionAware;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.web.filter.TopiaTransactionFilter;
import org.nuiton.web.struts2.BaseAction;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Map;

/**
 * EchoBase action support.
 * <p/>
 * <b>Note :</b> All actions must implement this class to have a better i18n
 * support, when a i18n key is not translated it has a empty translation and
 * we do NOT want this behaviour in gui, prefer to return the marked
 * untranslated key.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseActionSupport extends BaseAction implements TopiaTransactionAware {

    private static final long serialVersionUID = 1L;

    /** Key to store the {@link EchoBaseSession} instance in the session's map. */
    protected static final String SESSION_PARAMETER = "echoBaseSession";

    /** Key to store the single instance of the application context */
    public static final String APPLICATION_CONTEXT_PARAMETER = "echoBaseApplicationContext";

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EchoBaseActionSupport.class);

    private SimpleDateFormat dateFormat;

    private SimpleDateFormat monthFormat;

    private transient DecoratorService decoratorService;

    public static EchoBaseApplicationContext getEchoBaseApplicationContext() {
        Map<String, Object> application = getActionContext().getApplication();
        EchoBaseApplicationContext applicationContext =
                (EchoBaseApplicationContext) application.get(APPLICATION_CONTEXT_PARAMETER);
        return applicationContext;
    }

    protected EchoBaseConfiguration getConfiguration() {
        return getEchoBaseApplicationContext().getConfiguration();
    }

    /** EchoBase User session. */
    private transient EchoBaseSession echoBaseSession;

    /**
     * Provides a way to get a service.
     * <p/>
     * Actions may <strong>not</strong> call it directly by use
     * {@link #newService(Class)} instead.
     */
    protected transient EchoBaseServiceFactory serviceFactory;

    protected transient EchoBaseServiceContext serviceContext;

    protected transient TopiaContext transaction;

    public EchoBaseSession getEchoBaseSession() {
        if (echoBaseSession == null) {

            // load it from session
            Map<String, Object> strutsSession =
                    getActionContext().getSession();
            EchoBaseSession session = (EchoBaseSession)
                    strutsSession.get(SESSION_PARAMETER);

            if (session == null) {

                // create an empty session
                session = new EchoBaseSession();

                // and store it in the struts session
                strutsSession.put(SESSION_PARAMETER, session);
            }
            echoBaseSession = session;
        }
        return echoBaseSession;
    }

    public boolean isAdmin() {
        EchoBaseSession session = getEchoBaseSession();
        boolean result = session.isAdmin();
        return result;
    }

    public static String getApplicationVersion() {
        return getEchoBaseApplicationContext().getConfiguration().getApplicationVersion().toString();
    }

    /**
     * Fabrique pour récupérer le ServiceContext tel qu'il devrait être fourni
     * à la fabrication d'un service.
     *
     * @return service context
     */
    protected EchoBaseServiceContext getServiceContext() {
        if (serviceContext == null) {
            serviceContext = DefaultEchoBaseServiceContext.newContext(
                    getLocale(),
                    getTransaction(),
                    getConfiguration(),
                    getEchoBaseApplicationContext().getDbMeta(),
                    getServiceFactory()
            );
        }
        return serviceContext;
    }

    public EchoBaseServiceFactory getServiceFactory() {
        if (serviceFactory == null) {
            serviceFactory = new EchoBaseServiceFactory();
        }
        return serviceFactory;
    }

    /**
     * Sub-classes should use this method to easily get a service instance.
     *
     * @param serviceClass The type of service to instantiate
     * @return A newly created service of the expected type with necessary data set
     */
    public <E extends EchoBaseService> E newService(Class<E> serviceClass) {
        E service = getServiceFactory().newService(serviceClass,
                                                   getServiceContext());
        return service;
    }

    @Override
    public TopiaContext getTransaction() {
        if (transaction == null) {
            HttpServletRequest request = (HttpServletRequest)
                    getActionContext().get(StrutsStatics.HTTP_REQUEST);
            transaction = TopiaTransactionFilter.getTransaction(request);
        }
        return transaction;
    }

    @Override
    public void setTransaction(TopiaContext transaction) {
        this.transaction = transaction;
    }

    public String formatDate(Date date) {
        String result = getDateFormat().format(date);
        return result;
    }

    public String formatMonth(Date date) {
        String result = getMonthFormat().format(date);
        return result;
    }

    public <E extends TopiaEntity> Map<String, String> sortAndDecorate(Collection<E> beans, String context) {
        return getDecoratorService().sortAndDecorate(getLocale(), beans, context);
    }

    public String decorate(Object o, String context) {
        return getDecoratorService().decorate(getLocale(), o, context);
    }

    protected SimpleDateFormat getDateFormat() {
        if (dateFormat == null) {
            dateFormat = new SimpleDateFormat("dd/MM/yyyy");
        }
        return dateFormat;
    }

    protected SimpleDateFormat getMonthFormat() {
        if (monthFormat == null) {
            monthFormat = new SimpleDateFormat("mm-yyyy");
        }
        return monthFormat;
    }

    protected DecoratorService getDecoratorService() {
        if (decoratorService == null) {
            decoratorService = newService(DecoratorService.class);
        }
        return decoratorService;
    }

    protected static ActionContext getActionContext() {
        return ActionContext.getContext();
    }

    protected void copyFile(InputFile inputFile,
                            File dataDirectory) throws IOException {
        Preconditions.checkNotNull(inputFile);
        Preconditions.checkNotNull(dataDirectory);
        File source = inputFile.getFile();
        Preconditions.checkNotNull(source);

        File target = new File(dataDirectory, inputFile.getFileName());
        if (log.isInfoEnabled()) {
            log.info("Copy file " + source + " to " + target);
        }
        FileUtils.copyFile(source, target);

        // keep target (source file will be removed at the end of this action)
        inputFile.setFile(target);
    }

    protected <E extends Enum<E> & I18nAble> Map<String, String> decorateEnums(E... enumValues) {
        Map<String, String> result = Maps.newLinkedHashMap();
        for (E enumValue : enumValues) {
            result.put(enumValue.name(), _(enumValue.getI18nKey()));
        }
        return result;
    }
}
