package fr.ifremer.coselmar.services.indexation;

/*
 * #%L
 * Coselmar :: Rest Services
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 - 2015 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import fr.ifremer.coselmar.beans.QuestionBean;
import fr.ifremer.coselmar.beans.QuestionSearchBean;
import fr.ifremer.coselmar.services.CoselmarSimpleServiceSupport;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StringField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.WildcardQuery;

/**
 * This Services provides operation about {@link fr.ifremer.coselmar.persistence.entity.Document}
 * or more exactly {@link fr.ifremer.coselmar.beans.DocumentBean} indexation :
 * <ul>
 * <li>registration of a document in the indexation db</li>
 * <li>modification of a document in the indexation db</li>
 * <li>documents search from the indexation db</li>
 * </ul>
 *
 * The purpose is to use power of a indexation db (lucene) to increase search on
 * document text field, and make easier fulltext search
 *
 * @author ymartel <martel@codelutin.com>
 */
public class QuestionsIndexationService extends CoselmarSimpleServiceSupport {

    protected static final String QUESTION_ID_INDEX_PROPERTY = "questionId";
    protected static final String QUESTION_TITLE_INDEX_PROPERTY = "questionTitle";
    protected static final String QUESTION_SUMMARY_INDEX_PROPERTY = "questionSummary";
    protected static final String QUESTION_THEME_INDEX_PROPERTY = "questionTheme";
    protected static final String QUESTION_STATUS_INDEX_PROPERTY = "questionStatus";
    protected static final String QUESTION_PRIVACY_INDEX_PROPERTY = "questionPrivacy";
    protected static final String DOCUMENT_TYPE = "question";

    public void indexQuestion(QuestionBean question) throws IOException {

        // First : try to find if already exist to update it
        DirectoryReader ireader = DirectoryReader.open(getLuceneUtils().getIndexWriter(), false);
        IndexSearcher isearcher = new IndexSearcher(ireader);

        // Retrieve document
        BooleanQuery query = new BooleanQuery();
        query.add(new TermQuery(new Term(QUESTION_ID_INDEX_PROPERTY, question.getId())), BooleanClause.Occur.MUST);
        query.add(new TermQuery(new Term("type", DOCUMENT_TYPE)), BooleanClause.Occur.MUST);

        ScoreDoc[] hits = isearcher.search(query, null, 1000).scoreDocs;
        if (hits.length > 0) {
            Document doc = new Document();

            doc.add(new StringField(QUESTION_ID_INDEX_PROPERTY, question.getId(), Field.Store.YES));

            doc.add(new TextField(QUESTION_TITLE_INDEX_PROPERTY, question.getTitle(), Field.Store.YES));
            doc.add(new TextField(QUESTION_SUMMARY_INDEX_PROPERTY, question.getSummary(), Field.Store.YES));

            Set<String> themes = question.getThemes();
            if (themes != null) {
                for (String theme : themes) {
                    doc.add(new Field(QUESTION_THEME_INDEX_PROPERTY, theme, TextField.TYPE_STORED));
                }
            }

            doc.add(new TextField(QUESTION_STATUS_INDEX_PROPERTY, question.getStatus(), Field.Store.YES));

            doc.add(new TextField(QUESTION_PRIVACY_INDEX_PROPERTY, question.getPrivacy(), Field.Store.YES));

            doc.add(new Field("type", DOCUMENT_TYPE, TextField.TYPE_STORED));

            getLuceneUtils().getIndexWriter().updateDocument(new Term(QUESTION_ID_INDEX_PROPERTY, question.getId()), doc);

        } else {
            // Not exist yet : add it to index

            Document doc = new Document();
            doc.add(new StringField(QUESTION_ID_INDEX_PROPERTY, question.getId(), Field.Store.YES));

            doc.add(new TextField(QUESTION_TITLE_INDEX_PROPERTY, question.getTitle(), Field.Store.YES));
            doc.add(new TextField(QUESTION_SUMMARY_INDEX_PROPERTY, question.getSummary(), Field.Store.YES));

            doc.add(new TextField(QUESTION_STATUS_INDEX_PROPERTY, question.getStatus(), Field.Store.YES));
            doc.add(new TextField(QUESTION_PRIVACY_INDEX_PROPERTY, question.getPrivacy(), Field.Store.YES));

            Set<String> themes = question.getThemes();
            if (themes != null) {
                for (String theme : themes) {
                    doc.add(new Field(QUESTION_THEME_INDEX_PROPERTY, theme, TextField.TYPE_STORED));
                }
            }

            doc.add(new Field("type", DOCUMENT_TYPE, TextField.TYPE_STORED));

            getLuceneUtils().getIndexWriter().addDocument(doc);

        }

        // Commit, close reader.
        getLuceneUtils().getIndexWriter().commit();
        ireader.close();

    }

    public List<String> searchQuestion(QuestionSearchBean searchBean) throws IOException, ParseException {
        DirectoryReader ireader = DirectoryReader.open(getLuceneUtils().getIndexWriter(), false);
        IndexSearcher isearcher = new IndexSearcher(ireader);

        // Combine that with the type
        BooleanQuery fullQuery = new BooleanQuery();
        fullQuery.add(new TermQuery(new Term("type", DOCUMENT_TYPE)), BooleanClause.Occur.MUST);

        String searchPrivacy = searchBean.getPrivacy();
        if(StringUtils.isNotBlank(searchPrivacy)) {
            fullQuery.add(new TermQuery(new Term(QUESTION_PRIVACY_INDEX_PROPERTY, searchPrivacy.toLowerCase())), BooleanClause.Occur.MUST);
        }

        String searchStatus = searchBean.getStatus();
        if(StringUtils.isNotBlank(searchStatus)) {
            fullQuery.add(new TermQuery(new Term(QUESTION_STATUS_INDEX_PROPERTY, searchStatus.toLowerCase())), BooleanClause.Occur.MUST);
        }

        // Keywords part
        List<String> keywords = searchBean.getKeywords();
        if (keywords != null && !keywords.isEmpty()) {
            BooleanQuery keywordsQuery = new BooleanQuery();

            for (String text : keywords) {

                String[] words = text.replaceAll("[^a-zA-Z ]", "").toLowerCase().split(" ");

                // Parse a simple query that searches for the "text":
                BooleanQuery query = new BooleanQuery();

                BooleanQuery nameQuery = new BooleanQuery();
                BooleanQuery summaryQuery = new BooleanQuery();

                for (String word : words) {
                    String wildWord = String.format("*%s*", word.toLowerCase());
                    nameQuery.add(new WildcardQuery(new Term(QUESTION_TITLE_INDEX_PROPERTY, wildWord)), BooleanClause.Occur.MUST);
                    summaryQuery.add(new WildcardQuery(new Term(QUESTION_SUMMARY_INDEX_PROPERTY, wildWord)), BooleanClause.Occur.MUST);
                }

                query.add(nameQuery, BooleanClause.Occur.SHOULD);
                query.add(summaryQuery, BooleanClause.Occur.SHOULD);

                query.add(new TermQuery(new Term(QUESTION_THEME_INDEX_PROPERTY, text.toLowerCase())), BooleanClause.Occur.SHOULD);

                keywordsQuery.add(query, BooleanClause.Occur.MUST);
            }

            // add to complete query
            fullQuery.add(keywordsQuery, BooleanClause.Occur.MUST);
        }


        ScoreDoc[] hits = isearcher.search(fullQuery, null, 1000).scoreDocs;

        List<String> documentIds = new ArrayList(hits.length);

        for (ScoreDoc hit : hits) {
            Document doc = isearcher.doc(hit.doc);
            String documentId = doc.get(QUESTION_ID_INDEX_PROPERTY);
            documentIds.add(documentId);
        }

        ireader.close();
        return documentIds;
    }

    public void deleteQuestion(String documentId) throws IOException {

        // Retrieve document
        BooleanQuery query = new BooleanQuery();
        query.add(new TermQuery(new Term(QUESTION_ID_INDEX_PROPERTY, documentId)), BooleanClause.Occur.MUST);
        query.add(new TermQuery(new Term("type", DOCUMENT_TYPE)), BooleanClause.Occur.MUST);

        getLuceneUtils().getIndexWriter().deleteDocuments(query);
        getLuceneUtils().getIndexWriter().commit();

    }

    protected void cleanIndex() throws IOException {
        BooleanQuery query = new BooleanQuery();
        query.add(new TermQuery(new Term("type", DOCUMENT_TYPE)), BooleanClause.Occur.MUST);
        getLuceneUtils().getIndexWriter().deleteDocuments(query);
        getLuceneUtils().getIndexWriter().commit();
    }


}
