/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: ThesaurusChangesTreeTableModel.java 1774 2010-11-15 12:29:52Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.2/vradi-swing/src/main/java/com/jurismarches/vradi/ui/admin/query/ThesaurusChangesTreeTableModel.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.admin.query;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.tree.TreePath;

import com.jurismarches.vradi.ui.helpers.VradiComparators;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;
import org.nuiton.i18n.I18n;

import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.services.VradiStorageService;

/**
 * ThesaurusChangesTreeTableModel.
 *
 * @author schorlet
 * @version $Revision: 1774 $ $Date: 2010-11-15 13:29:52 +0100 (lun., 15 nov. 2010) $
 * @since 8 avr. 2010 15:19:16
 */
public class ThesaurusChangesTreeTableModel extends AbstractTreeTableModel {
    private static final Log log = LogFactory.getLog(ThesaurusChangesTreeTableModel.class);
    
    /** clients and groups */
    final ArrayList<Group> topLevels = new ArrayList<Group>();

    /** queryMakers indexed by wikittyId */
    final HashMap<String, Group> groupsMap = new HashMap<String, Group>();
    
    /** queries indexed by wikittyId */
    final HashMap<String, List<QueryBean>> queryBeanMap = new HashMap<String, List<QueryBean>>();
    
    /** key:before, value:after */
    final HashMap<QueryBean, QueryBean> modifiedBeanMap = new HashMap<QueryBean, QueryBean>();
    
    /** queries validation state */
    final HashMap<QueryBean, Boolean> queryBeanValidate = new HashMap<QueryBean, Boolean>();
    
    public ThesaurusChangesTreeTableModel(Map<Group, List<QueryBean>> queryMap, String rootThesaurusBefore,
                                          String rootThesaurusAfter, String thesaurusBefore, String thesaurusAfter) {
        super(1L);

        HashMap<String, Group> groupMap = new HashMap<String, Group>();
        
        for (Map.Entry<Group, List<QueryBean>> entry : queryMap.entrySet()) {
            // queryMaker
            Group group = entry.getKey();
            // original queries for queryMaker
            List<QueryBean> queries = entry.getValue();
            
            String wikittyId = group.getWikittyId();

            // index queries  by wikittyId
            queryBeanMap.put(wikittyId, queries);

            // group
            groupMap.put(wikittyId, group);

            // index queries by original:modified
            for (QueryBean original : queries) {
                String query = original.getQuery();

                // If root thesaurus modif
                if (!rootThesaurusBefore.equals(rootThesaurusAfter)) {
                    query = StringUtils.replace(query, rootThesaurusBefore + ':', rootThesaurusAfter + ':');
                }

                // If thesaurus modif
                if (thesaurusAfter != null) {
                    if (!thesaurusBefore.equals(thesaurusAfter)) {
                        query = StringUtils.replace(query, ':' + thesaurusBefore, ':' + thesaurusAfter);
                        query = StringUtils.replace(query, ":\"" + thesaurusBefore + "\"",  ":\"" + thesaurusAfter + "\"");
                    }
                }

                QueryBean modified = original.setQuery(query);
                
                modifiedBeanMap.put(original, modified);
                queryBeanValidate.put(original, Boolean.TRUE);
            }
        }

        List<Group> groupList = new ArrayList<Group>(groupMap.values());

        Collections.sort(groupList, VradiComparators.GROUP_COMPARATOR);

        topLevels.addAll(groupList);
        
        groupsMap.putAll(groupMap);
    }
    
    public List<Group> getUpdatedQueryMakers() throws IOException {
        List<Group> list = new ArrayList<Group>();
        
        for (Group queryMaker : groupsMap.values()) {
            list.add(queryMaker);
            Set<String> queries = new HashSet<String>();
            Set<String> initialQueries = queryMaker.getQueries();
            if (initialQueries != null) {
                queries.addAll(initialQueries);
            }
            queryMaker.clearQueries();
            
            for (String query : queries) {
                QueryBean original = new QueryBean(query, queryMaker.getWikittyId());
                
                if (queryBeanValidate.containsKey(original) && queryBeanValidate.get(original)) {
                    QueryBean modified = modifiedBeanMap.get(original);
                    queryMaker.addQueries(modified.getQueryLine());
                    
                } else {
                    queryMaker.addQueries(query);
                }
            }
            
        }
        return list;
    }
    
    @Override
    public int getColumnCount() {
        return 5;
    }

    @Override
    public Object getValueAt(Object node, int column) {
        if (node instanceof Group) {
            if (column == 0) {
                Group group = (Group) node;
                return group.getName();
            }
            
        } else if (node instanceof QueryBean) {
            QueryBean bean = (QueryBean) node;
            if (column == 0) {
                return bean.getName();
                
            } else if (column == 1) {
                return bean.getDescription();
                
            } else if (column == 2) {
                return bean.getQuery();
                
            } else if (column == 3) {
                QueryBean modifiedBean = modifiedBeanMap.get(bean);
                return modifiedBean.getQuery();
                
            } else if (column == 4) {
                boolean validate = queryBeanValidate.get(bean);
                return validate;
            }
        }
        
        return null;
    }

    @Override
    public boolean isCellEditable(Object node, int column) {
        if (column == 3 || column == 4) {
            boolean b = node instanceof QueryBean;
            return b;
        }
        return false;
    }
    
    @Override
    public void setValueAt(Object value, Object node, int column) {
        if (column == 3) {
            QueryBean bean = (QueryBean) node;
            String beanId = bean.getId();
            if (beanId == null) {
                return;
            }
            
            QueryBean initialBean = modifiedBeanMap.get(bean);
            QueryBean modifiedBean = initialBean.setQuery(String.valueOf(value));

            if (!initialBean.equals(modifiedBean)) {
                modifiedBeanMap.put(bean, modifiedBean);
                queryBeanValidate.put(bean, Boolean.TRUE);
                fireQueryChanged(bean);
            }
            
        } else if (column == 4) {
            QueryBean bean = (QueryBean) node;
            String beanId = bean.getId();
            if (beanId == null) {
                return;
            }
            
            queryBeanValidate.put(bean, (Boolean) value);
            fireQueryChanged(bean);
        }
    }
    
    private void fireQueryChanged(QueryBean bean) {
        String beanId = bean.getId();
        
        TreePath parentPath = null;
        int index;
        
        Group group = groupsMap.get(beanId);

        Object[] paths = new Object[2];
        paths[0] = 1L;
        paths[1] = group;

        parentPath = new TreePath(paths);
        index = getIndexOfChild(paths[1], bean);
        
        modelSupport.fireChildChanged(parentPath, index, bean);
    }
    
    @Override
    public Object getChild(Object parent, int index) {
        if (parent instanceof Long) {
            return topLevels.get(index);
            
        } else if (parent instanceof Group) {
            Group group = (Group) parent;
            List<QueryBean> list = queryBeanMap.get(group.getWikittyId());
            QueryBean bean = list.get(index);
            return bean;
        }
        
        return null;
    }

    @Override
    public int getChildCount(Object parent) {
        if (parent instanceof Long) {
            return topLevels.size();
            
        } else if (parent instanceof Group) {
            Group group = (Group) parent;
            List<QueryBean> list = queryBeanMap.get(group.getWikittyId());
            return list.size();
        }
        
        return 0;
    }
    
    @Override
    public int getIndexOfChild(Object parent, Object child) {
        if (parent instanceof Long) {
            return topLevels.indexOf(child);
            
        } else if (parent instanceof Group) {
            Group group = (Group) parent;
            List<QueryBean> list = queryBeanMap.get(group.getWikittyId());
            return list.indexOf(child);
        }
        
        return -1;
    }
    
    @Override
    public String getColumnName(int column) {
        String columnName = null;
        if (column == 0) {
            columnName = I18n._("vradi.requestFormView.resultTable.name");
        } else if (column == 1) {
            columnName = I18n._("vradi.requestFormView.resultTable.description");
        } else if (column == 2) {
            columnName = I18n._("vradi.requestFormView.resultTable.oldQuery");
        } else if (column == 3) {
            columnName = I18n._("vradi.requestFormView.resultTable.newQuery");
        }
        return columnName;
    }
    
    private VradiStorageService vradiStorageService = null;
    
    protected VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = VradiService.getVradiStorageService();
        }
        return vradiStorageService;
    }
    
}
