/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: VradiConfig.java 1802 2010-11-17 22:37:06Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.4.0/vradi-swing/src/main/java/com/jurismarches/vradi/VradiConfig.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi;

import static org.nuiton.i18n.I18n._;

import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.Locale;

import jaxx.runtime.context.DefaultApplicationContext.AutoLoad;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.util.StringUtil;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;
import org.nuiton.wikitty.WikittyServiceNotifier;
import org.nuiton.wikitty.XMPPNotifierTransporter;

/**
 * Vradi UI configuration.
 * 
 * Class is auto instanced by Jaxx when getting it from context.
 * 
 * @author chatellier
 * @version $Revision: 1802 $
 * 
 * Last update : $Date: 2010-11-17 23:37:06 +0100 (mer., 17 nov. 2010) $
 * By : $Author: sletellier $
 */
@AutoLoad
public class VradiConfig extends ApplicationConfig {

    /** to use log facility, just put in your code: log.info(\"...\");*/
    static private Log log = LogFactory.getLog(VradiConfig.class);

    /** Un drapeau pour savoir si on est en mode pleine écran. */
    protected boolean fullscreen;

    /** Un drepeau pour savoir s'il faut lancer l'interface graphique.
     * Cette valeur peut être programmées lors des actions. */
    private boolean displayMainUI = true;

    /**
     * drapeau pour savoir si on peut utiliser des ui dans l'environnement.
     * <p/>
     * Par defaut, on suppose qu'on peut utiliser l'environnement graphique
     * et si on désactive explicitement ou si pas d'environnement graphique trouvé.
     */
    private boolean canUseUI = true;

    /**
     * Constructor.
     * 
     * Init default options and actions.
     */
    public VradiConfig() {

        super();

        // init configuration with default options
        for (Option o : Option.values()) {
            if (o.defaultValue != null) {
                setDefaultOption(o.key, o.defaultValue);
            }
        }

        // init actions
        for (Action a : Action.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);
            }
        }

        // TODO sletellier 08/11/10 : find a better way
        setSwingConfigFile(getConfigPath() + File.separator + "vradi-swing.properties");
    }

    /**
     * Overwrite {@link ApplicationConfig#parse(String[])} to manage version after parsing.
     */
    @Override
    public void parse(String[] args) throws ArgumentsParserException {
        super.parse(args);
        
        // on supprime le stamp de snapshot s'il existe
        String sVersion = VersionUtil.removeSnapshot(getOption("application.version"));
        Version version = VersionUtil.valueOf(sVersion);
        setDefaultOption(Option.VRADI_VERSION.key, version.getVersion());

        // la version de la base est sans classifier (pas de alpha, ou rc,...)
        Version dbVersion;
        if (version.hasClassifier()) {
            dbVersion = VersionUtil.removeClassifier(version);
        } else {
            dbVersion = VersionUtil.valueOf(version.toString());
        }

        setDefaultOption(Option.VRADI_DATABASE_VERSION.key, dbVersion.getVersion());
    }

    public String getCopyrightText() {
        return "Version " + getVersion() + " JurisMarches @ 2009 - 2010";
    }

    /**
     * La version du logiciel.
     *
     * @return la version de l'application.
     */
    public Version getVersion() {
        Version option = getOption(Version.class, Option.VRADI_VERSION.key);
        return option;
    }

    /**
     * @return la version de la base
     */
    public Version getDatabaseVersion() {
        Version option = getOption(Version.class, Option.VRADI_DATABASE_VERSION.key);
        return option;
    }

    /**
     * La derniere version de lancement du logiciel.
     *
     * @return la derniere version de lancement
     */
    public String getLastVersion() {
        String lastVersion = getOption(Option.VRADI_LAST_VERSION.key);
        return lastVersion;
    }

    /**
     * Change last version.
     * 
     * @param lastVersion new last version
     */
    public void setLastVersion(String lastVersion) {
        setOption(Option.VRADI_LAST_VERSION.key, lastVersion);
    }
    
    public boolean isFullScreen() {
        Boolean result = getOptionAsBoolean(Option.FULL_SCREEN.key);
        return result != null && result;
    }

    public boolean isDisplayMainUI() {
        return displayMainUI;
    }

    public boolean isCanUseUI() {
        return canUseUI;
    }

    public Locale getLocale() {
        Locale result = getOption(Locale.class, Option.LOCALE.key);
        return result;
    }

    public void setFullscreen(boolean fullscreen) {
        boolean oldValue = isFullScreen();
        setOption(Option.FULL_SCREEN.key, fullscreen + "");
        saveForUser();
        firePropertyChange("fullscreen", oldValue, fullscreen);
    }

    public void setDisplayMainUI(boolean b) {
        displayMainUI = b;
    }

    public void setCanUseUI(boolean canUseUI) {
        this.canUseUI = canUseUI;
        if (!canUseUI) {
            // on ne pourra pas lancer l'ui principale
            setDisplayMainUI(false);
        }
    }

    public void setLocale(Locale newLocale) {
        setOption(Option.LOCALE.key, newLocale.toString());
        saveForUser();
        firePropertyChange("locale", null, newLocale);
    }

    /**
     * Get user login.
     * 
     * @return user login
     */
    public String getLogin() {
        return getOption(Option.USERNAME.key);
    }

    /**
     * Set user login.
     * 
     * @param login user login
     */
    public void setLogin(String login) {
        setOption(Option.USERNAME.key, login);
    }

    /**
     * Get user password.
     * 
     * @return user password
     */
    public String getPassword() {
        return getOption(Option.PASSWORD.key);
    }

    /**
     * Set user password.
     * 
     * @param password user password
     */
    public void setPassword(String password) {
        setOption(Option.PASSWORD.key, password);
    }

    /**
     * Get remote endpoint base url.
     * 
     * If not valid url, embedded mode is used.
     * 
     * @return remote endpoint base url
     */
    public String getRemoteEndpoint() {
        String endpoint = getOption(Option.REMOTE_ENDPOINT.key);
        return endpoint;
    }
    
    /**
     * Get user queries (comma separated).
     * 
     * Warning to inner comma in queries, use {@link StringUtil#split(String, String)}
     * 
     * @return user queries.
     */
    public String getQueries() {
        String queries = getOption(Option.QUERIES.key);
        return queries;
    }

    /**
     * Set user queries and save user configuration.
     * 
     * @param queries user queries (comma separated)
     */
    public void setQueries(String queries) {
        setOption(Option.QUERIES.key, queries);
        saveForUser();
    }
    
    /**
     * Get user criteria (comma separated).
     * 
     * @return user criteria.
     */
    public String getCriterias() {
        String criterias = getOption(Option.CRITERIAS.key);
        return criterias;
    }

    /**
     * Set user criterias and save user configuration.
     * 
     * @param criterias user criterias (comma separated)
     */
    public void setCriterias(String criterias) {
        setOption(Option.CRITERIAS.key, criterias);
        saveForUser();
    }

    /**
     * Get user root criteria (comma separated).
     *
     * @return user root criteria.
     */
    public String getRootCriterias() {
        String criterias = getOption(Option.CRITERIAS.key);
        return criterias;
    }

    /**
     * Set user root criterias and save user configuration.
     *
     * @param criterias user root criterias (comma separated)
     */
    public void setRootCriterias(String criterias) {
        setOption(Option.CRITERIAS.key, criterias);
        saveForUser();
    }

    /**
     * Get user rootThesaurus (comma separated).
     *
     * @return user thesaurus.
     */
    public String getRootThesaurus() {
        String rootThesaurus = getOption(Option.ROOT_THESAURUS.key);
        return rootThesaurus;
    }

    /**
     * Set user rootThesaurus and save user configuration.
     *
     * @param rootThesaurus user rootThesaurus (comma separated)
     */
    public void setRootThesaurus(String rootThesaurus) {
        setOption(Option.ROOT_THESAURUS.key, rootThesaurus);
        saveForUser();
    }

    /**
     * Get user thesaurus (comma separated).
     *
     * @return user thesaurus.
     */
    public String getThesaurus() {
        String thesaurus = getOption(Option.THESAURUS.key);
        return thesaurus;
    }

    /**
     * Set user thesaurus and save user configuration.
     *
     * @param thesaurus user thesaurus (comma separated)
     */
    public void setThesaurus(String thesaurus) {
        setOption(Option.THESAURUS.key, thesaurus);
        saveForUser();
    }

    /**
     * Get user columns list : offer (comma separated).
     *
     * @return user columns.
     */
    public String getOfferColumns() {
        String columns = getOption(Option.COLUMNS.key);
        return columns;
    }

    /**
     * Set user columns and save user configuration.
     *
     * @param columns user columns (comma separated)
     */
    public void setOfferColumns(String columns) {
        setOption(Option.COLUMNS.key, columns);
        saveForUser();
    }

    /**
     * Get thesaurus color and level.
     *
     * Storage is : 0=255,323,232;1=232,112,234;...
     *
     * @return thesaurus level and color
     */
    public String getThesaurusColors() {
        String columns = getOption(Option.THESAURUS_COLORS.key);
        return columns;
    }

    /**
     * Set user thesaurus color and level.
     *
     * Storage is : 0=255,323,232;1=232,112,234;...
     *
     * @param levelAndColors user columns (; separated)
     */
    public void setThesaurusColors(String levelAndColors) {
        setOption(Option.THESAURUS_COLORS.key, levelAndColors);
        saveForUser();
    }

    /**
     * Get the swing vradi config file (used by {@link org.nuiton.widget.SwingSession})
     *
     * @return file concerned
     */
    public File getSwingConfigFile() {
        return getOptionAsFile(Option.SWING_CONFIG_FILE.key);
    }

    public void setSwingConfigFile(String path) {
        setOption(Option.SWING_CONFIG_FILE.key, path);
    }

    /**
     * Get thesaurus expanded state.
     *
     * @return thesaurus expanded state
     */
    public boolean isOnClickThesaurusExpanded() {
        boolean result = getOptionAsBoolean(Option.THESAURUS_EXPANDED.key);
        return result;
    }

    /**
     * Set thesaurus expanded state.
     *
     * @param state thesaurus expanded state
     */
    public void setOnClickThesaurusExpanded(boolean state) {
        setOption(Option.THESAURUS_EXPANDED.key, String.valueOf(state));
        saveForUser();
    }

    /**
     * Change remote endpoint url.
     *
     * @param newEndpoint new base url
     */
    public void setRemoteEndPoint(String newEndpoint) {
        setOption(Option.REMOTE_ENDPOINT.key, newEndpoint);
    }

    /**
     * To specify XMPP room to use
     *
     * @param xmppRoom to use
     *
     * @see XMPPNotifierTransporter#WIKITTY_XMPP_ROOM
     */
    public void setWikittyXMPPRoom(String xmppRoom) {
        setOption(Option.WIKITTY_XMPP_ROOM.key, xmppRoom);
    }

    /**
     * to specify XMPP server to use
     *
     * @param xmppServer to use
     *
     * @see XMPPNotifierTransporter#WIKITTY_XMPP_SERVER
     */
    public void setWikittyXMPPServeur(String xmppServer) {
        setOption(Option.WIKITTY_XMPP_SERVER.key, xmppServer);
    }

    /**
     * To specify class to use for notifier transporter
     *
     * @param remoteNotifierClass class to use
     *
     * @see WikittyServiceNotifier.RemoteNotifier#WIKITTY_NOTIFIER_TRANSPORTER_CLASS
     */
    public void setWikittyNotifierTransporteurClass(Class remoteNotifierClass) {
        setOption(Option.WIKITTY_EVENT_PROPAGATE_OPTION.key, remoteNotifierClass.getName());
    }

    /**
     * Change propagate events use.
     *
     * @param b new value
     *
     * @see WikittyServiceNotifier.RemoteNotifier#WIKITTY_EVENT_PROPAGATE_OPTION
     */
    public void setWikittyPropagateEvents(boolean b) {
        setOption(Option.WIKITTY_EVENT_PROPAGATE_OPTION.key, String.valueOf(b));
    }

    protected static final String[] DEFAULT_JAXX_PCS = {
            "showNumberEditorButton", "autoPopupNumberEditor", "fullScreen", "locale"
    };

    /**
     * TODO EC20100519 used by ?
     */
    public void removeJaxxPropertyChangeListener() {
        PropertyChangeListener[] toRemove = jaxx.runtime.SwingUtil.findJaxxPropertyChangeListener(DEFAULT_JAXX_PCS, getPropertyChangeListeners());
        if (toRemove == null || toRemove.length == 0) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("before remove : " + getPropertyChangeListeners().length);
            log.debug("toRemove : " + toRemove.length);
        }
        for (PropertyChangeListener listener : toRemove) {
            removePropertyChangeListener(listener);
        }
        if (log.isDebugEnabled()) {
            log.debug("after remove : " + getPropertyChangeListeners().length);
        }
    }

    /**
     * Vradi option enum.
     */
    public static enum Option implements OptionDef {

        CONFIG_FILE(CONFIG_FILE_NAME, _("vradi.config.configFileName.description"), "vradi.properties", String.class, true, true),
        SWING_CONFIG_FILE("vradi.swing.config", _("vradi.config.swingConfigFileName.description"), "vradi-swing.properties", File.class, true, true),
        VRADI_VERSION("vradi.version", _("vradi.config.version.description"), null, String.class, true, true),
        VRADI_DATABASE_VERSION("vradi.database.version", _("vradi.config.database.version.description"), null, String.class, true, true),
        VRADI_LAST_VERSION("vradi.last.version", _("vradi.config.last.version.description"), null, String.class, true, true),

        // ui config
        FULL_SCREEN("ui.fullscreen", _("vradi.config.ui.fullscreen"), "false", Boolean.class, false, false),
        LOCALE("ui.locale", _("vradi.config.ui.locale"), Locale.FRANCE.toString(), Locale.class, false, false),
        // login and password
        USERNAME("login", _("vradi.config.login"), "", String.class, true, true),
        PASSWORD("password", _("vradi.config.password"), "", String.class, true, true),
        // achitecture client serveur
        REMOTE_ENDPOINT("vradi.remote.endpoint", _("vradi.config.remote.endpoint.description"), "", String.class, false, false),

        // XMPP server configuration
        WIKITTY_XMPP_ROOM(XMPPNotifierTransporter.WIKITTY_XMPP_ROOM, _("vradi.config.wikitty.xmpp.room.description"), null, String.class, false, false),
        WIKITTY_XMPP_SERVER(XMPPNotifierTransporter.WIKITTY_XMPP_SERVER, _("vradi.config.wikitty.xmpp.server.description"), null, String.class, false, false),
        WIKITTY_NOTIFIER_TRANSPORTER_CLASS(WikittyServiceNotifier.RemoteNotifier.WIKITTY_NOTIFIER_TRANSPORTER_CLASS,
                _("vradi.config.wikitty.notifier.transporter.class.description"),
                "org.nuiton.wikitty.XMPPNotifierTransporter", Class.class, false, false),
        WIKITTY_EVENT_PROPAGATE_OPTION(WikittyServiceNotifier.RemoteNotifier.WIKITTY_EVENT_PROPAGATE_OPTION,
                _("vradi.config.wikitty.propagate.propagateEvent.description"), "false", Boolean.class, false, false),

        // user datas
        QUERIES("vradi.queries", _("vradi.config.queries.description"), "", String.class, false, false),
        XML_STREAMS("vradi.xmlStreams", _("vradi.config.xmlStreams.description"), "", String.class, false, false),
        CRITERIAS("vradi.xmlStreams", _("vradi.config.criteria.description"), "", String.class, false, false),
        ROOT_CRITERIAS("vradi.root.criteria", _("vradi.config.root.criteria.description"), "", String.class, false, false),
        THESAURUS("vradi.thesaurus", _("vradi.config.thesaurus.description"), "", String.class, false, false),
        ROOT_THESAURUS("vradi.rootThesaurus", _("vradi.config.rootThesaurus.description"), "", String.class, false, false),
        COLUMNS("vradi.columns", _("vradi.config.columns.description"), "", String.class, false, false),
        THESAURUS_COLORS("vradi.thesaurus.color",  _("vradi.config.thesaurus.color.description"), "", String.class, false, false),
        STATUS_COLOR("vradi.status.color", _("vradi.config.status.color.description"), "", String.class, false, false),
        THESAURUS_EXPANDED("vradi.thesaurus.expanded", _("vradi.config.thesaurus.expanded.description"), "", Boolean.class, false, false);
        
        public String key;
        public String description;
        public String defaultValue;
        public Class<?> type;
        public boolean isTransient;
        public boolean isFinal;

        private Option(String key, String description, String defaultValue, Class<?> type, boolean isTransient, boolean isFinal) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this.isTransient = isTransient;
            this.isFinal = isFinal;
        }

        @Override
        public boolean isFinal() {
            return isFinal;
        }

        @Override
        public boolean isTransient() {
            return isTransient;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public String getDescription() {
            return description;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public Class<?> getType() {
            return type;
        }

        @Override
        public void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public void setTransient(boolean isTransient) {
            this.isTransient = isTransient;
        }

        @Override
        public void setFinal(boolean isFinal) {
            this.isFinal = isFinal;
        }
    }

    /**
     * Can't be used un @annotation.
     */
    @Deprecated
    public static enum Step {
        AfterInit, BeforeExit
    }

    public static enum Action {

        HELP(_("vradi.action.commandline.help"), VradiAction.class.getName() + "#help", "-h", "--help"),
        HELP_UI(_("vradi.action.commandline.help.ui"), VradiAction.class.getName() + "#helpUI", "--help-ui"),
        NO_MAIN_UI(_("vradi.action.commandline.disable.main.ui"), VradiAction.class.getName() + "#disableMainUI", "-n", "--no-main"),
        CONFIGURE_UI(_("vradi.action.commandline.configure.ui"), VradiAction.class.getName() + "#configure", "--configure");

        public String description;
        public String action;
        public String[] aliases;

        private Action(String description, String action, String... aliases) {
            this.description = description;
            this.action = action;
            this.aliases = aliases;
        }
    }
}
