/*
 * *##% 
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.admin.content.AdminThesaurusUI;
import com.jurismarches.vradi.ui.admin.ThesaurusPathChooserUI;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.decorator.Decorator;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.navigation.*;
import jaxx.runtime.swing.navigation.NavigationTreeHandler.Strategy;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.*;
import javax.swing.tree.TreePath;
import java.awt.*;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;

/**
 * @author letellier
 */
public class ThesaurusTreeHelper extends NavigationTreeHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusTreeHelper.class);

    public static String PATH_SEPARATOR = "/";

    protected NavigationTreeNode rootNode = null;

    protected ThesaurusTreeModelBuilder builder = null;

    protected VradiThesaurusDTO rootThesaurus = null;

    protected JAXXContext context = null;

    protected String refName = null;

    protected JAXXContextEntryDef<VradiThesaurusDTO> THESAURUS;

    // Use default ref and default root thesaurus
    public ThesaurusTreeHelper(JAXXContext context) {
        this(context, null, null);
    }

    // Use new ref with default root thesaurus
    public ThesaurusTreeHelper(JAXXContext context, String refName) {
        this(context, null, refName);
    }

    // Use default ref with custom root thesaurus
    public ThesaurusTreeHelper(JAXXContext context,  VradiThesaurusDTO rootThesaurus) {
        this(context, rootThesaurus, null);
    }

    // Use new ref (if refName is not null) and custom root thesaurus
    public ThesaurusTreeHelper(JAXXContext context, VradiThesaurusDTO rootThesaurus, String refName) {
        super(refName == null ? ThesaurusHandler.PREFIX_THESAURUS + rootThesaurus.getWikittyId() : refName);
        this.refName = refName;
        this.context = context;

        if (rootThesaurus != null){
            this.rootThesaurus = rootThesaurus;
        } else if (isCopy()){
            this.rootThesaurus = getRootRefThesaurus(context).clone();
        } else {
            this.rootThesaurus = getRootRefThesaurus(context);
        }

        // Create ENTRY def with unique name
        if (isCopy()){
            // Create custom referenciel
            THESAURUS = SwingUtil.newContextEntryDef(refName, VradiThesaurusDTO.class);

            // Create the model and store it in the given context.
            THESAURUS.setContextValue(context, this.rootThesaurus);
        }
    }

    // If this helper use an other referenciel
    protected boolean isCopy(){
        return refName != null;
    }

    // Get concerned ref
    protected JAXXContextEntryDef<VradiThesaurusDTO> getThesaurusRef() {
        return getThesaurusRef(isCopy());
    }

    /**
     * Get root thesaurus from VradiContext
     *
     * @param context
     * @return vradiThesaurusDTO from VradiContext
     */
    protected VradiThesaurusDTO getRootRefThesaurus(JAXXContext context){
        return getThesaurusRef(false).getContextValue(context);
    }

    /**
     * Get thesaurus ref
     *
     * @param copy
     * @return entryDef
     */
    protected JAXXContextEntryDef<VradiThesaurusDTO> getThesaurusRef(boolean copy) {
        if (copy){
             return THESAURUS;
        }
        return VradiContext.THESAURUS_ENTRY_DEF;
    }

    // Get thesaurus handler
    protected ThesaurusHandler getHandler(JAXXContext context) {
        return UIHelper.getHandler(context, ThesaurusHandler.class);
    }

    // Get the builder
    public ThesaurusTreeModelBuilder getBuilder() {
        return builder;
    }

    // Get this root thesaurus
    public VradiThesaurusDTO getRootThesaurus() {
        return rootThesaurus;
    }

    // Get this root node
    public NavigationTreeNode getRootNode() {
        return rootNode;
    }

    // Get this root name
    public String getRootName() {
        return rootThesaurus.getName();
    }

    // Get this root name path (like : Thesaurus/child)
    public String getRootNamePath() {
        return rootThesaurus.getNamePath(getPathSeparator());
    }

    public String getPathSeparator(){
        return PATH_SEPARATOR;
    }

    @Override
    public NavigationTreeModel createTreeModel(JAXXContext context) {
        this.context = context;
        if (builder == null) {
            builder = new ThesaurusTreeModelBuilder(context);
        }

        // construction du noeud root
        // il ne contient pas de context et ne sera passsss visible
        builder.buildRoot();

        getBuilder().buildChildren(context, rootNode);

        NavigationTreeModel model = builder.getModel();

        if (log.isInfoEnabled()) {
            builder.printModel(model.getRoot());
        }

        // save tree model in context
        setTreeModel(context, model);

        return model;
    }


    @Override
    public NavigationTreeHandler createTreeHandler(JAXXObject context) {

        if (log.isDebugEnabled()) {
            log.debug("create handler");
        }

        NavigationTreeHandler handler;

        handler = new NavigationTreeHandler(
                getPrefix(),
                context,
                Strategy.PER_NODE) {

            private static final long serialVersionUID = 1L;

            @Override
            protected NavigationTreeModel getNavigationTreeModel() {
                return getSafeTreeModel(getContext());
            }

            @Override
            protected void treateError(Exception e) {
                ErrorDialogUI.showError(e);
            }

            @Override
            public JAXXContext getContext() {
                return this.context;
            }

            @Override
            protected Component getCurrentUI() {
                return null;
            }

            @Override
            protected Component getUI(NavigationTreeNode node) {
                return (Component) context;
            }

            @Override
            protected boolean closeUI(Component component) throws Exception {
                return true;
            }

            @Override
            protected Component createUI(NavigationTreeNode node) throws Exception {
                return getHandler(context).getUI(context);
            }

            @Override
            protected void openUI(Component newUI, NavigationTreeNode node) throws Exception {
            }
        };
        // Selection multiple
        handler.setSelectionMode(NavigationTreeHandler.DISCONTIGUOUS_TREE_SELECTION);

        // save handler in ui context
        setTreeHandler(context, handler);

        // save tree in context
        // TODO : remove it
        if (context instanceof ThesaurusUI) {
            setTree(context, ((ThesaurusUI) context).getThesaurus());
        } else if (context instanceof AdminThesaurusUI) {
            setTree(context, ((AdminThesaurusUI) context).getThesaurus());
        } else {
            setTree(context, ((ThesaurusPathChooserUI) context).getThesaurus());
        }

        return handler;
    }

    public TreePath expendNode(JAXXContext context, NavigationTreeNode node) {

        javax.swing.tree.TreeNode[] treeNodes = node.getPath();
        TreePath path = new TreePath(treeNodes);
        getTree(context).expandPath(path);

        return path;
    }

    public void tryToSelect(JAXXContext context, Set<String> thesaurusId) {
        if (thesaurusId != null) {
            List<NavigationTreeNode> nodes = new ArrayList<NavigationTreeNode>();

            // Find recursivly
            for (String id : thesaurusId) {
                tryToSelect(id, getRootNode(), nodes);
            }

            // Select all
            JTree tree = getTree(context);
            TreePath[] paths = new TreePath[nodes.size()];
            int i = 0;
            for (NavigationTreeNode node : nodes) {

                // Expend
                TreePath path = expendNode(context, node);

                if (log.isDebugEnabled()) {
                    log.debug("Try to select path : " + path);
                }

                paths[i++] = path;

                if (log.isDebugEnabled()) {
                    log.debug("path " + paths[i - 1] + " for node " + node);
                }
            }
            // Select all
            tree.setSelectionPaths(paths);
        }
    }

    public void tryToSelect(String thesaurusId, NavigationTreeNode node, List<NavigationTreeNode> nodes) {
        if (thesaurusId != null) {
            if (thesaurusId.equals(getWikittyId(node))) {
                // Add to select list
                nodes.add(node);

            } else {
                // Find in child
                Enumeration<NavigationTreeNode> children = node.children();
                while (children.hasMoreElements()) {
                    NavigationTreeNode child = children.nextElement();
                    tryToSelect(thesaurusId, child, nodes);
                }
            }
        }
    }

    protected String getWikittyId(NavigationTreeNode node) {
        VradiThesaurusDTO thesaurus = (VradiThesaurusDTO) node.getBean();
        return thesaurus.getWikittyId();
    }

    public class ThesaurusTreeModelBuilder extends NavigationTreeModelBuilder {
        public ThesaurusTreeModelBuilder(JAXXContext context) {
            super(PATH_SEPARATOR, context, null, null);
        }

        // Return true if empty
        public boolean removeNodeFromParent(JAXXContext context, NavigationTreeNode node) {
            NavigationTreeModel treeModel = getTreeModel(context);
            treeModel.removeNodeFromParent(node);
            return treeModel.getRoot().getChildCount() == 0;
        }

        public NavigationTreeNode addThesaurusToSelected(JAXXContext context, VradiThesaurusDTO thesaurus) {
            NavigationTreeNode selectedNode = getSelectedNode(context);

            return addThesaurus(context, selectedNode, thesaurus);
        }

        public NavigationTreeNode addThesaurus(JAXXContext context, NavigationTreeNode parent, VradiThesaurusDTO thesaurus) {
            return addThesaurus(parent, thesaurus.getWikittyId());
        }

        public NavigationTreeNode addThesaurusAndChildrenRecursivly(NavigationTreeNode parent, VradiThesaurusDTO thesaurus) {
            NavigationTreeNode newNode = addThesaurus(parent, thesaurus.getWikittyId());
            for (VradiThesaurusDTO child : thesaurus.getChildren()){
                addThesaurusAndChildrenRecursivly(newNode, child);
            }
            return newNode;
        }

        public NavigationTreeNode addThesaurus(NavigationTreeNode parent, String thesaurusId) {
            NavigationTreeNode result = null;
            Decorator<VradiThesaurusDTO> tDecorator = getThesaurusDecorator();
            String path = "";
            if (parent == null) {
                parent = getRootNode();
                path = "..children[@wikittyId=\"" + thesaurusId + "\"]";
            } else {
                path = ".."  + PATH_SEPARATOR + "children[@wikittyId=\"" + thesaurusId + "\"]";
            }

            result = builder.build(parent, tDecorator, path, thesaurusId, null, null);
            if (log.isDebugEnabled()) {
                log.debug("addThesaurus : " + thesaurusId + " to parent : " +
                        (parent == null ? "null" : parent.getFullPath()) + " fullPath : " +
                        (result == null ? "result is null" : result.getFullPath()));
            }
            return result;
        }

        // Build the root node to display
        public void buildRoot() {
            if (log.isDebugEnabled()){
                log.debug("Building root : " + getRootName());
            }
            NavigationTreeNode invisibleRootNode = builder.buildEmptyRoot(getThesaurusRef(), getThesaurusRef().getName());

            // When it's the root one
            if (rootThesaurus.getParentThesaurus() == null){
                rootNode = invisibleRootNode;
                return;
            }
            
            // Add root node
            rootNode = builder.build(invisibleRootNode, getThesaurusDecorator(), rootThesaurus.getBuildPath(PATH_SEPARATOR), rootThesaurus.getWikittyId(), null, null);
        }

        public void buildChildren(JAXXContext context, NavigationTreeNode parentNode) {
            if (log.isDebugEnabled()) {
                log.debug("buildChildren of thesaurus " + parentNode.getFullPath());
            }

            VradiThesaurusDTO parent = (VradiThesaurusDTO) parentNode.getBean(context);
            if (parent != null){
                for (VradiThesaurusDTO child : parent.getChildren()) {
                    NavigationTreeNode childNode = addThesaurus(context, parentNode, child);

                    // Build childs
                    if (child.getWikittyId() != null) {
                        buildChildren(context, childNode);
                    }
                }
            }
        }

    }

    protected Decorator<VradiThesaurusDTO> getThesaurusDecorator() {
        Decorator<VradiThesaurusDTO> tDecorator = VradiContext.get().getDecorateurProvinder().getDecorator(VradiThesaurusDTO.class);
        return tDecorator;
    }
}
