/**
 * *##% observe-swing
 * Copyright (C) 2008 - 2009 IRD
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>. ##%*
 */
package com.jurismarches.vradi.ui;

import com.jurismarches.vradi.VradiConfig;
import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.ui.admin.AdminHandler;
import com.jurismarches.vradi.ui.admin.AdminPopupUI;
import com.jurismarches.vradi.ui.email.EmailHandler;
import com.jurismarches.vradi.ui.email.EmailPopupUI;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.DefaultApplicationContext.AutoLoad;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.editor.config.ConfigUI;
import jaxx.runtime.swing.editor.config.ConfigUIBuilder;
import jaxx.runtime.swing.editor.config.model.ConfigUIModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

import javax.swing.*;
import java.awt.*;
import java.net.URL;
import java.util.Locale;

/**
 * Le handler de l'ui principale.
 *
 * @author chemit
 * @see VradiMainUI
 */
@AutoLoad
public class VradiMainUIHandler { //implements JAXXHelp {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private Log log = LogFactory.getLog(VradiMainUIHandler.class);

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param fullscreen  flag pour indiquer si on doit ouvrir l'ui en model console (pleine ecran).
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public VradiMainUI initUI(JAXXContext rootContext, boolean fullscreen) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        // show main ui
        final VradiMainUI ui = new VradiMainUI(context);

        VradiContext.MAIN_UI_ENTRY_DEF.setContextValue(rootContext, ui);

        ErrorDialogUI.init(ui);

        // set fullscreen propery on main ui
        ui.getGraphicsConfiguration().getDevice().setFullScreenWindow(fullscreen ? ui : null);

        // init home screen
        OfferListHandler handler = new OfferListHandler();
        context.add(handler);

        changeContent(context, handler.initUI(ui, this));

        return ui;
    }

    public void changeContent(JAXXContext context, JComponent content) {
        VradiMainUI ui = getUI(context);
        ui.setContentPane(content);
//        ui.setVisible(false);
        ui.setVisible(true);
//        ((CardLayout) ui.getContent().getLayout()).show(wizardPanels, name);
    }

    public void edit(JAXXContext context, Form data) {
        OfferEditHandler handler = new OfferEditHandler();
        changeContent(VradiContext.get(), handler.initUI(context, data));
    }

    public void goToHome(JAXXContext context) {
        OfferListHandler handler = new OfferListHandler();
        changeContent(context, handler.initUI(context, this));
    }

    public void changeLanguage(VradiMainUI mainUI, Locale newLocale) {
        VradiConfig config = mainUI.getConfig();
        // sauvegarde de la nouvelle locale
        config.setLocale(newLocale);
        // chargement de la nouvelle locale dans le système i18n
        I18n.init(newLocale);
        // on recharge les decorateurs
        //DecoratorProvider decoratorProvider = mainUI.getContextValue(DecoratorProvider.class);
        //decoratorProvider.reloadCache();
        // on recharge l'ui
        reloadUI(VradiContext.get(), config.isFullScreen());
    }

    /**
     * Ferme l'application.
     *
     * @param ui l'ui principale de l'application
     */
    public void close(VradiMainUI ui) {
        log.info("Vradi quitting...");
        boolean canContinue = ensureModification(ui);
        if (!canContinue) {
            return;
        }
        try {
            ui.dispose();
        } finally {
            System.exit(0);
        }
    }

    /**
     * Méthode pour changer de mode d'affichage.
     * <p/>
     * Si <code>fullscreen</code> est à <code>true</code> alors on passe en
     * mode console (c'est à dire en mode plein écran exclusif), sinon on
     * passe en mode fenetré normal.
     *
     * @param ui         l'ui principale de l'application
     * @param fullscreen le nouvel état requis.
     */
    public void changeScreen(VradiMainUI ui, final boolean fullscreen) {
        boolean canContinue = ensureModification(ui);
        if (!canContinue) {
            return;
        }
        ui.getConfig().setFullscreen(fullscreen);

        reloadUI(VradiContext.get(), fullscreen);
    }

    protected void openAdminPopup(JAXXContext rootContext, int i) {

        AdminPopupUI ui = VradiContext.ADMIN_UI_ENTRY_DEF.getContextValue(rootContext);

        if (ui == null) {

            // create ui

            VradiMainUI mainUI = getUI(rootContext);

            //FIXME : this handler is stateless, so it could be put in rootContext ?
            //AdminHandler handler = new AdminHandler();

            JAXXInitialContext context = new JAXXInitialContext().add(rootContext).add(new AdminHandler());

            ui = new AdminPopupUI(context, mainUI);

            VradiContext.ADMIN_UI_ENTRY_DEF.setContextValue(rootContext, ui);
        }

        ui.showTab(i);

//        AdminPopupUI admin = handler.initUI(context, i);
    }

    public void showAdminClient(JAXXContext context) {
        openAdminPopup(context, 0);
    }

    public void showAdminGroup(JAXXContext context) {
        openAdminPopup(context, 1);
    }

    public void showAdminForm(JAXXContext context) {
        openAdminPopup(context, 2);
    }

    public void showAdminXmlStream(JAXXContext context) {
        openAdminPopup(context, 3);
    }

    protected void openEmailPopup(JAXXContext context, int i) {
        EmailHandler handler = new EmailHandler();
        EmailPopupUI admin = handler.initUI(context, i);
    }

    public void showEmailByClient(JAXXContext context) {
        openEmailPopup(context, 0);
    }

    public void showEmailByForm(JAXXContext context) {
        openEmailPopup(context, 1);
    }

    public void showConfig(JAXXContext context) {
        VradiMainUI ui = getUI(context);
        VradiConfig config = context.getContextValue(VradiConfig.class);
        ConfigUIModel model = new ConfigUIModel(config);

        // categorie repertoires
        model.addCategory(
                n_("vradi.config.category.directories"),
                n_("vradi.config.category.directories.description"),
                VradiConfig.Option.CONFIG_FILE);

        // others
        model.addCategory(
                n_("vradi.config.category.other"),
                n_("vradi.config.category.other.description"),
                VradiConfig.Option.FULL_SCREEN,
                VradiConfig.Option.LOCALE);

        ConfigUI configUI = ConfigUIBuilder.newConfigUI(context, model, "vradi.config.category.directories");

        ConfigUIBuilder.showConfigUI(configUI, ui, false);
    }

    //    @Override
    public void showHelp(JAXXContext context, String helpId) {
//        VradiMainUI mainUI = getUI(context);
//
//        ObserveHelpBroker helpBroker = context.getContextValue(ObserveHelpBroker.class);
//
//        if (helpId == null) {
//            helpId = helpBroker.getDefaultID();
//        }
//        log.debug("show help " + helpId);
//        mainUI.getHelp().setCurrentID(helpId);
//
//        mainUI.setContextValue(mainUI.getMode(), "oldMode");
//        mainUI.setMode(ObserveUIMode.HELP);
    }

    public void closeHelp(JAXXContext context) {
//        VradiMainUI mainUI = getUI(context);
//        ObserveUIMode oldMode = mainUI.getContextValue(ObserveUIMode.class, "oldMode");
//        if (oldMode == null) {
//            // on regarde si une base est chargee
//            ObserveDataContext dataContext = context.getContextValue(ObserveDataContext.class);
//            StorageService<?> mainStorage = dataContext.getStorage();
//
//            if (mainStorage == null) {
//                oldMode = ObserveUIMode.NO_DB;
//            } else {
//                oldMode = ObserveUIMode.DB;
//            }
//        }
//        mainUI.setMode(oldMode);
    }

    public void gotoSite(JAXXContext rootContext) {

        VradiConfig config = rootContext.getContextValue(VradiConfig.class);

        URL siteURL = config.getOptionAsURL("application.site.url");

//        UIHelper.displayInfo(rootContext, _("vradi.message.goto.site", siteURL));

        log.info("goto " + siteURL);
        if (Desktop.isDesktopSupported() && Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
            try {
                Desktop.getDesktop().browse(siteURL.toURI());
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                ErrorDialogUI.showError(ex);
            }
        }
    }

    public void showAbout(VradiMainUI ui) {
        AboutPanel about = new AboutPanel() {

            private static final long serialVersionUID = 1L;

//            @Override
//            public void buildTopPanel() {
//                topPanel.setLayout(new BorderLayout());
//                JLabel labelIcon;
//                Icon logoIcon;
//                logoIcon = Resource.getIcon("/icons/logo-OT_web.png");
//                labelIcon = new JLabel(logoIcon);
//                topPanel.add(labelIcon, BorderLayout.WEST);
//
//                logoIcon = Resource.getIcon("/icons/logo_ird.png");
//                labelIcon = new JLabel(logoIcon);
//                topPanel.add(labelIcon, BorderLayout.EAST);
//            }
        };

        about.setTitle(_("vradi.title.about"));
        about.setAboutText(_("vradi.about.message"));
        about.setBottomText(ui.getConfig().getCopyrightText());
//        about.setIconPath("/icons/logo-OT_web.png");
        about.setLicenseFile("META-INF/vradi-swing-LICENSE.txt");
        about.setThirdpartyFile("META-INF/vradi-swing-THIRD-PARTY.txt");
        about.init();
        about.showInDialog(ui, true);
    }

    /**
     * Permet de recharger l'ui principale et de changer de le mode d'affichage.
     *
     * @param rootContext le contexte applicatif
     * @param fullscreen  le type de fenetre à reouvrir
     */
    protected void reloadUI(VradiContext rootContext, boolean fullscreen) {

        // must remove all properties listener on config        
        rootContext.getContextValue(VradiConfig.class).removeJaxxPropertyChangeListener();

        // scan main ui
        VradiMainUI ui = getUI(rootContext);

        if (ui != null) {

            ErrorDialogUI.init(null);

            VradiContext.MAIN_UI_ENTRY_DEF.removeContextValue(rootContext);
            VradiContext.ADMIN_UI_ENTRY_DEF.removeContextValue(rootContext);

            ui.dispose();

            ui.setVisible(false);
        }

        ui = initUI(rootContext, fullscreen);

        // show ui
        ui.setVisible(true);
    }

    /**
     * Test if there is some modification on screen,
     *
     * @param rootContext the context
     * @return <code>true</code> if no more modification is detected
     * @throws IllegalArgumentException if rootContext is null
     */
    protected boolean ensureModification(JAXXContext rootContext) throws IllegalArgumentException {
        if (rootContext == null) {
            throw new IllegalArgumentException("rootContext can not be null");
        }
        VradiMainUI ui = getUI(rootContext);
        if (ui == null) {
            // no ui, so no modification
            return true;
        }
        // check ui is not modified
        return true;
    }

    VradiMainUI getUI(JAXXContext context) {
        if (context instanceof VradiMainUI) {
            return (VradiMainUI) context;
        }
        VradiMainUI ui = VradiContext.MAIN_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }

}
