package com.jurismarches.vradi;

import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.Properties;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;

import static org.nuiton.i18n.I18n._;

@jaxx.runtime.context.DefaultApplicationContext.AutoLoad
public class VradiConfig extends org.nuiton.util.ApplicationConfig {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(VradiConfig.class);
    /**
     * le fichier de configuration de l'application avec les informations sur
     * le projet (version, license,...) et la configuration des ui (icons, ...)
     */
    public static final String APPLICATION_PROPERTIES = "/vradi.properties";
    /**
     * le nom du repertoire ou sont les donnees de l'application
     */
    public static final String USER_DIRECTORY_FILENAME = ".vradi";
    /**
     * le repertoire utilisateur de l'application
     */
    protected static File vradiUserDirectory;

    public static File getVradiUserDirectory() {
        if (vradiUserDirectory == null) {
            vradiUserDirectory = new File(new File(VradiConfig.getUserHome()), USER_DIRECTORY_FILENAME);
        }
        return vradiUserDirectory;
    }

    public File getUserFile(){
        return new File(getUserHome() + File.separator + "." + getConfigFileName());
    }

    /**
     * un drapeau pour savoir si on est en mode pleine écran
     */
    protected boolean fullscreen;
    /**
     * un drepeau pour savoir s'il faut lancer l'interface graphique.
     * Cette valeur peut être programmées lors des actions.
     */
    private boolean displayMainUI = true;
    /**
     * drapeau pour savoir si on peut utiliser des ui dans l'environnement.
     *
     * Par defaut, on suppose qu'on peut utiliser l'environnement graphique
     * et si on désactive explicitement ou si pas d'environnement graphique trouvé.
     */
    private boolean canUseUI = true;

    public VradiConfig() {

        setConfigFileName(Option.CONFIG_FILE.defaultValue);

        // chargement de la configuration interne

        InputStream stream = getClass().getResourceAsStream(APPLICATION_PROPERTIES);

        Properties p = new Properties();
        try {
            p.load(stream);
            for (Object k : p.keySet()) {
                String key = k + "";
                Object value = p.get(k);
                if (log.isDebugEnabled()) {
                    log.debug("install properties " + k + " : " + value);
                }
                setDefaultOption(key, "" + value);
            }
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }

        for (Option o : Option.values()) {
            setDefaultOption(o.key, o.defaultValue);
        }

        // on supprime le stamp de snapshot s'il existe
        String sVersion = VersionUtil.removeSnapshot(getOption("application.version"));
        Version version = VersionUtil.valueOf(sVersion);
        setDefaultOption("version", version.getVersion());

        // la version de la base est sans classifier (pas de alpha, ou rc,...)
        Version dbVersion = null;
        if (version.hasClassifier()) {
            dbVersion = VersionUtil.removeClassifier(version);
        } else {
            dbVersion = VersionUtil.valueOf(version.toString());
        }

        setDefaultOption("db.version", dbVersion.getVersion());

        // initialisation des répertoires
        FileUtil.setCurrentDirectory(getVradiUserDirectory());
        getVradiUserDirectory().mkdirs();

        initActions();
    }

    public void initActions() {
        for (Action a : Action.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);
            }
        }
    }

    public String getCopyrightText() {
        return "Version " + getVersion() + " JurisMarches @ 2009";
    }

    /**
     * La version du logiciel constitue de l.d.r
     * <li>l: le numero de version du logiciel
     * <li>d: le numero de version du schema de la base de donnees
     * <li>r: le numero de version de de l'interface graphique ou autre modif mineur
     * <p/>
     * lors de l'increment de l, d et r sont remis a 0
     * lors de l'increment de d et r sont remis a 0
     * lors de l'increment de r est remis a 0
     * <p/>
     * Un changement d'UI ne modifie jamais le numero de version de database
     * Un changement de schema de base pour lequel on ne peut pas faire de
     * migration de donnees demande automatiquement un changement de version
     * d'application.
     *
     * @return la version de l'application.
     */
    public Version getVersion() {
        Version option = getOption(Version.class, "version");
        return option;
    }

    /**
     *
     * @return la version de la base
     */
    public Version getDbVersion() {
        Version option = getOption(Version.class, "db.version");
        return option;
    }

    public boolean isFullScreen() {
        Boolean result = getOptionAsBoolean(Option.FULL_SCREEN.key);
        return result != null && result;
    }

    public boolean isDisplayMainUI() {
        return displayMainUI;
    }

    public boolean isCanUseUI() {
        return canUseUI;
    }

    public Locale getLocale() {
        Locale result = getOption(Locale.class, Option.LOCALE.key);
        return result;
    }

    public void setFullscreen(boolean fullscreen) {
        boolean oldValue = isFullScreen();
        setOption(Option.FULL_SCREEN.key, fullscreen + "");
        saveForUser();
        firePropertyChange("fullscreen", oldValue, fullscreen);
    }

    public void setDisplayMainUI(boolean b) {
        displayMainUI = b;
    }

    public void setCanUseUI(boolean canUseUI) {
        this.canUseUI = canUseUI;
        if (!canUseUI) {
            // on ne pourra pas lancer l'ui principale
            setDisplayMainUI(false);
        }
    }

    public void setLocale(Locale newLocale) {
        setOption(Option.LOCALE.key, newLocale.toString());
        saveForUser();
        firePropertyChange("locale", null, newLocale);
    }
    public static final String[] DEFAULT_JAXX_PCS = {
        "showNumberEditorButton", "autoPopupNumberEditor", "fullScreen", "locale"
    };

    public void removeJaxxPropertyChangeListener() {
        PropertyChangeListener[] toRemove = jaxx.runtime.Util.findJaxxPropertyChangeListener(DEFAULT_JAXX_PCS, getPropertyChangeListeners());
        if (toRemove == null || toRemove.length == 0) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("before remove : " + getPropertyChangeListeners().length);
            log.debug("toRemove : " + toRemove.length);
        }
        for (PropertyChangeListener listener : toRemove) {
            removePropertyChangeListener(listener);
        }
        if (log.isDebugEnabled()) {
            log.debug("after remove : " + getPropertyChangeListeners().length);
        }
    }

    //////////////////////////////////////////////////
    // Toutes les options disponibles
    //////////////////////////////////////////////////
    public static enum Option implements OptionDef {

        CONFIG_FILE(CONFIG_FILE_NAME, _("vradi.config.configFileName.description"), "vradi-config", String.class, true, true),
        // ui config
        FULL_SCREEN("ui.fullscreen", _("vradi.config.ui.fullscreen"), "false", Boolean.class, false, false),
        LOCALE("ui.locale", _("vradi.config.ui.locale"), Locale.FRANCE.toString(), Locale.class, false, false);
        public final String key;
        public final String description;
        public final String defaultValue;
        public final Class<?> type;
        public final boolean _transient;
        public final boolean _final;

        private Option(String key, String description, String defaultValue, Class<?> type, boolean _transient, boolean _final) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this._final = _final;
            this._transient = _transient;
        }

        @Override
        public boolean isFinal() {
            return _final;
        }

        @Override
        public boolean isTransient() {
            return _transient;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public String getDescription() {
            return description;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public Class<?> getType() {
            return type;
        }
    }

    public static enum Step {

        AfterInit, BeforeExit
    }

    public static enum Action {

        HELP(_("vradi.action.commandline.help"), VradiAction.class.getName() + "#help", "-h", "--help"),
        HELP_UI(_("vradi.action.commandline.help.ui"), VradiAction.class.getName() + "#helpUI", "--help-ui"),
        NO_MAIN_UI(_("vradi.action.commandline.disable.main.ui"), VradiAction.class.getName() + "#disableMainUI", "-n", "--no-main"),
        CONFIGURE_UI(_("vradi.action.commandline.configure.ui"), VradiAction.class.getName() + "#configure", "--configure");
        public String description;
        public String action;
        public String[] aliases;

        private Action(String description, String action, String... aliases) {
            this.description = description;
            this.action = action;
            this.aliases = aliases;
        }
    }
}
