/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: QueryBean.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://tchemit@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3/vradi-services/src/main/java/com/jurismarches/vradi/beans/QueryBean.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.beans;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;

import au.com.bytecode.opencsv.CSVParser;
import au.com.bytecode.opencsv.CSVWriter;

/**
 * Query bean.
 * 
 * Used to manipulate User, Group, Client queries, parse from CSV ad encode to CSV.
 *
 * @author schorlet
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 * @since 16 mars 2010 14:25:12
 * 
 * @see CSVWriter
 * @see CSVParser
 */
public final class QueryBean implements Serializable {

    private static final long serialVersionUID = 2L;

    protected final String queryLine;

    protected final String[] queryArray;

    protected final String id;

    public QueryBean() {
        this.id = null;
        this.queryLine = "";
        this.queryArray = new String[3];
    }

    /**
     * Decode the specified <code>queryLine</code>. The queryLine must be CSV formatted.
     * 
     * @param queryLine the formatted query.
     * @throws IOException if <code>queryLine</code> is badly formatted.
     */
    public QueryBean(String queryLine) throws IOException {
        this(queryLine, null);
    }

    public QueryBean(String queryLine, String id) throws IOException {
        this.id = id;
        this.queryLine = queryLine;
        this.queryArray = decode(queryLine, 3);
    }

    /**
     * Creates a CSV formatted query from the specified <code>queryArray</code>.
     * @param queryArray the string composing the query
     */
    private QueryBean(String[] queryArray, String id) {
        if (queryArray.length < 3) {
            throw new IllegalArgumentException("queryArray length is less than 3");
        }

        this.id = id;
        this.queryArray = queryArray.clone();
        this.queryLine = encode(queryArray);
    }

    public String getQueryLine() {
        return queryLine;
    }

    public String getQuery() {
        return queryArray[0];
    }

    public String getName() {
        return queryArray[1];
    }

    public String getDescription() {
        return queryArray[2];
    }

    public String getCSVQuery() {
        return queryLine;
    }

    public String[] toArray() {
        return queryArray;
    }

    public QueryBean setQuery(String value) {
        return create(value, 0);
    }

    public QueryBean setName(String value) {
        return create(value, 1);
    }

    public QueryBean setDescription(String value) {
        return create(value, 2);
    }

    private QueryBean create(String value, int pos) {
        String[] newArray = queryArray.clone();
        newArray[pos] = value;
        return new QueryBean(newArray, id);
    }

    /**
     * Parse the specified line and returns a string array of the specified size.
     * 
     * @param line the string to parse
     * @return the comma-tokenized list of elements
     * @throws IOException
     */
    static String[] decode(String line, int size) throws IOException {
        CSVParser csvParser = new CSVParser();
        String[] parseLine = csvParser.parseLine(line);
        
        String[] result = new String[size];
        for (int i = 0; i < parseLine.length; i++) {
            result[i] = parseLine[i];
        }
        
        return result;
    }

    /**
     * Escape each columns of the specified array and return a valid CSV line.
     * 
     * @param columns columns composing the CVS string
     * @return a CSV formatted string
     */
    static String encode(String[] columns) {
        StringWriter writer = new StringWriter();
        // creates a CSVWriter with all default options but with no line end
        CSVWriter csvwriter = new CSVWriter(writer, CSVWriter.DEFAULT_SEPARATOR,
                CSVWriter.DEFAULT_QUOTE_CHARACTER, CSVWriter.DEFAULT_ESCAPE_CHARACTER, "");
        csvwriter.writeNext(columns);
        return writer.toString();
    }

    /**
     * Additionnal attribute. It does not belongs to the query definition.
     * 
     * @return query id
     */
    public String getId() {
        return id;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((id == null) ? 0 : id.hashCode());
        result = prime * result
                + ((queryLine == null) ? 0 : queryLine.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }

        QueryBean other = (QueryBean) obj;
        if (id == null) {
            if (other.id != null) {
                return false;
            }
        } else if (!id.equals(other.id)) {
            return false;
        }

        if (queryLine == null) {
            if (other.queryLine != null) {
                return false;
            }
        } else if (!queryLine.equals(other.queryLine)) {
            return false;
        }

        return true;
    }
}
