/* *##%
 * Vradi :: Services
 * Copyright (C) 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services.managers;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jurismarches.vradi.services.Configuration;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.ooo.SingletonOOo;
import com.sun.star.beans.PropertyValue;
import com.sun.star.beans.XPropertySet;
import com.sun.star.container.XEnumerationAccess;
import com.sun.star.container.XNameAccess;
import com.sun.star.frame.XComponentLoader;
import com.sun.star.frame.XStorable;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XMultiServiceFactory;
import com.sun.star.text.ControlCharacter;
import com.sun.star.text.TextContentAnchorType;
import com.sun.star.text.XText;
import com.sun.star.text.XTextContent;
import com.sun.star.text.XTextDocument;
import com.sun.star.text.XTextFieldsSupplier;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XComponentContext;
import com.sun.star.util.XRefreshable;

/**
 * Template manager.
 * 
 * Handle:
 * <ul>
 *  <li>OpenOffice.org PDF creation</li>
 * </ul>
 * 
 * @author chatellier
 */
public class TemplateManager {
    private static final Log log = LogFactory.getLog(TemplateManager.class);

    protected XComponentContext context = null;
    protected XComponentLoader loader = null;
    protected XComponent document = null;

    public TemplateManager(String templateDoc) throws VradiException {
        document = createDoc(templateDoc);
    }

    public void generateDoc(String targetFileName,
            Map<String, Object> myValues, File... images) throws VradiException {
        fillFields(myValues);
        if (images != null) {
            for (File image : images) {
                insertImageInEndOfDocument(image);
            }
        }
        storeDocComponent(document, targetFileName);
    }

    /**
     * Créer un document
     *
     * @param templateFile Le template du document
     * @return Le document construit
     * 
     * @throws VradiException if document can't be created
     */
    public XComponent createDoc(String templateFile) throws VradiException {
        try {
            // Get Context et service manager de SingletonOOo
            if (context == null) {
                String oOoExecFolder = Configuration.getInstance()
                        .getOpenOfficeExecDir();
                context = SingletonOOo.GetInstance(oOoExecFolder).getXContext();
                loader = SingletonOOo.GetInstance(oOoExecFolder).getLoader();
            }

            // Propriété générale du document (voir com.sun.star.document.MediaDescriptor).
            List<PropertyValue> props = new ArrayList<PropertyValue>();

            // Autorise l'utilisation d'un Template.
            PropertyValue p1 = new PropertyValue();
            p1.Name = "AsTemplate";
            p1.Value = new Boolean(true);
            props.add(p1);

            // Rend le document invisible.
            PropertyValue p2 = new PropertyValue();
            p2.Name = "Hidden";
            p2.Value = new Boolean(true);
            props.add(p2);

            PropertyValue[] properties = new PropertyValue[props.size()];
            props.toArray(properties);

            String templateFileURL = filePathToURL(templateFile);
            document = loader.loadComponentFromURL(templateFileURL, // URL du temlpate
                    "_blank", // Nom de la fenetre (_blank en cré une nouvelle).
                    0, // On ne cherche pas de flag.
                    properties); // Propriétées.

            return document;

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't create document", e);
            }
            throw new VradiException("Can't create document", e);
        }
    }

    /**
     * Sauver un doc donné a l'url donnée au format pdf.
     *
     * @param xDoc     Le document a sauvegarder (XComponent)
     * @param storeUrl Url complete du fichier (String)
     * 
     * @throws VradiException if pdf can't be exported
     */
    public synchronized void storeDocComponent(XComponent xDoc, String storeUrl)
            throws VradiException {
        try {
            XStorable xStorable = (XStorable) UnoRuntime.queryInterface(
                    XStorable.class, xDoc);
            storeUrl = filePathToURL(storeUrl);
            
            if (log.isDebugEnabled()) {
                log.debug("Storing pdf file to " + storeUrl);
            }

            PropertyValue[] storeProps = new PropertyValue[2];
            storeProps[0] = new PropertyValue();
            storeProps[0].Name = "FilterName";
            storeProps[0].Value = "writer_pdf_Export";
            storeProps[1] = new PropertyValue();
            storeProps[1].Name = "Overwrite";
            storeProps[1].Value = new Boolean(true);
            xStorable.storeToURL(storeUrl, storeProps);
            
            // TODO EC20100427 les lignes suivantes ont l'air
            // de fermer un truc qu'il ne faut pas, le les génération suivantes
            // echouent

            //xDoc.dispose();
            //XComponent xComp = (XComponent) UnoRuntime.queryInterface(
                    //XComponent.class, xStorable);
            //xComp.dispose();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export template as PDF", eee);
            }
            throw new VradiException("Can't export template as PDF", eee);
        }
    }

    /**
     * Converti si besoin l'url.
     */
    protected String filePathToURL(String file) {
        File f = new File(file);
        
        // file:/// (with 3 /) is mandatory on windows
        StringBuffer sb = new StringBuffer("file:///");
        try {
            sb.append(f.getCanonicalPath().replace('\\', '/'));
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get fi canonical path", e);
            }
        }
        return sb.toString();
    }

    /**
     * Remplir les champs de mailing.
     * 
     * @param myValues value to replace into template fields
     * 
     * @throws VradiException if values can't be replaced
     */
    protected void fillFields(Map<String, Object> myValues) throws VradiException {
        
        try {
            XTextFieldsSupplier xTextFieldsSupplier = (XTextFieldsSupplier) UnoRuntime
                    .queryInterface(XTextFieldsSupplier.class, document);
    
            XNameAccess xNamedFieldMasters = xTextFieldsSupplier.getTextFieldMasters();
            XEnumerationAccess xEnumeratedFields = xTextFieldsSupplier.getTextFields();
            String names[] = xNamedFieldMasters.getElementNames();
    
            String[] fieldNames = new String[names.length];
            Object[] fieldMasters = new Object[names.length];
            XPropertySet[] xPropertySets = new XPropertySet[names.length];
            int fieldCount = 0;
    
            for (int i = 0; i < names.length; i++) {
                String user = names[i].length() > 34 ? names[i].substring(30,
                        34) : null;
                String wordUser = names[i].length() > 38 ? names[i].substring(
                        30, 38) : null;
    
                if (user != null && user.equalsIgnoreCase("User")) {
                    fieldNames[fieldCount] = names[i].substring(35, names[i]
                            .length());
                    fieldMasters[fieldCount] = xNamedFieldMasters
                            .getByName(names[i]);
                    xPropertySets[fieldCount] = (XPropertySet) UnoRuntime
                            .queryInterface(XPropertySet.class,
                                    fieldMasters[fieldCount]);
                    fieldCount++;
    
                } else if (wordUser != null
                        && wordUser.equalsIgnoreCase("database")) {
                    fieldNames[fieldCount] = names[i].substring(38, names[i]
                            .length());
                    fieldMasters[fieldCount] = xNamedFieldMasters
                            .getByName(names[i]);
                    xPropertySets[fieldCount] = (XPropertySet) UnoRuntime
                            .queryInterface(XPropertySet.class,
                                    fieldMasters[fieldCount]);
                    fieldCount++;
                }
            }
    
            XRefreshable xRefreshable = (XRefreshable) UnoRuntime
                    .queryInterface(XRefreshable.class, xEnumeratedFields);
    
            for (int i = 0; i < fieldCount; i++) {
                if (log.isDebugEnabled()) {
                    log.debug("Replacing " + fieldNames[i] + " with "
                            + myValues.get(fieldNames[i]));
                }
    
                xPropertySets[i].setPropertyValue("Content", myValues
                        .get(fieldNames[i]));
                xRefreshable.refresh();
                if (log.isDebugEnabled() && myValues.get(fieldNames[i]) == null) {
                    log.debug("Field " + fieldNames[i] + " could not be found");
                }
            }
        }
        catch (Exception eee) {
            throw new VradiException("Can't replace template fields", eee);
        }
    }

    /**
     * Inserts an image to the end of the document.
     * 
     * @param image the file containing the image to insert
     * @throws VradiException
     */
    public void insertImageInEndOfDocument(File image) throws VradiException {

        try {
            XTextDocument xTextDoc = (XTextDocument) UnoRuntime.queryInterface(
                    XTextDocument.class, document);

            // Querying for the interface XMultiServiceFactory on the XTextDocument
            XMultiServiceFactory xMSFDoc = (XMultiServiceFactory) UnoRuntime
                    .queryInterface(XMultiServiceFactory.class, xTextDoc);

            // Creating the service GraphicObject
            Object oGraphic = null;
            oGraphic = xMSFDoc
                    .createInstance("com.sun.star.text.TextGraphicObject");

            // Querying for the interface XTextContent on the GraphicObject
            XTextContent xTextContent = (XTextContent) UnoRuntime
                    .queryInterface(XTextContent.class, oGraphic);

            // Getting the text
            XText xText = xTextDoc.getText();
            xText.insertControlCharacter(xText.getEnd(),
                    ControlCharacter.APPEND_PARAGRAPH, false);

            // Inserting the content
            xText.insertTextContent(xText.getEnd(), xTextContent, false);

            // Querying for the interface XPropertySet on the graphic object
            XPropertySet xPropSet = (XPropertySet) UnoRuntime.queryInterface(
                    XPropertySet.class, oGraphic);

            // Setting the anchor type
            xPropSet.setPropertyValue("AnchorType",
                    TextContentAnchorType.AT_PARAGRAPH);

            // Setting the graphic url
            // FIXME EC20100428 : replace this hard coded images
            xPropSet.setPropertyValue("GraphicURL",
                    "file:///home/morin/Images/cyclope-veja-fixed-gear.jpg");

            //Setting the top margin
            xPropSet.setPropertyValue("TopMargin", 1500);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't insert image", eee);
            }
            throw new VradiException("Can't insert image", eee);
        }

    }

    /**
     * Gets the user fields or database fields of the ooo document.
     * 
     * @return an array of String containing the field names
     */
    public String[] getDocumentFields() {
        XTextFieldsSupplier xTextFieldsSupplier = (XTextFieldsSupplier) UnoRuntime
                .queryInterface(XTextFieldsSupplier.class, document);
        XNameAccess xNamedFieldMasters = xTextFieldsSupplier.getTextFieldMasters();
        
        // EC-20100503 unused variable
        //XEnumerationAccess xEnumeratedFields = xTextFieldsSupplier.getTextFields();

        String names[] = xNamedFieldMasters.getElementNames();

        List<String> fieldNames = new ArrayList<String>();

        for (int i = 0; i < names.length; i++) {
            String user = names[i].length() > 34 ? names[i].substring(30, 34)
                    : null;
            String wordUser = names[i].length() > 38 ? names[i].substring(30,
                    38) : null;

            if (user.equalsIgnoreCase("User")) {
                fieldNames.add(names[i].substring(35, names[i].length()));
            } else if (wordUser.equalsIgnoreCase("database")) {
                fieldNames.add(names[i].substring(38, names[i].length()));
            }
        }
        return fieldNames.toArray(new String[fieldNames.size()]);
    }
}
