/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services;

import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.dto.VradiCartographyDTO;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.services.dto.VradiUserDTO;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * VradiStorageService.
 *
 * @author morin
 * @version $Revision: 890 $ $Date: 2010-05-10 17:16:48 +0200 (lun., 10 mai 2010) $
 */
public interface VradiStorageService {

    /**
     * Returns a <code>BusinessEntity</code> of the specified <code>id</code> and <code>clazz</code>.
     *
     * @param id    the id of the entity
     * @param clazz the class of the entity
     * @return a <code>BusinessEntity</code>
     * @throws VradiException
     */
    <E extends BusinessEntity> E getEntity(String id, Class<E> clazz) throws VradiException;

    /**
     * Returns the <code>BusinessEntity</code> of the specified <code>ids</code> and <code>clazz</code>.
     *
     * @param ids    the id of the entities
     * @param clazz the class of the entity
     * @return a <code>BusinessEntity</code>
     * @throws VradiException
     */
     <E extends BusinessEntity> List<E> getEntities(List<String> ids, Class<E> clazz) throws VradiException;

    /**
     * Updates the specified <code>BusinessEntity</code>.
     *
     * @param entity the entity to update
     * @return the updated entity
     * @throws VradiException
     */
    <E extends BusinessEntity> E updateEntity(E entity) throws VradiException;

    /**
     * Updates the specified <code>BusinessEntity</code>.
     *
     * @param entities list of the entity to update
     * @return the updated entities
     * @throws VradiException
     */
    public <E extends BusinessEntity> E[] updateEntities(E ... entities) throws VradiException;

    /**
     * Updates the specified <code>BusinessEntity</code>.
     *
     * @param entities list of the entity to update
     * @return the updated entities
     * @throws VradiException
     */
    <E extends BusinessEntity> List<E> updateEntities(List<E> entities) throws VradiException;

    /**
     * Delete the specified <code>BusinessEntity</code> by id.
     *
     * @param id of the entity the entity to delete
     * @throws VradiException
     */
    void deleteEntity(String id) throws VradiException;

    /**
     * Delete the specified <code>BusinessEntity</code>.
     *
     * @param entity the entity to delete
     * @throws VradiException
     */
    void deleteEntity(BusinessEntity entity) throws VradiException;

    /**
     * Returns a <code>Client</code> with the specified <code>clientId</code>.
     *
     * @param clientId the id of the client
     * @return a <code>Client</code>
     * @throws VradiException
     */
    Client getClient(String clientId) throws VradiException;

    /**
     * Returns a <code>User</code> with the specified <code>userId</code>.
     *
     * @param userId the id of the user
     * @return a <code>User</code>
     * @throws VradiException
     */
    User getUser(String userId) throws VradiException;

    /**
     * Returns a <code>Group</code> with the specified <code>groupId</code>.
     *
     * @param groupId the id of the group
     * @return a <code>Group</code>
     * @throws VradiException
     */
    Group getGroup(String groupId) throws VradiException;
    
    /**
     * Return the last part of the query history file,
     * eg. queryHistory/ff4bb261-d6b2-42b3-be24-ab14e95cb244.rss.
     * 
     * The protocol, host and context of the file depends on the relation between
     * modules vradi-swing and vradi-services and whether services are invoked
     * locally or remotely.
     * 
     * @param id the id of the queryMaker
     */
    String getQueryHistoryFile(String id);
    
    /**
     * Returns a list of <code>Users</code> belonging to the specified <code>groupId</code>.
     *
     * @param groupId the id of the group
     * @return a list of <code>Users</code>
     * @throws VradiException
     */
    List<User> getGroupUsers(String groupId) throws VradiException;

    /**
     * Returns a list of <code>Users</code> belonging to the specified <code>clientId</code>.
     *
     * @param clientId the id of the client
     * @return a list of <code>Users</code>
     * @throws VradiException
     * 
     * @deprecated since 20100507 unused
     */
    @Deprecated
    List<User> getClientUsers(String clientId) throws VradiException;

    /**
     * Returns a list of <code>Clients</code> belonging to the specified <code>groupId</code>.
     *
     * @param groupId the id of the group
     * @return a list of <code>Clients</code>
     * @throws VradiException
     */
    List<Client> getGroupClients(String groupId) throws VradiException;

    /**
     * Returns a <code>Client</code> whose specified <code>userId</code> belongs.
     *
     * @param userId the id of the user
     * @return a <code>Client</code>
     * @throws VradiException
     */
    Client getClientByUserId(String userId) throws VradiException;

    /**
     * Returns all the clients.
     *
     * @return all the clients
     * @throws VradiException
     */
    List<Client> getAllClients() throws VradiException;

    /**
     * Returns all the groups.
     *
     * @return all the groups
     * @throws VradiException
     */
    List<Group> getAllGroups() throws VradiException;

    /**
     * Update a list of <code>XmlFieldBinding</code>.
     *
     * @param bindings the list of XmlFieldBinding to update
     * @return the list of updated XmlFieldBinding
     */
    List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings) throws VradiException;

    XmlStream updateXmlStream(XmlStream xmlStream, List<XmlFieldBinding> bindings) throws VradiException;

    /**
     * Creates or Update a form type.
     *
     * @param extension the extension designing the form type
     * @return a <code>WikittyExtension</code>
     */
    WikittyExtension updateFormType(WikittyExtension extension) throws VradiException;

    /**
     * Creates or Update a form type.
     *
     * @param name      the name of the form type
     * @param fields    the fields of the form type
     * @param requires  the extension requred by the extension to update
     * @param tagValues the map containing the tags and their values
     * @return a <code>WikittyExtension</code>
     */
    WikittyExtension updateFormType2(String name, Map<java.lang.String, org.sharengo.wikitty.FieldType> fields, String requires, Map<java.lang.String, java.lang.String> tagValues) throws VradiException;

    /**
     * Creates or Update the specified <code>form</code>.
     *
     * @param form the form to create or update
     * @return the stored <code>Form</code>
     */
    Form updateForm(Form form) throws VradiException;

    /**
     * Returns a <code>XmlStream</code> with the specified <code>xmlStreamId</code>.
     *
     * @param xmlStreamId the id of the XmlStream
     * @return a <code>XmlStream</code>
     * @throws VradiException
     */
    XmlStream getXmlStream(String xmlStreamId) throws VradiException;

    /**
     * Returns a list of <code>XmlFieldBinding</code> belonging to the specified <code>xmlStream</code>.
     *
     * @param xmlStream the <code>XmlStream</code>
     * @return a list of <code>XmlFieldBinding</code>
     */
    List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) throws VradiException;

    /**
     * Returns a <code>XmlFieldBinding</code> with the specified <code>xmlFieldBindingId</code>.
     *
     * @param xmlFieldBindingId the id of the XmlFieldBinding
     * @return a <code>XmlFieldBinding</code>
     */
    XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) throws VradiException;

    /**
     * Returns a <code>TreeNodeImpl</code> with the specified <code>thesaurusId</code>.
     *
     * @param thesaurusId the id of the thesaurus
     * @return a <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    TreeNodeImpl getThesaurus(String thesaurusId) throws VradiException;

    /**
     * Returns the root Thesaurus Node.
     * 
     * @return a <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    TreeNodeImpl getRootThesaurus() throws VradiException;

    /**
     * Returns all children of the specified <code>thesaurusId</code>.
     * 
     * @return all list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    List<TreeNodeImpl> getChildrenThesaurus(String thesaurusId) throws VradiException;

    /**
     * Finds fields of a form type.
     *
     * @param name the name of the form type
     * @return a map containing the names of the fields and their type
     */
    Map<String, FieldType> getFormTypeFields(String name) throws VradiException;

    /**
     * Returns a form type with the specified <code>name</code>.
     *
     * @param name the name of the form type
     * @return a <code>WikittyExtension</code>
     */
    WikittyExtension getFormType(String name) throws VradiException;

    /**
     * Returns a <code>Form</code> with the specified <code>formId</code>.
     *
     * @param formId the id of the form
     * @return the form whose id is id
     */
    Form getForm(String formId) throws VradiException;

    /**
     * Returns all <code>XmlStreams</code>.
     * 
     * @return a list of <code>XmlStream</code>
     */
    List<XmlStream> getAllXmlStreams() throws VradiException;

    /**
     * Returns all <code>TreeNodeImpl</code> considered as thesaurus nodes.
     * 
     * @return a list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    List<TreeNodeImpl> getAllThesaurus() throws VradiException;

    /**
     * Returns all Form Types.
     *
     * @return a list of <code>WikittyExtension</code>
     */
    List<WikittyExtension> getAllFormTypes() throws VradiException;

    /**
     * Returns all <code>Forms</code>.
     *
     * @return a list of <code>Form</code>
     */
    List<Form> getAllForms() throws VradiException;

    /**
     * Searches for forms with the specified filters.
     * 
     * The <code>beginDate</code> and <code>endDate</code> specifies a
     * date range filter which is applied on <code>dateType</code> field.
     * The date range filter is applied only if the three parameters are not null.
     * 
     * @param query     the query filter
     * @param extension the extension filter
     * @param dateType  the date field. Must be fully qualified (extension.field)
     * @param beginDate the begin date filter
     * @param endDate   the end date filter
     * @param thesaurus the thesaurus filter
     * @param statusIds  the ids of the status filters
     * @param formPageDTO ...
     * @return VradiFormPageDTO
     * @throws UnsupportedQueryException
     */
    VradiFormPageDTO findForms2(String query, WikittyExtension extension, String dateType,
                   Date beginDate, Date endDate, java.util.List<String>[] thesaurus,
                   String[] statusIds, final VradiFormPageDTO formPageDTO)
            throws VradiException, UnsupportedQueryException;

    /**
     * Computes the number of forms associated with every thesaurus node.
     * 
     * @param query     the query filter
     * @param extension the extension filter
     * @param dateType  the date field. Must be fully qualified (extension.field)
     * @param beginDate the begin date filter
     * @param endDate   the end date filter
     * @param thesaurus the thesaurus filter
     * @param statusIds  the ids of the status filters
     * @return a <code>VradiCartographyDTO</code> containing forms result and a map of form count indexed by thesaurus nodes
     * @throws VradiException
     * @throws UnsupportedQueryException
     */
    VradiCartographyDTO getThesaurusCartography(String query,
            WikittyExtension extension, String dateType, Date beginDate,
            Date endDate, java.util.List<String>[] thesaurus, String[] statusIds)
            throws VradiException, UnsupportedQueryException;

    /**
     * Searches for forms with the specified query.
     *
     * @param query the query to filter the forms
     * @param formPageDTO ...
     * @return VradiFormPageDTO
     * @throws UnsupportedQueryException
     */
    VradiFormPageDTO findForms(String query, final VradiFormPageDTO formPageDTO) throws VradiException, UnsupportedQueryException;

    /**
     * Deletes a <code>Form</code> by the specified <code>formId</code>.
     *
     * @param formId the id of the Form to delete
     */
    void deleteForm(String formId) throws VradiException;

    /**
     * Binds all queryMaker with the forms found by their queries
     * and not already sent.
     */
    void bindForms() throws VradiException;

    /**
     * Create new Sending
     *
     * @param session
     * @param queryMaker
     * @param formsToBind
     * @return
     * @throws VradiException
     */
    List<Sending> createSending(Session session, QueryMaker queryMaker, List<Form> formsToBind) throws VradiException;

    /**
     * Get all session by date
     *
     * @param sessionDate           Date concerned
     * @return                      a list of VradiSessionDTO
     * @throws VradiException
     */
    List<Session> getSessions(Date sessionDate) throws VradiException;

    /**
     * Creates and store forms from an <code>XmlStream</code> by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * @param xmlStream        the xml stream to read
     * @param vradiUser        the VradiUser that launched the import
     * @return an array containing :
     * - the number of created forms
     * - the number of already existing forms (non created)
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     * 
     * TODO EC-20100428 return a serializable structure (easier to use)
     * 
     * @throws VradiException for various possible errors
     */
    int[] getFormsFromXmlStream(XmlStream xmlStream, VradiUser vradiUser) throws VradiException;

    /**
     * Returns all the <code>Users</code>.
     *
     * @return a list of <code>User</code>
     * @throws VradiException
     */
    List<User> getAllUsers() throws VradiException;

    /**
     * Returns all the <code>Users</code>.
     * 
     * @return a list of <code>VradiUserDTO</code>
     * @throws VradiException
     */
    List<VradiUserDTO> getAllUserDTOs() throws VradiException;
    
    /**
     * Computes a list of queries returning the specified <code>Form</code>.
     *
     * @param form the form which match queries
     * @return a list of queries
     */
    Map<QueryMaker, List<VradiQueryBean>> findQueriesReturningForm(Form form) throws VradiException;

    /**
     * Returns all the <code>Forms</code> with the specified list of <code>formIds</code>.
     *
     * @param formIds the ids of the forms
     * @return a list of <code>Form</code>
     */
    List<Form> getForms(List<String> formIds) throws VradiException;

    /**
     * Updates a <code>VradiUser</code>.
     *
     * @param vradiUser the VradiUser to update
     * @return the vradiUser updated
     * @throws VradiException
     */
    VradiUser updateVradiUser(VradiUser vradiUser) throws VradiException;

    /**
     * Checks if <code>userName</code> and <code>userPassword</code> matches a valid <code>VradiUser</code>.
     *
     * @param userName     the username of a VradiUser
     * @param userPassword the password of a VradiUser
     * @return a <code>VradiUser</code> or null if either userName or userPassword don't match
     */
    VradiUser logVradiUser(String userName, String userPassword) throws VradiException;

    /**
     * Update a list of Sending.
     *
     * @param sendings the list of Sending to update
     */
    void updateSendings(List<Sending> sendings) throws VradiException;

    /**
     * Import data from an XML file.
     *
     * @param file the XML file containing datas
     * @throws VradiException
     */
    void importData(File file) throws VradiException;

    /**
     * Export the datas into an XML format.
     *
     * @return the datas as an XML String
     * @throws VradiException
     */
    String exportData() throws VradiException;

    /**
     * Create or Update a list of <code>Forms</code>.
     *
     * @param forms the list of forms to insert or update
     * @return an updated list of <code>Form</code>
     */
    List<Form> updateForms(List<Form> forms) throws VradiException;

    /**
     * Import all data as CSV.
     * 
     * FIXME EC-20100415 will not work in remote mode (or uri maybe be
     * valid for remote server too)
     * 
     * @param uri uri used to read input stream
     * @throws VradiException if any error happen during import
     */
    void importAsCSV(String uri) throws VradiException;

    /**
     * Get csv export.
     * 
     * @param criteria export criteria
     * @return csv export
     * @throws VradiException if any error happen during export
     */
    String exportAsCSV(Criteria criteria) throws VradiException;

    /**
     * Reindex the datas.
     */
    void reindexData() throws VradiException;

    /**
     * Computes the number of forms associated with the specified <code>thesaurusId</code>.
     *
     * @return the number of forms associated
     * @throws VradiException
     */
    int getNbFormsForThesaurus(String thesaurusId) throws VradiException;

    /**
     * Returns a <code>Status</code> with the specified <code>statusId</code>.
     *
     * @param statusId the id of the status
     * @return a <code>Status</code>
     * @throws VradiException
     */
    Status getStatus(String statusId) throws VradiException;

    /**
     * Returns a list of <code>Status</code> with the specified
     * <code>statusIds</code>.
     *
     * @param statusIds the ids of the statuses
     * @return a list of <code>Status</code>
     * @throws VradiException
     */
    List<Status> getStatuses(List<String> statusIds) throws VradiException;

    /**
     * Returns all <code>Status</code>.
     *
     * @return a list of <code>Status</code>
     * @throws VradiException
     */
    List<Status> getAllStatuses() throws VradiException;

    /**
     * Updates a <code>Status</code>.
     *
     * @param status the Status to update
     * @return the <code>Status</code> updated
     * @throws VradiException
     */
    Status updateStatus(Status status) throws VradiException;

    /**
     * Updates a list of <code>Status</code>.
     *
     * @param statuses the list of <code>Status</code> to update
     * @return the list of the <code>Status</code> updated
     * @throws VradiException
     */
    List<Status> updateStatuses(List<Status> statuses)
            throws VradiException;

    /**
     * Deletes a <code>Status</code>.
     *
     * @param statusId the id of the Status to delete
     * @throws VradiException
     */
    void deleteStatus(String statusId) throws VradiException;

    /**
     * Deletes a list of <code>Status</code>.
     *
     * @param statusIds the list of ids of the Status to delete
     * @throws VradiException
     */
    void deleteStatuses(List<String> statusIds) throws VradiException;

    /**
     * Propose thesaurus nodes that might be in relation with a specified form
     *
     * @param form the <code>Form</code> containing the information needed
     * to search the thesaurus nodes
     * @param thesaurus the list of all the tree nodes of the thesaurus
     * @return a list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    List<TreeNodeImpl> proposeThesaurus(Form form, List<TreeNodeImpl> thesaurus)
            throws VradiException;

    /**
     * Archives the queries of a user, client or group in a RSS file.
     * @param queryMaker the query maker whose queries are archived
     * @throws VradiException
     */
    void archiveQueries(QueryMaker queryMaker) throws VradiException;

    /**
     * Regularly retrieves the information from all the xml streams
     * and create new forms
     *
     * @param intervalUnit unit of te interval between 2 retrievings (minute, hour or day)
     * @param intervalValue interval value between two retrievings
     * @param hour hour of the retrieving if the unit is day
     * @param minute of the retrieving if the unit is day or hour
     * @throws VradiException
     */
    void autoLoadFormsFromXmlStreams(String intervalUnit,
            int intervalValue, Integer hour, Integer minute)
            throws VradiException;

    /**
     * Upload files attached to a form.
     * 
     * @param form
     * @param files
     * 
     * @deprecated can't use non serializable parameters with hessian
     */
    @Deprecated
    void uploadFiles(Form form, List<File> files) throws IOException;

    /**
     * Upload attachments attached to a form.
     * 
     * @param form
     * @param attachments
     * 
     * @deprecated can't use non serializable parameters with hessian
     */
    @Deprecated
    void uploadAttachments(Form form, List<File> attachments) throws IOException;

    /**
     * Associates a new template file to the specified extension
     *
     * @param extension
     * @param template
     * @return the file copied in the vradi template folder
     * @throws VradiException
     * 
     * @deprecated can't use non serializable parameters with hessian
     */
    @Deprecated
    File addTemplate(WikittyExtension extension, File template)
            throws VradiException;

    /**
     * Lists the template files associated with the specified extension
     * @param extension
     * @return
     * 
     * @deprecated can't use non serializable parameters with hessian
     */
    @Deprecated
    File[] getTemplates(WikittyExtension extension);

    /**
     *
     * @param extensionName
     * @param templateName
     * @return
     * 
     * @deprecated can't use non serializable parameters with hessian
     */
    @Deprecated
    File getTemplate(String extensionName, String templateName);

    void setAssociatedFields(String extensionName, String templateName,
                                    Map<String, String> fieldMap)
            throws VradiException, IOException;

    Map<String, String> getAssociatedFields(String extensionName,
                                            String templateName)
            throws VradiException;

    /**
     * Gets the query makers whose queries are potentially to modify
     * after a thesaurus node modification
     * @param thesaurusName the modified thesaurus name node
     * @return a map containing the query makers and their queries which contains
     * the thesaurus node name
     */
    Map<QueryMaker, List<VradiQueryBean>> getQueriesToModifyAfterThesaurusModification(
            String thesaurusName);

    List<Sending> removeAllSending(Session session, Form form, QueryMaker queryMaker)
            throws VradiException;
}
