/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.WikittyProxy;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.managers.ClientManager;
import com.jurismarches.vradi.services.managers.FormManager;
import com.jurismarches.vradi.services.managers.FormTypeManager;
import com.jurismarches.vradi.services.managers.MailingManager;
import com.jurismarches.vradi.services.managers.SearchManager;
import com.jurismarches.vradi.services.managers.TemplateManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;

/**
 * @author morin
 * @date 18 févr. 2010 11:48:42
 */
public class MailingServiceImpl implements MailingService {

    private static final Log log = LogFactory.getLog(MailingServiceImpl.class);

    protected WikittyProxy proxy;
    protected ThesaurusManager thesaurusManager;
    protected FormManager formManager;
    protected FormTypeManager formTypeManager;
    protected ClientManager clientManager;
    protected SearchManager searchManager;
    protected MailingManager mailingManager;

    /** Mail cron task to check mails every 10 minutes. */
    protected Timer mailCronTask;

    public MailingServiceImpl() {
        proxy = ServiceFactory.getWikittyProxy();

        thesaurusManager = new ThesaurusManager(proxy);
        clientManager = new ClientManager(proxy);
        formTypeManager = new FormTypeManager(proxy);
        mailingManager = new MailingManager(proxy);

        searchManager = new SearchManager(proxy, thesaurusManager);
        formManager = new FormManager(proxy, thesaurusManager, searchManager);
        
        // start mail cron task (check every 10 minutes)
        mailCronTask = new Timer();
        mailCronTask.schedule(mailingManager, 0, 10 * 60 * 1000);
    }
    
    @Override
    public String[] getDocumentFields(File template) throws VradiException {
        if(template == null) {
            return null;
        }
        TemplateManager templateManager = new TemplateManager(template.getPath());
        return templateManager.getDocumentFields();
    }

    @Override
    public List<File> generateFilledDocumentInPDF(File template, List<Form> forms,
                                            Map<String, String> fieldBindings)
            throws VradiException {
        return generateFilledDocumentInPDF(template, forms, fieldBindings, true);
    }

    @Override
    public List<File> generateFilledDocumentInPDF(File template, List<Form> forms,
                                            Map<String, String> fieldBindings,
                                            boolean replace)
            throws VradiException {

        List<File> generatedPDFList = null;
        
        if(template == null) {
            if (log.isWarnEnabled()) {
                log.warn("Try to generate form pdf without template");
            }
        }
        else {
            TemplateManager templateManager = new TemplateManager(template.getPath());
            
            // split fqn form field name in an array
            // composed of extension name, and extension field
            // TODO EC20100510 set in commun with same function in
            // {@link generateFilledDocumentInPDF(TemplateManager, Form form, Map<String, String[]>, boolean)
            Map<String, String[]> map = new HashMap<String, String[]>();
            for(Map.Entry<String, String> binding : fieldBindings.entrySet()) {
                String fieldFQName = binding.getValue();
                if(fieldFQName != null && fieldFQName.indexOf(".") > 0) {
                    int dot = fieldFQName.indexOf(".");
                    String[] fqField = new String[2];
                    fqField[0] = fieldFQName.substring(0, dot);
                    fqField[1] = fieldFQName.substring(dot + 1);
                    map.put((String)binding.getKey(), fqField);
                }
            }

            generatedPDFList = new ArrayList<File>();
            for (Form form : forms) {
                File f = generateFilledDocumentInPDF(templateManager, form, map, replace);
                generatedPDFList.add(f);
            }
        }

        return generatedPDFList;
    }
    
    /**
     * Generate a single pdf file for specified form.
     * 
     * @param templateManager pdf template manager
     * @param form form to get pdf
     * @param fieldBindings association between field name and form fields (couple extension name, extenstion field)
     * @param replace replace already existing pdf
     * @return the pdf file associated with pdf
     * @throws VradiException
     */
    protected File generateFilledDocumentInPDF(TemplateManager templateManager,
            Form form, Map<String, String[]> fieldBindings,
            boolean replace) throws VradiException {
        
        File pdfDir = Configuration.getInstance().getPdfDir();
        File result = new File(pdfDir, form.getWikittyId() + ".pdf");
        if (!replace && result.exists()) {
            return result;
        }

        Map<String, Object> fieldValues = new HashMap<String, Object>();
        for(Map.Entry<String, String[]> binding : fieldBindings.entrySet()) {
            String[] fqField = binding.getValue();
            Object value = form.getField(fqField[0], fqField[1]);
            fieldValues.put((String)binding.getKey(),
            value != null ? value.toString() : null);
        }
        List<File> attachments = new ArrayList<File>();
        if(form.getFiles() != null) {
            File embededFilesDir = Configuration.getInstance().getEmbededFilesDir();
            File formFilesDir = new File(embededFilesDir, form.getWikittyId());
            
            for(String file : form.getFiles()) {
                File f = new File(formFilesDir, file);
                attachments.add(f);
            }
        }
        templateManager.generateDoc(result.getPath(), fieldValues,
                attachments.toArray(new File[attachments.size()]));
        
        return result;
    }

    @Override
    public Session sendMessages(Session session, List<Sending> sendings) throws VradiException {
        VradiStorageService storageService = ServiceFactory.getVradiStorageService();
        boolean resumeSession = false;
        if (session.getStatus() == VradiConstants.SessionStatus.STOPPED.getValue()){
            log.info("Session was stopped, resume sending");
            session.setStatus(VradiConstants.SessionStatus.ACTIVE.getValue());
            session = storageService.updateEntity(session);
            resumeSession = true;
        }
        for(Sending sending : sendings) {

            // Flag to detect if it's canceled by user
            session = storageService.getEntity(session.getWikittyId(), Session.class);
            if (session.getStatus() == VradiConstants.SessionStatus.STOPPED.getValue()){
                log.warn("Sending stopped by user");
                return session;
            }

            if (resumeSession && sending.getStatus() != VradiConstants.SendingStatus.TO_SEND.getValue()){
                if (log.isDebugEnabled()) {
                    log.debug("Sending " + sending.getWikittyId() + " already sent, skip");
                }
            }
            else {
                // Send
                sendMessage(sending, session.getParagraph());
            }
        }

        // Set session to sent
        session.setStatus(VradiConstants.SessionStatus.SENT.getValue());
        session.setSessionDate(new Date());

        // Update all
        session = storageService.updateEntity(session);

        return session;
    }

    @Override
    public Sending sendMessage(Sending sending, String sessionParagraph) throws VradiException {
        return sendMessage(sending, sessionParagraph, null);
    }

    @Override
    public Sending sendMessage(Sending sending, String sessionParagraph, String email) throws VradiException {

        // Throw sending in error (email or pdf not found) or deleted
        if (sending.getStatus() == VradiConstants.SendingStatus.ERROR.getValue() ||
            sending.getStatus() == VradiConstants.SendingStatus.DELETED.getValue()){
            return sending;
        }

        VradiStorageService storageService = ServiceFactory.getVradiStorageService();

        // If its client or group sending : do nothing
        if (sending.getUser() == null) {
            sending.setStatus(VradiConstants.SendingStatus.SENT.getValue());
            return storageService.updateEntity(sending);
        }

        // Get email address
        if (email == null || email.isEmpty()) {
            User user = clientManager.getUser(sending.getUser());
            email = user.getEmail();
        }

        String subject = Configuration.getInstance().getOfferMailSubject();

        // Set message
        String message = sessionParagraph + "\n\n" + sending.getParagraph();

        List<Form> forms = formManager.getForms(
                new ArrayList(sending.getForm()));

        boolean receptionProof = sending.getReceptionProof();

        // Post mail and save msg id
        String messageId = null;
        try {
            messageId = mailingManager.postMail(
                email, subject, message, forms, receptionProof);
        } catch (VradiException eee) {
            // Set sending in error
            sending.setStatus(VradiConstants.SendingStatus.ERROR.getValue());
            proxy.store(sending);
            throw eee;
        }
        sending.setMessageId(messageId);

        // Set sending status
        sending.setStatus(receptionProof ?
                VradiConstants.SendingStatus.WAITING_RECEPTION_PROOF.getValue() :
                VradiConstants.SendingStatus.SENT.getValue());

        // Update sending
        sending = storageService.updateEntity(sending);

        return sending;
    }

    @Override
    public void receiveMessages() throws VradiException {
        mailingManager.receiveMails();
    }

    @Override
    public Session stopSentMail(Session session) throws VradiException {

        // Flag status to stop traitement
        session.setStatus(VradiConstants.SessionStatus.STOPPED.getValue());

        // Update
        session = ServiceFactory.getVradiStorageService().updateEntity(session);

        return session;
    }
}
