/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

import org.apache.commons.lang.UnhandledException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;

/**
 * Configuration.
 *
 * @author schorlet
 * @version $Revision: 847 $ $Date: 2010-05-05 15:50:56 +0200 (mer., 05 mai 2010) $
 * @since 26 mars 2010 21:18:08
 */
public class Configuration extends ApplicationConfig {

    /** log. */
    private static final Log log = LogFactory.getLog(Configuration.class);

    /** Singleton instance. */
    protected static Configuration instance = null;

    protected Configuration() {
        super();
        
        // set defaut option (included configuration file name : important)
        for (Option o : Option.values()) {
            if (o.defaultValue != null) {
                setDefaultOption(o.key, o.defaultValue);
            }
        }
    }

    public static synchronized Configuration getInstance() {
        if (instance == null) {
            try {
                instance = new Configuration();
                instance.parse(new String[]{});
                
                if (log.isDebugEnabled()) {
                    instance.printConfig();
                }
            } catch (Exception e) {
                throw new UnhandledException(e);
            }
        }
        return instance;
    }

    /**
     * Redefined parse to set System.property based on properties value
     * from configuration file.
     */
    @Override
    public void parse(String[] args) throws ArgumentsParserException {
        super.parse(args);
        setSystemDefaultProps();
    }

    /**
     * @deprecated redefined to not report a bug in nuiton-utils !!!
     */
    @Deprecated
    public File getOptionAsFile2(String key) {
        String optionAsString = getOption(key);
        if (optionAsString != null) {
            File optionAsFile = new File(optionAsString);
            return optionAsFile.getAbsoluteFile();
        } else {
            return null;
        }
    }

    /**
     * Get version as string.
     * 
     * @return version
     */
    public String getVersion() {
        return getOption(Option.APPLICATION_VERSION.key);
    }

    public File getDataDirAsFile() {
        File option = getOptionAsFile(Option.DATA_DIR.key);
        return option;
    }

    public File getCommonsPropertiesFile() {
        return getOptionAsFile("vradi.data.dir", ".vradi-properties");
    }

    public File getAttachmentsDir() {
        return getOptionAsFile("vradi.data.dir", "attachments");
    }

    public File getEmbededFilesDir() {
        return getOptionAsFile("vradi.data.dir", "embededFiles");
    }

    public File getQueryHistoryDir() {
        return getOptionAsFile("vradi.data.dir", "queryHistory");
    }

    public File getTemplatesDir() {
        return getOptionAsFile("vradi.data.dir", "templates");
    }

    public File getPdfDir() {
        return getOptionAsFile("vradi.data.dir", "pdf");
    }

    public String getOpenOfficeExecDir() {
        return getOption("oOo.exec.folder");
    }

    // Email config
    /*public String getMailerDaemonEmail() {
        return getOption("vradi.mail.mailerdaemonemail");
    }*/
    
    public String getSmtpHost() {
        return getOption("vradi.smtp.host");
    }

    public int getSmtpPort() {
        return getOptionAsInt("vradi.smtp.port");
    }

    public String getImapHost() {
        return getOption("vradi.imap.host");
    }

    public int getImapPort() {
        return getOptionAsInt("vradi.imap.port");
    }

    public String getMailUser() {
        return getOption("vradi.mail.user");
    }
    
    public String getMailFrom() {
        return getOption("vradi.mail.from");
    }
    
    public String getMailFromName() {
        return getOption("vradi.mail.fromname");
    }

    public String getMailPassword() {
        return getOption("vradi.mail.password");
    }

    public String getOfferMailSubject() {
        String result = getOption("vradi.mail.offer.subject");
        return result;
    }

    public boolean isMailDebug() {
        return getOptionAsBoolean("vradi.mail.debug");
    }

    public String getSessionDefaultParagraph() {
        return getOption("vradi.session.defaultParagraph");
    }

    public boolean skipMigration() {
        return getOptionAsBoolean("vradi.migration.skip");
    }

    public Properties getJdbcConfigProperties() {
        File optionAsFile = getOptionAsFile2("vradi.jdbc.config.file");
        return getFileAsProperties(optionAsFile);
    }

    public Properties getJmsPublisherProperties() {
        File optionAsFile = getOptionAsFile2("vradi.jms.publisher.file");
        return getFileAsProperties(optionAsFile);
    }

    public URL getJmsPublisherURL() {
        File optionAsFile = getOptionAsFile2("vradi.jms.publisher.file");
        try {
            if (optionAsFile != null && optionAsFile.exists() && optionAsFile.isFile()) {
                return optionAsFile.toURI().toURL();
                
            } else if (optionAsFile != null) {
                String configFileName = optionAsFile.getName();
                URL url = ClassLoader.getSystemResource(configFileName);
                
                if (url == null) {
                    ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                    url = contextClassLoader.getResource(configFileName);
                }
                
                return url;
            }
        } catch (Exception e) {
            log.warn(e.getMessage(), e);
        }

        return null;
    }

    public Properties getJmsSubscriberProperties() {
        File optionAsFile = getOptionAsFile2("vradi.jms.subscriber.file");
        return getFileAsProperties(optionAsFile);
    }

    protected File getOptionAsFile(String parent, String child) {
        String parentDir = getOption(parent);
        File option = new File(parentDir, child);
        return option;
    }

    protected Properties getFileAsProperties(File configFile) {
        Properties properties = new Properties();

        if (configFile != null && configFile.exists() && configFile.isFile()) {
            try {
                URL url = configFile.toURI().toURL();
                InputStream openStream = url.openStream();
                properties.load(openStream);
                openStream.close();
            } catch (Exception e) {
                log.warn(e.getMessage(), e);
                properties = new Properties();
            }

        } else if (configFile != null) {
            String configFileName = configFile.getName();
            URL url = ClassLoader.getSystemResource(configFileName);
            if (url == null) {
                ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                url = contextClassLoader.getResource(configFileName);
            }

            if (url != null) {
                try {
                    InputStream openStream = url.openStream();
                    properties.load(openStream);
                    openStream.close();
                } catch (Exception e) {
                    log.warn(e.getMessage(), e);
                    properties = new Properties();
                }
            }
        }

        return properties;
    }

    protected void setSystemDefaultProps() {
        String[] datadirs = new String[] { "solr.data.dir", "vradi.objectStore.dir"};
        
        for (String datadir : datadirs) {
            String value = System.getProperty(datadir, null);
            if (value == null) {
                value = getOption(datadir);
                log.info("Setting System property " + datadir + ": " + value);
                
                System.setProperty(datadir, value);
                env.put(datadir, value);
            }
        }
    }

    /**
     * Vradi option definition.
     */
    public static enum Option {

        CONFIG_FILE(CONFIG_FILE_NAME, _("vradi.config.configFileName.description"), "vradi-services.properties", String.class, true, true),
        APPLICATION_VERSION("application.version", _("vradi.config.application.version.description"), null, String.class, false, false),
        DATA_DIR("vradi.data.dir", _("vradi.config.data.dir.description"), null, Boolean.class, false, false);

        public final String key;
        public final String description;
        public final String defaultValue;
        public final Class<?> type;
        public final boolean _transient;
        public final boolean _final;

        private Option(String key, String description, String defaultValue,
                Class<?> type, boolean _transient, boolean _final) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this._final = _final;
            this._transient = _transient;
        }
    }
}
