/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.search;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.PrefixQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.RangeQuery;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.WildcardQuery;

/**
 * <code>LuceneQueryParser</code> parses a lucene <code>Query</code> object.
 * Inner queries are passed to a <code>QueryHandler</code>.
 * 
 * @author $Author: schorlet $
 * @version $Revision: 439 $ $Date: 2010-01-24 15:27:45 +0100 (dim., 24 janv. 2010) $
 * @since
 */
public class LuceneQueryParser {
    private static final Log log = LogFactory.getLog(LuceneQueryParser.class);

    private final QueryHandler termsFactory;
    private BooleanClause booleanClause = null;
    
    public LuceneQueryParser(QueryHandler termsFactory) {
        this.termsFactory = termsFactory;
    }

    public synchronized void parse(Query query) throws UnsupportedQueryException {
        if (log.isTraceEnabled()) {
            log.trace("parse: " + query);
        }

        if (query instanceof BooleanQuery) {
            render((BooleanQuery) query);
        } else if (query instanceof PrefixQuery) {
            render((PrefixQuery) query);
        } else if (query instanceof TermQuery) {
            render((TermQuery) query);
        } else if (query instanceof PhraseQuery) {
            render((PhraseQuery) query);
        } else if (query instanceof WildcardQuery) {
            render((WildcardQuery) query);
        } else if (query instanceof RangeQuery) {
            render((RangeQuery) query);
        } else {
            throw new UnsupportedQueryException(query);
        }
    }

    private void render(BooleanQuery query) throws UnsupportedQueryException {
        BooleanClause booleanClauses[] = query.getClauses();
        
        booleanClause = booleanClauses[0];
        termsFactory.group(booleanClause);
        
        for (int i = 0; i < booleanClauses.length; i++) {
            booleanClause = booleanClauses[i];
            parse(booleanClause.getQuery());
        }

        termsFactory.ungroup();
    }

    private void render(TermQuery query) {
        termsFactory.termQuery(query, booleanClause);
    }

    private void render(PrefixQuery query) {
        termsFactory.prefixQuery(query, booleanClause);
    }

    private void render(WildcardQuery query) {
        termsFactory.wildcardQuery(query, booleanClause);
    }

    private void render(RangeQuery query) {
        termsFactory.rangeQuery(query, booleanClause);
    }

    private void render(PhraseQuery query) {
        Term terms[] = query.getTerms();
        StringBuilder sb = new StringBuilder();

        for (int i = 0; i < terms.length; i++) {
            Term term = terms[i];
            sb.append(term.text()).append(" ");
        }

        String toString = sb.toString().trim();
        Term term = new Term(terms[0].field(), toString);

        render(new TermQuery(term));
    }
}
