/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.search;

import java.util.List;
import java.util.Stack;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.PrefixQuery;
import org.apache.lucene.search.RangeQuery;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.WildcardQuery;

import com.jurismarches.vradi.services.search.FilterList.Operator;

/**
 * <code>FilterListHandler</code> handles lucene queries and assembles them into
 * a <code>FilterList</code> tree.
 * 
 * @author $Author: schorlet $
 * @version $Revision: 439 $ $Date: 2010-01-24 15:27:45 +0100 (dim., 24 janv. 2010) $
 * @since 0.0.4
 */
public class FilterListQueryHandler implements QueryHandler {
    private static final Log log = LogFactory.getLog(FilterListQueryHandler.class);

    // filterList stack
    private final Stack<FilterList> filterListStack = new Stack<FilterList>();

    public FilterListQueryHandler() {
        FilterList filterList = new FilterList();
        filterListStack.push(filterList);
    }

    @Override
    public void prefixQuery(PrefixQuery query, BooleanClause booleanClause) {
        Term term = query.getPrefix();
        filter(term.field(), term.text() + "*", booleanClause);
    }

    @Override
    public void termQuery(TermQuery query, BooleanClause booleanClause) {
        Term term = query.getTerm();
        filter(term.field(), term.text(), booleanClause);
    }

    @Override
    public void rangeQuery(RangeQuery query, BooleanClause booleanClause) {
        String field = query.getField();
        Term lowerTerm = query.getLowerTerm();
        Term upperTerm = query.getUpperTerm();
        // TODO: manage "between filter"
    }

    @Override
    public void wildcardQuery(WildcardQuery query, BooleanClause booleanClause) {
        Term term = query.getTerm();
        filter(term.field(), term.text(), booleanClause);
    }

    @Override
    public void group(BooleanClause booleanClause) {
        if (booleanClause == null) {
            return;
        }
        
        FilterList.Operator operator = Operator.getOperator(booleanClause);
        FilterList filterList = new FilterList(operator);
        
        if (log.isTraceEnabled()) {
            log.trace("group: " + operator.toString());
        }
        
        FilterList peek = filterListStack.peek();
        peek.addFilter(filterList);

        filterListStack.push(filterList);
    }

    @Override
    public void ungroup() {
        if (filterListStack.size() > 1) {
            filterListStack.pop();
        }
    }

    private void filter(String term, String text, BooleanClause booleanClause) {
        Filter filter = new CompareFilter(term, text);
        FilterList peek = filterListStack.peek();
        
        FilterList.Operator operator = Operator.getOperator(booleanClause);
        
        if (operator == FilterList.Operator.MUST_NOT_PASS &&
                peek.getOperator() != operator) {
            
            group(booleanClause);
            peek = filterListStack.peek();
            peek.addFilter(filter);
            ungroup();
            
        } else {
            peek.addFilter(filter);
        }
    }

    public FilterList getFilterList() {
        FilterList list = filterListStack.get(0);
        
        if (log.isDebugEnabled()) {
            traceFilterList(list, "");
        }
        
        return list;
    }

    static void traceFilterList(FilterList list, String spaces) {
        if (log.isDebugEnabled()) {
            Operator operator = list.getOperator();
            log.debug(String.format("%s%s [", spaces, operator.toString()));
            spaces = spaces + "  ";
            List<Filter> filters = list.getFilters();
            for (Filter filter : filters) {
                if (filter instanceof FilterList) {
                    traceFilterList((FilterList) filter, spaces);
                } else {
                    log.debug(String.format("%s%s", spaces, filter.toString()));
                }
            }
            spaces = spaces.substring(0, spaces.length() - 2);
            log.debug(spaces + "]");
        }
    }

}
