/*
 * #%L
 * SGQ :: Business
 * $Id: ConfigurationService.java 364 2013-05-13 14:26:18Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/ConfigurationService.java $
 * %%
 * Copyright (C) 2012 - 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.entities.SgqConfiguration;
import com.herbocailleau.sgq.entities.SgqConfigurationDAO;

public class ConfigurationService extends SgqService {

    private static final Log log = LogFactory.getLog(ConfigurationService.class);

    public static final String NEXT_NON_BIO_BATCH_NUMBER_CONFIG = "nextNonBioBatchNumber";

    public static final String NEXT_BIO_BATCH_NUMBER_CONFIG = "nextBioBatchNumber";

    /**
     * Get {@code nextNonBioBatchNumber} from database or appConfiguration.
     * 
     * @return current nextNonBioBatchNumber
     */
    public int getNextNonBioBatchNumber() {
        int result;
        try {
            SgqConfigurationDAO configurationDAO = daoHelper.getSgqConfigurationDAO();
            SgqConfiguration sqgConfig = configurationDAO.findByKey(NEXT_NON_BIO_BATCH_NUMBER_CONFIG);

            if (sqgConfig != null) {
                result = Integer.parseInt(sqgConfig.getValue());
            } else {
                result = config.getBatchRangeNonBioStart();
            }
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't read configuration", ex);
        }
        return result;
    }

    /**
     * Save {@code nextBioBatchNumber} into database.
     * 
     * @return current nextBioBatchNumber
     */
    public void saveBatchLimit(int nextNonBioBatchNumber, int nextBioBatchNumber) {
        try {
            SgqConfigurationDAO configurationDAO = daoHelper.getSgqConfigurationDAO();

            // nextNonBioBatchNumber
            SgqConfiguration sqgConfig = configurationDAO.findByKey(NEXT_NON_BIO_BATCH_NUMBER_CONFIG);
            if (sqgConfig == null) {
                sqgConfig = configurationDAO.create();
                sqgConfig.setKey(NEXT_NON_BIO_BATCH_NUMBER_CONFIG);
            }
            sqgConfig.setValue(String.valueOf(nextNonBioBatchNumber));
            configurationDAO.update(sqgConfig);
            
            // nextBioBatchNumber
            SgqConfiguration sqgBioConfig = configurationDAO.findByKey(NEXT_BIO_BATCH_NUMBER_CONFIG);
            if (sqgBioConfig == null) {
                sqgBioConfig = configurationDAO.create();
                sqgBioConfig.setKey(NEXT_BIO_BATCH_NUMBER_CONFIG);
            }
            sqgBioConfig.setValue(String.valueOf(nextBioBatchNumber));
            configurationDAO.update(sqgBioConfig);

            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't read configuration", ex);
        }
    }

    /**
     * Get {@code nextBioBatchNumber} from database or appConfiguration.
     * 
     * @return current nextBioBatchNumber
     */
    public int getNextBioBatchNumber() {
        int result;
        try {
            SgqConfigurationDAO configurationDAO = daoHelper.getSgqConfigurationDAO();
            SgqConfiguration sqgConfig = configurationDAO.findByKey(NEXT_BIO_BATCH_NUMBER_CONFIG);

            if (sqgConfig != null) {
                result = Integer.parseInt(sqgConfig.getValue());
            } else {
                result = config.getBatchRangeBioStart();
            }
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't read configuration", ex);
        }
        return result;
    }
    
    /**
     * Get {@code lastNonBioBatchNumber} from appConfiguration.
     * 
     * @return current lastNonBioBatchNumber
     */
    public int getLastNonBioBatchNumber() {
        int result = config.getBatchRangeNonBioEnd();
        return result;
    }
    
    /**
     * Get {@code lastBioBatchNumber} from appConfiguration.
     * 
     * @return current lastBioBatchNumber
     */
    public int getLastBioBatchNumber() {
        int result = config.getBatchRangeBioEnd();
        return result;
    }

    /**
     * Execute list devices script to get connected devices map (id and names).
     * 
     * @return devices id and size
     */
    public Map<String, String> getBackupDevices() {
        Map<String, String> result = new HashMap<String, String>();

        String listDevicesCommand = config.getBackupDevicesCommand();
        try {

            // execute command
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            ByteArrayOutputStream errorStream = new ByteArrayOutputStream();
            CommandLine commandline = CommandLine.parse(listDevicesCommand);
            DefaultExecutor exec = new DefaultExecutor();
            PumpStreamHandler streamHandler = new PumpStreamHandler(outputStream, errorStream);
            exec.setStreamHandler(streamHandler);
            exec.execute(commandline);

            // test error output
            String error = errorStream.toString();
            if (StringUtils.isNotBlank(error)) {
                throw new SgqBusinessException(error);
            }

            // parse output stream
            String output = outputStream.toString();
            String[] lines = output.split("\n");
            for (String line : lines) {
                // non empty line with space
                int indexOfTab = line.indexOf('\t');
                if (indexOfTab > 1) {
                    String deviceId = line.substring(0, indexOfTab).trim();
                    String size = line.substring(indexOfTab + 1).trim();
                    result.put(deviceId, size);
                }
            }
        } catch (IOException ex) {
            // ex.getMessage() is usefull for ui display
            throw new SgqBusinessException(ex.getMessage(), ex);
        }

        return result;
    }

    /**
     * Run save backup command and wait for command ending.
     * 
     * @param deviceId selected device id
     * @param notificationEmail notification email
     */
    public void saveBackupOnDevice(String deviceId, String notificationEmail) {
        // get configuration
        String exportCommand = config.getBackupPerformCommand();
        // replace commend variables
        exportCommand = exportCommand.replace("%d", deviceId);
        exportCommand = exportCommand.replace("%f", "sgq-backup-" + DateFormatUtils.format(new Date(), "ddMMyyyy") + ".sql");
        exportCommand = exportCommand.replace("%m", notificationEmail);

        if (log.isDebugEnabled()) {
            log.debug("Executing script command : " + exportCommand);
        }

        try {
            // execute command
            CommandLine commandline = CommandLine.parse(exportCommand);
            DefaultExecutor exec = new DefaultExecutor();
            exec.execute(commandline);
        } catch (IOException ex) {
            // ex.getMessage() is usefull for ui display
            throw new SgqBusinessException(ex.getMessage(), ex);
        }
    }
}
