/*
 * #%L
 * SGQ :: Business
 * $Id: PresentationDAOImpl.java 164 2012-10-12 14:08:00Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.4/sgq-business/src/main/java/com/herbocailleau/sgq/entities/PresentationDAOImpl.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.entities;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.topia.TopiaException;

import com.herbocailleau.sgq.business.model.PresentationModel;

public class PresentationDAOImpl<E extends Presentation> extends PresentationDAOAbstract<E> {

    /**
     * Find all presentation for non expired and valid batch.
     * 
     * @param productName 
     * @return
     * @throws TopiaException 
     */
    public List<Presentation> findAllNonExpiredValid(String productName) throws TopiaException {
        String query = "from " + Presentation.class.getName() +
                " where " + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " and " + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_INVALID + " = false";

        List<Presentation> result = null;
        if (StringUtils.isNotBlank(productName)) {
            query += " and " + BatchDAOImpl.getFieldLikeInsensitive(Presentation.PROPERTY_BATCH +
                        "." + Batch.PROPERTY_PRODUCT + "." + Product.PROPERTY_NAME, ":product");
            result = context.findAll(query, "product", "%" + StringUtils.stripAccents(productName) + "%");
        } else {
            result = context.findAll(query);
        }

        return result;
    }
    
    /**
     * Find all presentation for non expired and valid batch.
     * 
     * @param productName 
     * @return
     * @throws TopiaException 
     */
    public List<PresentationModel> findAllNonExpiredValidModel(String productName) throws TopiaException {
        String query = "select P, PP.place," +
                " (select count(*) from " + Expedition.class.getName() +
                "  where " + Expedition.PROPERTY_PRESENTATION + " = P and " + Expedition.PROPERTY_SOURCE + " = :zp)," +
                " (select count(*) from " + Production.class.getName() +
                "  where " + Production.PROPERTY_PRESENTATION + " = P and " + Production.PROPERTY_DESTINATION + " = :zc)" +
                " from " + Presentation.class.getName() + " P," +
                " " + ProductPlace.class.getName() + " PP" +
                " where P." + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_EXPIRED_DATE + " = null" +
                " and P." + Presentation.PROPERTY_BATCH + "." + Batch.PROPERTY_INVALID + " = false" +
                " and P.batch.product = PP.product";

        List<Object[]> lines = null;
        if (StringUtils.isNotBlank(productName)) {
            query += " and " + BatchDAOImpl.getFieldLikeInsensitive("P." + Presentation.PROPERTY_BATCH +
                        "." + Batch.PROPERTY_PRODUCT + "." + Product.PROPERTY_NAME, ":product");
            lines = context.findAll(query, "product", "%" + StringUtils.stripAccents(productName) + "%",
                    "zc", Zone.ZC, "zp", Zone.ZP);
        } else {
            lines = context.findAll(query, "zc", Zone.ZC, "zp", Zone.ZP);
        }

        // les lignes doivent être triées par Zone pour l'algorithme
        // les lignes ZE, ZP, ZC dans cet ordre
        Collections.sort(lines, new Comparator<Object[]>() {
            @Override
            public int compare(Object[] o1, Object[] o2) {
                int result = -1;
                Zone o1Zone = ((Place)o1[1]).getZone();
                Zone o2Zone = ((Place)o2[1]).getZone();
                // fix Java 7 Comparison method violates its general contract!
                if (o1Zone == o2Zone) {
                    result = 0;
                } else {
                    switch (((Place)o1[1]).getZone()) {
                    case ZC:
                        result = 1;
                        break;
                    case ZP:
                        if (o2Zone == Zone.ZE) {
                            result = 1;
                        }
                        break;
                    }
                }
                return result;
            }
        });
        
        // les résultats doivent être trié par le code presentation pour le rendu
        Map<Presentation, PresentationModel> presentationCache = new HashMap<Presentation, PresentationModel>();
        for (Object[] line : lines) {
            Presentation presentation = (Presentation)line[0];
            Place place = (Place)line[1];
            long expeditionCount = ((Number)line[2]).longValue();
            long productionCount = ((Number)line[3]).longValue();
            PresentationModel model = presentationCache.get(presentation);
            if (model == null) {
                model = new PresentationModel(presentation);
                presentationCache.put(presentation, model);
            }
            // ZE, on l'ajoute tout le temps
            if (place.getZone() == Zone.ZE ||
                    (place.getZone() == Zone.ZP && (model.getPlaces().isEmpty() || expeditionCount > 0)) ||
                    (place.getZone() == Zone.ZC && (model.getPlaces().isEmpty() || productionCount > 0))) {
                model.addPlace(place);
            }
        }
        
        // build new result list sorted by presentation code
        List<PresentationModel> result = new ArrayList<PresentationModel>(presentationCache.values());
        Collections.sort(result, new Comparator<PresentationModel>() {
            @Override
            public int compare(PresentationModel o1, PresentationModel o2) {
                int result = o1.getPresentation().getPresentationCode().ordinal()
                        - o2.getPresentation().getPresentationCode().ordinal();
                if (result == 0) {
                    result = -1;
                }
                return result;
            }
        });
        return result;
    }
    
    /**
     * Retourne les presentations d'un lot ordonnées par le code presentation,
     * avec pour chaque présentation, ses emplacements théoriques disponibles.
     * 
     * Pour la détermination des emplacements:
     * <ul>
     * <li>plante qui arrive par ordre de présence des zones: ZE, ZP, ZC
     * <li>une plante est présente en ZC s'il y a un mouvement pour le client 99997
     * <li>une plante est présente en ZP s'il y a un mouvement pour un client externe depuis le fichier (FIC_HIST.TXT)
     * </ul>
     * 
     * @param batch batch
     * @return batch presentation
     * @throws TopiaException 
     */
    public List<PresentationModel> findAllByBatchOrderByPresentation(Batch batch) throws TopiaException {
        String query = "select P, PP.place," +
                " (select count(*) from " + Expedition.class.getName() +
                "  where " + Expedition.PROPERTY_PRESENTATION + " = P and " + Expedition.PROPERTY_SOURCE + " = :zp)," +
                " (select count(*) from " + Production.class.getName() +
                "  where " + Production.PROPERTY_PRESENTATION + " = P and " + Production.PROPERTY_DESTINATION + " = :zc)" +
                " from " + Presentation.class.getName() + " P," +
                " " + ProductPlace.class.getName() + " PP" +
                " where P." + Presentation.PROPERTY_BATCH + " = :batch" +
                " and P.batch.product = PP.product";

        // les lignes doivent être triées par Zone pour l'algorithme
        // les lignes ZE, ZP, ZC dans cet ordre
        List<Object[]> lines = (List<Object[]>)context.findAll(query, "batch", batch, "zp", Zone.ZP, "zc", Zone.ZC);
        Collections.sort(lines, new Comparator<Object[]>() {
            @Override
            public int compare(Object[] o1, Object[] o2) {
                int result = -1;
                Zone o1Zone = ((Place)o1[1]).getZone();
                Zone o2Zone = ((Place)o2[1]).getZone();
                // fix Java 7 Comparison method violates its general contract!
                if (o1Zone == o2Zone) {
                    result = 0;
                } else {
                    switch (((Place)o1[1]).getZone()) {
                    case ZC:
                        result = 1;
                        break;
                    case ZP:
                        if (o2Zone == Zone.ZE) {
                            result = 1;
                        }
                        break;
                    }
                }
                return result;
            }
        });

        // les résultats doivent être trié par le code presentation pour le rendu
        Map<Presentation, PresentationModel> presentationCache = new HashMap<Presentation, PresentationModel>();
        for (Object[] line : lines) { 
            Presentation presentation = (Presentation)line[0];
            Place place = (Place)line[1];
            long expeditionCount = ((Number)line[2]).longValue();
            long productionCount = ((Number)line[3]).longValue();
            PresentationModel model = presentationCache.get(presentation);
            if (model == null) {
                model = new PresentationModel(presentation);
                presentationCache.put(presentation, model);
            }
            
            // ZE, on l'ajoute tout le temps
            if (place.getZone() == Zone.ZE ||
                    (place.getZone() == Zone.ZP && (model.getPlaces().isEmpty() || expeditionCount > 0)) ||
                    (place.getZone() == Zone.ZC && (model.getPlaces().isEmpty() || productionCount > 0))) {
                model.addPlace(place);
            }
        }

        // build new result list sorted by presentation code
        List<PresentationModel> result = new ArrayList<PresentationModel>(presentationCache.values());
        Collections.sort(result, new Comparator<PresentationModel>() {
            @Override
            public int compare(PresentationModel o1, PresentationModel o2) {
                int result = o1.getPresentation().getPresentationCode().ordinal()
                        - o2.getPresentation().getPresentationCode().ordinal();
                if (result == 0) {
                    result = -1;
                }
                return result;
            }
        });
        return result;
    }

} //PresentationDAOImpl<E extends Presentation>
