package com.franciaflex.faxtomail.ui.swing.actions;

/*
 * #%L
 * FaxToMail :: UI
 * $Id: LoadFolderEmailsAction.java 660 2014-10-02 11:30:33Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-1.0/faxtomail-ui-swing/src/main/java/com/franciaflex/faxtomail/ui/swing/actions/LoadFolderEmailsAction.java $
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import com.franciaflex.faxtomail.persistence.entities.WaitingState;
import com.franciaflex.faxtomail.persistence.entities.MailField;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.swing.table.filter.TableRowFilterSupport;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.util.pagination.PaginationResult;

import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.persistence.entities.RangeRow;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeListUI;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeListUIHandler;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeListUIModel;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.content.demande.RangeRowModel;

/**
 * Action de chargement des demandes d'un dossier lors de la selection d'un dossier dans l'arbre
 * des dossiers.
 * 
 * @author Kevin Morin (Code Lutin)
 */
public class LoadFolderEmailsAction extends AbstractFaxToMailAction<DemandeListUIModel, DemandeListUI, DemandeListUIHandler> {

    private final static Log log = LogFactory.getLog(LoadFolderEmailsAction.class);

    public LoadFolderEmailsAction(DemandeListUIHandler handler) {
        super(handler, false);
    }

    @Override
    public boolean prepareAction() throws Exception {
        boolean result;

        DemandeListUIModel model = getModel();
        MailFolder folder = model.getSelectedFolder();

        if (folder == null) {
            result = false;

        } else {
            final JXTable dataTable = getUI().getDataTable();

            setActionDescription(t("faxtomail.action.loadFolderEmails.tip", folder.getName()));

            // change name to save the state of the column width for every folder
            // (as every folder can have a different header)
            getContext().getSwingSession().updateState();
            String columns = handler.populateColumnModel(dataTable, false);
            dataTable.setName("dataTable" + columns);
            getContext().getSwingSession().add(dataTable, true);

            // fixes #5528 filtre sur les colonnes : il manque l'icone qui indique que la colonne est filtrée
            handler.initTableFilter();
            result = true;
        }

        return result;
    }

    @Override
    public void doAction() throws Exception {
        DemandeListUIHandler handler = getHandler();
        DemandeListUIModel model = getModel();
        MailFolder folder = model.getSelectedFolder();
        FaxToMailUser currentUser = getContext().getCurrentUser();

        List<MailField> columns = handler.getColumns();
        Collection<MailField> fields = new HashSet<MailField>();
        if (columns != null) {
            fields.addAll(columns);
        }
        fields.add(MailField.TAKEN_BY);

        PaginationResult<Email> paginationResult = getContext().newServiceContext().getEmailService()
                                                    .getEmailForFolder(folder,
                                                                       currentUser,
                                                                       getModel().getEmailFilter(),
                                                                       model.getPaginationParameter());
        List<Email> emails = paginationResult.getElements();
        if (log.isDebugEnabled()) {
            log.debug(emails.size() + " emails in folder " + folder.getName());
        }

        Map<String, WaitingState> waitingStateById = Maps.uniqueIndex(getContext().getWaitingStateCache(), TopiaEntities.getTopiaIdFunction());

        List<DemandeUIModel> demands = new ArrayList<DemandeUIModel>();
        for (Email email : emails) {
            DemandeUIModel demand = new DemandeUIModel();
            demand.fromEntityExcluding(email, Sets.newHashSet(Email.PROPERTY_MAIL_FOLDER, Email.PROPERTY_HISTORY));

            // on ajoute toutes les lignes de rangeRow,
            // qui sont toutes valides puisqu'on les récupère de la base et qu'on ne sauvegarde que les lignes valides
            Collection<RangeRow> rangeRows = demand.getRangeRow();
            if (rangeRows != null) {
                for (RangeRow rangeRow : rangeRows) {
                    if (rangeRow != null) {
                        RangeRowModel rangeRowModel = new RangeRowModel();
                        rangeRowModel.fromEntity(rangeRow);
                        if (rangeRowModel.isValid()) {
                            demand.addValidRangeRow(rangeRowModel);
                        }
                    }
                }
            }

            demand.setValid(handler.isDemandeValid(demand));
            demand.setMailFolder(folder);
            demand.setEditable(folder.isFolderWritable() && demand.getDemandStatus().isEditableStatus());
            WaitingState waitingState = email.getWaitingState();
            if (waitingState != null) {
                demand.setWaitingState(waitingStateById.get(waitingState.getTopiaId()));
            }
            demands.add(demand);
        }

        model.setEmails(demands);
        model.setPaginationParameter(paginationResult.getCurrentPage());
        model.setPaginationResult(paginationResult);
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        DemandeUIModel currentEmail = getContext().getCurrentEmail();

        if (currentEmail != null) {
            JXTable dataTable = getUI().getDataTable();
            AbstractApplicationTableModel<DemandeUIModel> dataTableModel =
                    (AbstractApplicationTableModel<DemandeUIModel>) dataTable.getModel();
            int row = dataTableModel.getRowIndex(currentEmail);
            if (row > 0) {
                dataTable.setRowSelectionInterval(row, row);
                dataTable.scrollRowToVisible(row);
            }
        }
    }

    @Override
    protected void releaseAction() {
        super.releaseAction();
        getUI().getNavigationTree().setEnabled(true);
    }
}
