/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyServiceStorage.java 699 2011-01-31 12:13:07Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.4/wikitty-api/src/main/java/org/nuiton/wikitty/services/WikittyServiceStorage.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty.services;

import org.nuiton.wikitty.storage.WikittyExtensionStorage;
import org.nuiton.wikitty.storage.WikittySearchEngine;
import org.nuiton.wikitty.storage.WikittyStorage;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.LinkedHashSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.WikittyTreeNodeHelper;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.TreeNodeResult;
import org.nuiton.wikitty.search.operators.Element;

/**
 * WikittyService is main service
 *
 * @author poussin
 * @version $Revision: 699 $
 *
 * Last update: $Date: 2011-01-31 13:13:07 +0100 (lun., 31 janv. 2011) $
 * by : $Author: bpoussin $
 */
public class WikittyServiceStorage implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceStorage.class);

    /** Default migration use to migrate a wikitty in last extension version */
    protected WikittyExtensionMigration defaultExtensionMigration =
            new WikittyExtensionMigrationRename();

    protected WikittySearchEngine searchEngine;
    protected WikittyExtensionStorage extensionStorage;
    protected WikittyStorage wikittyStorage;

    /**
     * TODO poussin 20101027 remove it when all used WikittyServiceHelper.build
     *
     * Used by specific child
     * {@link org.nuiton.wikitty.jdbc.WikittyServiceJDBC}
     * {@link org.nuiton.wikitty.jdbc.WikittyServiceJPA}
     * {@link org.nuiton.wikitty.jdbc.WikittyServiceHbase}
     */
    protected WikittyServiceStorage() {
    }

    public WikittyServiceStorage(WikittyExtensionStorage extensionStorage,
            WikittyStorage wikittyStorage,
            WikittySearchEngine searchEngine) {
        this.extensionStorage = extensionStorage;
        this.wikittyStorage = wikittyStorage;
        this.searchEngine = searchEngine;
    }

    public WikittySearchEngine getSearchEngine() {
        return searchEngine;
    }

    public WikittyExtensionStorage getExtensionStorage() {
        return extensionStorage;
    }

    public WikittyStorage getWikittyStorage() {
        return wikittyStorage;
    }

    /*
     * @see org.nuiton.wikitty.WikittyService#addWikittyServiceListener(org.nuiton.wikitty.WikittyListener, org.nuiton.wikitty.WikittyService.ServiceListenerType)
     */
    @Override
    public void addWikittyServiceListener(WikittyListener listener, ServiceListenerType type) {
        throw new UnsupportedOperationException("Can't add listener on " + WikittyServiceStorage.class.getName());
    }

    /*
     * @see org.nuiton.wikitty.WikittyService#removeWikittyServiceListener(org.nuiton.wikitty.WikittyListener, org.nuiton.wikitty.WikittyService.ServiceListenerType)
     */
    @Override
    public void removeWikittyServiceListener(WikittyListener listener, ServiceListenerType type) {
        throw new UnsupportedOperationException("Can't remove listener on " + WikittyServiceStorage.class.getName());
    }

    @Override
    public String login(String login, String password) {
        log.warn("login asked, but there is no security service");
        return null;
    }

    @Override
    public void logout(String securityToken) {
        log.warn("logout asked, but there is no security service");
    }

    @Override
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        return true;
    }

    @Override
    public boolean canDelete(String securityToken, String wikittyId) {
        return true;
    }

    @Override
    public boolean canRead(String securityToken, String wikittyId) {
        return true;
    }

    protected void checkConstraint(Collection<Wikitty> wikitties) {
        for(Wikitty w : wikitties) {
            for(WikittyExtension ext : w.getExtensions()) {
                for (String fieldName : ext.getFieldNames()) {
                    FieldType type = ext.getFieldType(fieldName);
                    if (type.isNotNull()) {
                        if (null == w.getFieldAsObject(ext.getName(), fieldName)) {
                            throw new WikittyException(String.format(
                                    "Field '%s' must not be null", fieldName));
                        }
                    }
                }
            }
        }
    }

    @Override
    public WikittyEvent store(String securityToken,
            Collection<Wikitty> wikitties, boolean force) {
        if (!(wikitties instanceof Set)) {
            // use all time Set to prevent duplicated wikitty in collection
            wikitties = new LinkedHashSet<Wikitty>(wikitties);
        }
        checkConstraint(wikitties);

        // update/store extension if necessary
        Set<WikittyExtension> allExtensions = new LinkedHashSet<WikittyExtension>();
        for (Wikitty w : wikitties) {
            // collect all extensions used by all wikitties
            allExtensions.addAll(w.getExtensions());
        }

        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            // try to commit command
            WikittyEvent extUpdate =
                    getExtensionStorage().store(tx, allExtensions);
            WikittyEvent wikUpdate =
                    getWikittyStorage().store(tx, wikitties, force);
            getSearchEngine().store(tx, wikitties, force);

            WikittyEvent result = new WikittyEvent(this);
            // prepare update client response
            result.add(extUpdate);
            result.add(wikUpdate);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't store wikitty", eee);
        }
    }

    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            List<String> result = getExtensionStorage().getAllExtensionIds(tx);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't retrieve all extension's ids", eee);
        }
    }

    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            List<String> result = getExtensionStorage()
                    .getAllExtensionsRequires(tx, extensionName);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException(String.format(
                    "Can't retrieve all required extension for %s", extensionName), eee);
        }
    }

    /**
     * Check some constraint on extension
     * <li> extension name (ex: '#--AAA' is invalide name)
     * <li> extension version (ex: '-1.0' is invalide version)
     * <li> extension field name (ex: '=na' is invalide name)
     *
     * @see WikittyUtil#extensionNamePattern
     * @see WikittyUtil#extensionFieldNamePattern
     * @param exts
     */
    protected void checkExtension(Collection<WikittyExtension> exts) {
        for (WikittyExtension ext : exts) {
            // if extension version is invalide, raise exception
            if (WikittyUtil.versionGreaterThan("0", ext.getVersion())) {
                throw new WikittyException(String.format(
                        "Invalide extension version %s", ext.getVersion()));
            }

            // if extension name is invalide, raise exception
            if (!ext.getName().matches(WikittyUtil.extensionNamePattern)) {
                throw new WikittyException(String.format(
                        "Invalide extension name %s", ext.getName()));
            }

            // if field name is invalide, raise exception
            for(String fieldName : ext.getFieldNames()) {
                if (!fieldName.matches(WikittyUtil.extensionFieldNamePattern)) {
                    throw new WikittyException(String.format(
                        "Invalide extension field name '%s' for extension '%s'",
                        fieldName, ext.getName()));
                }
            }
        }
    }

    @Override
    public WikittyEvent storeExtension(String securityToken,
            Collection<WikittyExtension> exts) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            // check extension
            checkExtension(exts);
            WikittyEvent result =
                    getExtensionStorage().store(tx, exts);
            
            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't store extensions", eee);
        }
    }

    @Override
    public WikittyEvent deleteExtension(
            String securityToken, Collection<String> extNames) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            // check that all extensions are not used
            for(String name : extNames) {
                // only name are stored in index, search only on name
                Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, name).criteria();
                criteria.setEndIndex(0); // result is not use, just numFound
                PagedResult<String> wikittyWithExt = findAllByCriteria(
                        securityToken, criteria);
                int numFound = wikittyWithExt.getNumFound();
                if (numFound > 0) {
                    throw new WikittyException(String.format(
                            "Can't delete %s extension, this extension"
                            + " is in used by %s wikitty",
                            name, numFound));
                }
            }

            WikittyEvent result =
                    getExtensionStorage().delete(tx, extNames);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't delete extensions", eee);
        }
    }

    @Override
    public WikittyExtension restoreExtension(
            String securityToken, String extensionId) {
         WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            //split the id to ensure that version is normalized
            String name = WikittyExtension.computeName(extensionId);
            String version = WikittyExtension.computeVersion(extensionId);

            WikittyExtension result = getExtensionStorage().restore(tx, name, version);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't restore extensions", eee);
        }
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            String version = getExtensionStorage().getLastVersion(tx, name);
            if (version == null) {
                if (txBeginHere) {
                    tx.commit();
                }
                return null;
            }


            WikittyExtension result = getExtensionStorage().restore(tx, name, version);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't store extensions", eee);
        }
    }

    /**
     * restore one wikitty
     * @param securityToken
     * @param id
     * @return
     */
    protected Wikitty restore(String securityToken, String id) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            if (id == null) {
                return null;
            }
            if (!getWikittyStorage().exists(tx, id)) {
                // object doesn't exist, we return null
                return null;
            }

            if (getWikittyStorage().isDeleted(tx, id)) {
                // object deleted, we return null
                return null;
            }
            Wikitty result = getWikittyStorage().restore(tx, id);
            if (result != null) {
                result = upgradeData(securityToken, result);
            }
            
            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't store extensions", eee);
        }
    }

    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        List<Wikitty> result = new ArrayList<Wikitty>();
        WikittyTransaction transaction = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!transaction.isStarted()) {
                transaction.begin();
                txBeginHere = true;
            }

            for (String id : ids) {
                Wikitty w = restore(securityToken, id);
                // on l'ajoutde tout le temps, meme si w est nul lorsqu'il y a
                // une demande et qu'elle echoue on ajout
                // bien null, pour qu'il y ait une correspondance 1 pour 1
                // avec la demande
                result.add(w);
            }
            if (txBeginHere) {
                transaction.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (transaction != null) {
                transaction.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (transaction != null) {
                transaction.rollback();
            }
            throw new WikittyException("Can't restore wikitty", eee);
        }
    }

    /**
     * Upgrade wikitty but not saved it.
     * @param securityToken
     * @param wikitty
     * @return
     */
    protected Wikitty upgradeData(String securityToken, Wikitty wikitty) {
        Wikitty result = wikitty;

        Collection<WikittyExtension> extensions = wikitty.getExtensions();
        for (WikittyExtension extension : extensions) {
            String extensionName = extension.getName();

            if (log.isDebugEnabled()) {
                log.debug("extensionName="  + extensionName);
            }
            
            WikittyExtension currentExtension = extension;
            String currentExtensionVersion = currentExtension.getVersion();

            WikittyExtension lastExtension = restoreExtensionLastVersion(
                    securityToken, extensionName);
            String lastExtensionVersion = lastExtension.getVersion();

            if (log.isDebugEnabled()) {
                log.debug("lastExtensionVersion="  + lastExtensionVersion);
            }

            WikittyExtensionMigration migration =
                    WikittyExtensionMigration.migrationRegistry.get(extensionName);
            if (migration == null) {
                migration = defaultExtensionMigration;
            }

            // Loop on between extension in wikitty and last version
            while(WikittyUtil.versionGreaterThan(lastExtensionVersion, currentExtensionVersion)) {

                // Get extension after the current version
                String nextExtensionVersion =
                        WikittyUtil.incrementMajorRevision(currentExtensionVersion);
                String nextExtensionId =
                        WikittyExtension.computeId(extensionName, nextExtensionVersion);
                WikittyExtension nextExtension = restoreExtension(
                        securityToken, nextExtensionId);

                if (log.isDebugEnabled()) {
                    log.debug("currentExtensionVersion="  + currentExtensionVersion);
                    log.debug("nextExtensionVersion="  + nextExtensionVersion);
                }

                // Test if extension is never use in this version
                if(nextExtension != null) {
                    result = migration.migrate(this, result,
                            currentExtension, nextExtension);
                    currentExtension = nextExtension;
                }

                // Follow
                currentExtensionVersion = nextExtensionVersion;
            }
        }
        
        return result;
    }

    @Override
    public WikittyEvent delete(String securityToken,
            Collection<String> ids) throws WikittyException {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            // work only on valid id
            Collection<Wikitty> storedWikitties = new LinkedHashSet<Wikitty>();
            // copy ids because we can remove some element, and modify it
            // use set to prevent id duplication and preformance (contains method call)
            Set<String> idSet = new LinkedHashSet<String>(ids);
            for (Iterator<String> i = idSet.iterator(); i.hasNext();) {
                String id = i.next();
                // test if wikitty exists
                if (!getWikittyStorage().exists(tx, id)) {
                    // don't exist, remove this id in id list
                    i.remove();
                    // go to the next id, because this id doesn't exist and can't
                    // be used in tree
                    continue;
                }
                if (getWikittyStorage().isDeleted(tx, id)) {
                    // already deleted, remove this id in id list
                    i.remove();
                    // go to the next id, because this id already deleted and can't
                    // be used in tree
                    continue;
                }

                // Store node with have deleted node as parent
                Criteria criteria =
                        Search.query().eq(WikittyTreeNode.
                        FQ_FIELD_WIKITTYTREENODE_PARENT, id).criteria();
                List<String> wikittyNodesId = findAllByCriteria(
                        securityToken, criteria).getAll();
                for (String wikittyNodeId : wikittyNodesId) {
                    if (!idSet.contains(wikittyNodeId)) {
                        Wikitty treeNode = restore(
                                securityToken, wikittyNodeId);
                        WikittyTreeNodeHelper.setParent(treeNode, null);
                        storedWikitties.add(treeNode);
                    }
                }

                // Store node with have deleted child
                criteria = Search.query().eq(WikittyTreeNode.
                        FQ_FIELD_WIKITTYTREENODE_ATTACHMENT, id).criteria();
                wikittyNodesId = findAllByCriteria(
                        securityToken, criteria).getAll();
                for (String wikittyNodeId : wikittyNodesId) {
                    if (!idSet.contains(wikittyNodeId)) {
                        Wikitty treeNode = restore(
                                securityToken, wikittyNodeId);
                        WikittyTreeNodeHelper.removeAttachment(treeNode, id);
                        storedWikitties.add(treeNode);
                    }
                }
            }

            WikittyEvent eventDelete =
                    getWikittyStorage().delete(tx, idSet);
            getSearchEngine().delete(tx, idSet);

            WikittyEvent eventStore = store(securityToken, storedWikitties, false);

            WikittyEvent result = new WikittyEvent(this);
            result.add(eventDelete);
            result.add(eventStore);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't delete wikitty", eee);
        }
    }

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     */
    @Override
    public WikittyEvent clear(String securityToken) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            getSearchEngine().clear(tx);
            WikittyEvent eventWik = getWikittyStorage().clear(tx);
            WikittyEvent eventExt = getExtensionStorage().clear(tx);

            WikittyEvent result = new WikittyEvent(this);
            result.add(eventWik);
            result.add(eventExt);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't clear all data", eee);
        }
    }

    /**
     * Assume that this PagedResult contains wikitty id as result and
     * return new PagedResult with Wikitty instance
     */
    @Override
    public PagedResult<String> findAllByCriteria(
            String securityToken, Criteria criteria) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            PagedResult<String> result =
                    getSearchEngine().findAllByCriteria(tx, criteria);
            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Error during find", eee);
        }
    }

    @Override
    public String findByCriteria(String securityToken, Criteria criteria) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            // TODO poussin 20101222 perhaps we must do a copy of criteria
            // before modify it with index, to prevent bad side effect ?
            criteria.setFirstIndex(0).setEndIndex(1);
            PagedResult<String> pages = findAllByCriteria(
                    securityToken, criteria);

            String result = null;
            if (pages.size() > 0) {
                result = pages.getFirst();
            }

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Error during find", eee);
        }
    }

    @Override
    public WikittyEvent deleteTree(String securityToken, String thesaurusId) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            List<String> allTreeNodeId = getRecursiveTreeNodeId(
                    securityToken, thesaurusId);
            WikittyEvent result = delete(securityToken, allTreeNodeId);
           
            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't delete tree", eee);
        }
    }

    /**
     * Get recursive id of tree node children of {@code treeNodeId}.
     * 
     * @param securityToken security token
     * @param transaction transaction
     * @param treeNodeId tree node id
     * @return all id of {@code treeNodeId}'s children 
     */
    protected List<String> getRecursiveTreeNodeId(String securityToken, String treeNodeId) {
        Search search = Search.query();
        search = search.eq(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT, treeNodeId);
        Criteria criteria = search.criteria();

        PagedResult<String> childTreeNodeIds = findAllByCriteria(securityToken, criteria);
        List<String> treeNodeIds = new ArrayList<String>();
        treeNodeIds.add(treeNodeId);
        for (String childTreeNodeId : childTreeNodeIds.getAll()) {
            List<String> subTreeNodeIds = getRecursiveTreeNodeId(securityToken, childTreeNodeId);
            treeNodeIds.addAll(subTreeNodeIds);
        }
        return treeNodeIds;
    }

    @Override
    public TreeNodeResult<String> findTreeNode(String securityToken,
            String wikittyId, int depth, boolean count, Criteria filter) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            TreeNodeResult<String> result = null;
            Wikitty w = restore(securityToken, wikittyId);
            if(w != null) {
                result = getSearchEngine().findAllChildrenCount(
                        tx, wikittyId, depth, count, filter);
            }

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't restore children", eee);
        }
    }

    @Override
    public Wikitty restoreVersion(String securityToken,
            String wikittyId, String version) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * This method do some commit during execution. Transaction must be not
     * started when we call it.
     * @param securityToken
     */
    @Override
    public void syncSearchEngine(final String securityToken) {
        final WikittyTransaction tx = WikittyTransaction.get();
        if (tx.isStarted()) {
            throw new WikittyException("Transaction must be not started for syncSearchEngine method");
        }

        boolean txBeginHere = false;
        try {
            final int numberForCommit = 1000;
            final WikittySearchEngine searchEngine = getSearchEngine();
            final List<Wikitty> wikitties = new ArrayList<Wikitty>(numberForCommit);

            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }
            searchEngine.clear(tx);
            if (txBeginHere) {
                tx.commit();
            }
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }
            getWikittyStorage().scanWikitties(tx, new WikittyStorage.Scanner() {
                int count = 0;

                @Override
                public void scan(String wikittyId) {
                    Wikitty wikitty = restore(securityToken, wikittyId);
                    Date deleteDate = wikitty.getDeleteDate();
                    if(deleteDate == null) {
                        count ++;
                        wikitties.add(wikitty);

                        if(count == numberForCommit) {
                            // Reindex
                            searchEngine.store(tx, wikitties, true);
                            tx.commit();
                            // Reinit
                            count = 0;
                            wikitties.clear();
                            tx.begin();
                        }
                    }
                }
            });

            // Last wikitties
            searchEngine.store(tx, wikitties, true);
            if (txBeginHere) {
                tx.commit();
            }
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't sync searchable index with data", eee);
        }
    }

    /**
     * Question:
     * <li> on ne force pas le store de wikitty, ils vont donc avoir potentiellement
     * des versions differentes sur plusieurs serveurs, est-ce problematique ?
     * <li> on ne passe pas la date de suppression des wikitties, ils vont donc
     * avoir différente date de suppression sur différent serveur, est-ce problematique ?
     *
     *
     * @param securityToken
     * @param events
     * @return
     */
    @Override
    public WikittyEvent replay(
            String securityToken, List<WikittyEvent> events, boolean force) {
        // indique qu'il faut vider la base avant de faire les ajouts
        boolean mustClear = false;

        // tous les objets a sauver
        Map<String, Wikitty> toAddWikitty = new LinkedHashMap<String, Wikitty>();
        // tous les id a supprimer
        Map<String, Date> toRemoveWikitty = new LinkedHashMap<String, Date>();

        // toutes les extensions a sauver
        Set<WikittyExtension> toAddExt = new LinkedHashSet<WikittyExtension>();
        // toutes les extensions a supprimer
        Set<String> toDeleteExt = new LinkedHashSet<String>();

        // recherche un event avec un clear pour ne pas jouer des events inutiles
        // recherche un store + delete du meme wikitty
        // recherche le dernier store du wikitty
        for (WikittyEvent e : events) {
            // check clear must be the first, if event have clear and other type
            // clear is all time play first
            if (e.getType().contains(
                    WikittyEvent.WikittyEventType.CLEAR_WIKITTY)
                    || e.getType().contains(
                    WikittyEvent.WikittyEventType.CLEAR_EXTENSION)) {
                mustClear = true;
                toAddWikitty.clear();
                toRemoveWikitty.clear();
                toAddExt.clear();
            }
            if (e.getType().contains(WikittyEvent.WikittyEventType.PUT_WIKITTY)) {
                for (Wikitty w : e.getWikitties().values()) {
                    toAddWikitty.put(w.getId(), w);
                }
            }
            if (e.getType().contains(WikittyEvent.WikittyEventType.REMOVE_WIKITTY)) {
                for (Map.Entry<String, Date> entry : e.getRemoveDate().entrySet()) {
                    toAddWikitty.remove(entry.getKey());
                    toRemoveWikitty.put(entry.getKey(), entry.getValue());
                }
            }
            if (e.getType().contains(WikittyEvent.WikittyEventType.PUT_EXTENSION)) {
                for (WikittyExtension ext : e.getExtensions().values()) {
                    toAddExt.add(ext);
                }
            }
            if (e.getType().contains(WikittyEvent.WikittyEventType.REMOVE_EXTENSION)) {
                for (String extName : e.getDeletedExtensions()) {
                    toDeleteExt.add(extName);
                }
            }
        }

        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            WikittyEvent result = new WikittyEvent(this);
            if (mustClear) {
                WikittyEvent eventClear = clear(securityToken);
                result.add(eventClear);
            }
            WikittyEvent eventStoreExtension =
                    storeExtension(securityToken, toAddExt);
            result.add(eventStoreExtension);

            WikittyEvent eventDeleteExtension =
                    deleteExtension(securityToken, toDeleteExt);
            result.add(eventDeleteExtension);

            WikittyEvent eventStoreWikitty =
                    store(securityToken, toAddWikitty.values(), force);
            result.add(eventStoreWikitty);

            WikittyEvent eventDeleteWikitty =
                    delete(securityToken, toRemoveWikitty.keySet());
            result.add(eventDeleteWikitty);

            if(txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't replay data", eee);
        }

    }

    @Override
    public boolean exists(String securityToken, String wikittyId) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            boolean result = getWikittyStorage().exists(null, wikittyId);

            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't test existance", eee);
        }
    }

    @Override
    public boolean isDeleted(String securityToken, String wikittyId) {
        WikittyTransaction tx = WikittyTransaction.get();
        boolean txBeginHere = false;
        try {
            if (!tx.isStarted()) {
                tx.begin();
                txBeginHere = true;
            }

            boolean result = getWikittyStorage().isDeleted(tx, wikittyId);
            if (txBeginHere) {
                tx.commit();
            }
            return result;
        } catch (WikittyException ex) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw ex;
        } catch (Exception eee) {
            if (tx != null && tx.isStarted()) {
                tx.rollback();
            }
            throw new WikittyException("Can't test existance", eee);
        }
    }

}
