/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaConnectionProviderHardCoded.java 2245 2011-04-14 12:47:09Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.6.2/topia-persistence/src/test/java/org/nuiton/topia/framework/TopiaConnectionProviderHardCoded.java $
 * %%
 * Copyright (C) 2004 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.framework;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.cfg.Environment;
import org.hibernate.connection.ConnectionProvider;
import org.hibernate.connection.ConnectionProviderFactory;
import org.hibernate.util.PropertiesHelper;
import org.hibernate.util.ReflectHelper;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Properties;

/**
 * Customized connection provider.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.5.3
 */
public class TopiaConnectionProviderHardCoded implements ConnectionProvider {

    private String url;

    private Properties connectionProps;

    private Integer isolation;

    private final ArrayList pool = new ArrayList();

    private int poolSize;

    private int checkedOut = 0;

    private boolean autocommit;

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TopiaConnectionProviderHardCoded.class);

    @Override
    public void configure(Properties props) throws HibernateException {
        String driverClass = props.getProperty(Environment.DRIVER);

        poolSize = PropertiesHelper.getInt(Environment.POOL_SIZE, props, 20); //default pool size 20
        log.info("Using Hibernate built-in connection pool (not for production use!)");
        log.info("Hibernate connection pool size: " + poolSize);

        autocommit = PropertiesHelper.getBoolean(Environment.AUTOCOMMIT, props);
        log.info("autocommit mode: " + autocommit);

        isolation = PropertiesHelper.getInteger(Environment.ISOLATION, props);
        if (isolation != null)
            log.info("JDBC isolation level: " + Environment.isolationLevelToString(isolation));

        if (driverClass == null) {
            log.warn("no JDBC Driver class was specified by property " + Environment.DRIVER);
        } else {
            try {
                // trying via forName() first to be as close to DriverManager's semantics
                Class.forName(driverClass);
            } catch (ClassNotFoundException cnfe) {
                try {
                    ReflectHelper.classForName(driverClass);
                } catch (ClassNotFoundException e) {
                    String msg = "JDBC Driver class not found: " + driverClass;
                    log.error(msg, e);
                    throw new HibernateException(msg, e);
                }
            }
        }

        // use a dummy directory to make sure only the connection provider knows
        // the real directory where db is and then make sure hibernate always
        // use the connection provider...
        String directory =
                props.getProperty(TopiaConnectionProviderTest.TEST_URL);

        url = directory;
//        url = props.getProperty(Environment.URL);

//        if (url == null) {
//            String msg = "JDBC URL was not specified by property " + Environment.URL;
//            log.error(msg);
//            throw new HibernateException(msg);
//        }

        connectionProps = ConnectionProviderFactory.getConnectionProperties(props);

        log.info("using driver: " + driverClass + " at URL: " + url);
        // if debug level is enabled, then log the password, otherwise mask it
        if (log.isDebugEnabled()) {
            log.info("connection properties: " + connectionProps);
        } else if (log.isInfoEnabled()) {
            log.info("connection properties: " + PropertiesHelper.maskOut(connectionProps, "password"));
        }
    }

    @Override
    public Connection getConnection() throws SQLException {
        if (log.isTraceEnabled())
            log.trace("total checked-out connections: " + checkedOut);

        synchronized (pool) {
            if (!pool.isEmpty()) {
                int last = pool.size() - 1;
                if (log.isTraceEnabled()) {
                    log.trace("using pooled JDBC connection, pool size: " + last);
                }
                checkedOut++;
                Connection pooled = (Connection) pool.remove(last);
                if (isolation != null)
                    pooled.setTransactionIsolation(isolation.intValue());
                if (pooled.getAutoCommit() != autocommit)
                    pooled.setAutoCommit(autocommit);
                return pooled;
            }
        }

        log.debug("opening new JDBC connection");
        Connection conn = DriverManager.getConnection(url, connectionProps);
        if (isolation != null) conn.setTransactionIsolation(isolation);
        if (conn.getAutoCommit() != autocommit) conn.setAutoCommit(autocommit);

        if (log.isDebugEnabled()) {
            log.debug("created connection to: " + url + ", Isolation Level: " + conn.getTransactionIsolation());
        }
//		if ( log.isTraceEnabled() )
        checkedOut++;

        return conn;
    }

    @Override
    public void closeConnection(Connection conn) throws SQLException {
//        if ( log.isDebugEnabled() )
        checkedOut--;

        synchronized (pool) {
            int currentSize = pool.size();
            if (currentSize < poolSize) {
                if (log.isTraceEnabled()) {
                    log.trace("returning connection to pool, pool size: " + (currentSize + 1));
                }
                pool.add(conn);
                return;
            }
        }

        log.debug("closing JDBC connection");

        conn.close();
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }

    public void close() {

        log.info("cleaning up connection pool: " + url);

        Iterator iter = pool.iterator();
        while (iter.hasNext()) {
            try {
                ((Connection) iter.next()).close();
            } catch (SQLException sqle) {
                log.warn("problem closing pooled connection", sqle);
            }
        }
        pool.clear();

    }

    @Override
    public boolean supportsAggressiveRelease() {
        return false;
    }
}
