/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaSQLQuery.java 2186 2011-01-25 17:08:07Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.5.4/topia-persistence/src/main/java/org/nuiton/topia/framework/TopiaSQLQuery.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.framework;

import org.hibernate.jdbc.Work;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaRuntimeException;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

/**
 * An executor of sql query which permits to obtain a single result via
 * the method {@link #findSingleResult(TopiaContextImplementor)}
 * or a multiple result with method {@link #findMultipleResult(TopiaContextImplementor)}.
 *
 * @param <O> the type of result data
 * @since 2.5
 */
public abstract class TopiaSQLQuery<O> {

    /**
     * Prepare the statement used to do the sql query.
     *
     * @param connection jdbc connection to use
     * @return the statement containing the query to execute
     * @throws SQLException if any problem
     */
    protected abstract PreparedStatement prepareQuery(Connection connection) throws SQLException;

    /**
     * given a result set, extract the data.
     *
     * @param set the result set
     * @return the data extracted from the current set, or {@code null}
     * @throws SQLException if any prob
     */
    protected abstract O prepareResult(ResultSet set) throws SQLException;

    /**
     * Obtain a single result from the builded sql query.
     *
     * @param tx the transaction used to execute the query.
     * @return the single result or {@code null} if none found.
     * @throws TopiaException for any pb
     */
    public O findSingleResult(TopiaContextImplementor tx) throws TopiaException {
        final List<O> result = new ArrayList<O>();

        tx.getHibernate().doWork(new Work() {

            @Override
            public void execute(Connection connection) throws SQLException {

                PreparedStatement ps = prepareQuery(connection);

                try {
                    ResultSet set = ps.executeQuery();

                    if (set.next()) {
                        O singleResult = prepareResult(set);
                        if (singleResult != null) {
                            result.add(singleResult);
                        }
                    }
                } catch (Exception e) {
                    throw new TopiaRuntimeException("Could not execute query", e);
                } finally {
                    ps.close();
                }
            }
        });
        return result.isEmpty() ? null : result.get(0);
    }

    /**
     * Obtain a multiple results fro the builded sql query.
     *
     * @param tx the transaction used to execute the query.
     * @return the list of results (the list is empty if non result is found).
     * @throws TopiaException for any pb
     */
    public List<O> findMultipleResult(TopiaContextImplementor tx) throws TopiaException {
        final List<O> result = new ArrayList<O>();

        tx.getHibernate().doWork(new Work() {

            @Override
            public void execute(Connection connection) throws SQLException {

                PreparedStatement ps = prepareQuery(connection);
                try {
                    ResultSet set = ps.executeQuery();

                    while (set.next()) {
                        O singleResult = prepareResult(set);

                        if (singleResult != null) {
                            result.add(singleResult);
                        }
                    }
                } catch (Exception e) {
                    throw new TopiaRuntimeException("Could not execute query", e);
                } finally {
                    ps.close();
                }
            }
        });
        return result;
    }

}
