/* *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/

/* *
 * TopiaDAO.java
 *
 * Created: 30 déc. 2005 03:00:57
 *
 * @author poussin
 * @version $Revision: 1815 $
 *
 * Last update: $Date: 2010-02-26 19:48:25 +0100 (ven., 26 févr. 2010) $
 * by : $Author: tchemit $
 */
package org.nuiton.topia.persistence;

import java.security.Permission;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.nuiton.topia.TopiaException;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaQuery;

/**
 * @author poussin
 *
 * @param <Entity>
 */
public interface TopiaDAO<Entity extends TopiaEntity> {

    /**
     * When TopiaContextImpl create the TopiaDAOHibernate, it must call this
     * method just after.
     * 
     * @param context context
     * @param entityClass entity class
     * @throws TopiaException  if any pb while init
     */
    void init(TopiaContextImplementor context, Class<Entity> entityClass)
            throws TopiaException;

    /**
     * Return class of entity managed by this DAO
     * @return entity managed by this DAO
     */
    Class<Entity> getEntityClass();

    /**
     * @return Returns the context.
     */
    TopiaContextImplementor getContext();

    /**
     *  Appelé lorsque le context a eu un commit de fait.
     *  
     *  TODO il faudrait peut-etre plutot faire un commit a deux phase
     *  car plusieurs type de persistence peuvent etre melangé, et il 
     *  serait bien que si l'une echoué les autres echoue aussi.
     *  
     * @throws TopiaException if any pb while commit
     */
    void commitTransaction() throws TopiaException;

    /**
     * Appelé lorsque le context a eu un rollback de fait.
     * 
     * @throws TopiaException if any pb while rollback
     */
    void rollbackTransaction() throws TopiaException;

    /**
     * Construit une nouvelle instance de l'objet géré par ce DAO
     * @param properties la liste des propriétés que doit avoir l'objet créé
     * les arguments vont par paire (propertyName, value)
     * @return un nouvel objet
     * @throws TopiaException si un problème est rencontré durant l'instanciation
     * @throws IllegalArgumentException Si le nombre on le type des arguments
     * n'est pas bon ou que le type ou le nom d'une propriété est fausse
     */
    Entity create(Object... properties) throws TopiaException;

    /**
     * Construit une nouvelle instance de l'objet géré par ce DAO
     * @param properties la liste des propriétés que doit avoir l'objet créé
     * @return un nouvel objet
     * @throws TopiaException si un problème est rencontré durant l'instanciation
     * @throws IllegalArgumentException Si le nombre on le type des arguments
     * n'est pas bon ou que le type ou le nom d'une propriété est fausse
     */
    Entity create(Map<String, Object> properties) throws TopiaException;

    /**
     * Permet d'ajouter ou de mettre a jour un objet. Cela permet d'ajouter
     * par exemple un objet provenant d'un autre context mais du meme type
     * de DAO
     * @param e l'entite a ajouter ou mettre a jour
     * @return l'entity passé en paramètre.
     * @throws TopiaException if any pb while updating datas
     */
    Entity update(Entity e) throws TopiaException;

    void delete(Entity e) throws TopiaException;

    Entity findByTopiaId(String k) throws TopiaException;

    List<Entity> findAll() throws TopiaException;

    /**
     * Recuperation de tous les ids en base pour le type d'entite du dao.
     *
     * @return la liste de tous les ids
     * @throws TopiaException si pb en base
     */
    List<String> findAllIds() throws TopiaException;

    List<Entity> findAllWithOrder(String... propertyNames)
            throws TopiaException;

    /**
     * Compte le nombre d'entites en base
     * 
     * @return le nombre total d'entites existantes
     * @throws TopiaException if any pb while getting datas
     */
    long size() throws TopiaException;

    /**
     * Crée une requete basé sur l'entité lié au DAO.
     * Résultat attendu : "FROM Entity"
     *
     * @return une nouvelle TopiaQuery vide. (uniquement avec le From sur le
     *         type d'entité)
     */
    TopiaQuery createQuery();

    /**
     * Crée une requête basé sur l'entité lié au DAO et lui assigne un alias
     * valable dans la requête..
     *
     * Résultat attendu : "FROM Entity AS entityAlias"
     *
     * @param entityAlias alias permettant de manipuler l'entité dans la requête
     * @return une nouvelle TopiaQuery
     */
    TopiaQuery createQuery(String entityAlias);

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une
     * entité correspondant à la requête.
     *
     * @param query la requête
     * @return l'entité correspondant à la recherche ou null si aucune entité 
     * n'a été trouvée
     * @throws TopiaException if any pb while getting datas
     * @see org.nuiton.topia.framework.TopiaQuery#executeToEntity(org.nuiton.topia.TopiaContext, java.lang.Class)
     */
    Entity findByQuery(TopiaQuery query) throws TopiaException;

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une
     * liste d'entités correspondant à la requête.
     *
     * @param query la requête
     * @return la liste d'entités correspondant à la recherche
     * @throws TopiaException if any pb while getting datas
     * @see org.nuiton.topia.framework.TopiaQuery#executeToEntityList(org.nuiton.topia.TopiaContext, java.lang.Class)
     */
    List<Entity> findAllByQuery(TopiaQuery query) throws TopiaException;

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une
     * map d'entités correspondant à la requête. La clé de la map étant le
     * topiaId de l'entité.
     *
     * @param query la requête
     * @return la map d'entités correspondant à la recherche
     * @throws TopiaException if any pb while getting datas
     * @see org.nuiton.topia.framework.TopiaQuery#executeToEntityMap(org.nuiton.topia.TopiaContext, java.lang.Class)
     */
    Map<String, Entity> findAllMappedByQuery(TopiaQuery query) throws TopiaException;

    /**
     * Execute une requête basé sur l'entité du DAO. Permet de récupérer une
     * map d'entités correspondant à la requête. Le type et le nom de la
     * propriété utilisé comme clé de la map doit être passé en argument.
     *
     * @param <K> type de la clé de la map
     * @param query la requête
     * @param keyName nom de la propriété de l'entité utilisée comme clé
     * @param keyClass type de la propriété de l'entité utilisée comme clé
     * @return la map d'entités correspondant à la recherche
     * @throws TopiaException if any pb while getting datas
     * @see org.nuiton.topia.framework.TopiaQuery#executeToEntityMap(org.nuiton.topia.TopiaContext, java.lang.Class)
     */
    <K> Map<K, Entity> findAllMappedByQuery(TopiaQuery query,
            String keyName, Class<K> keyClass) throws TopiaException;

    /**
     * Recherche la classe en utilisant la cle naturelle, chaque champs de la
     * cle naturelle est une entre de la map passe en argument.
     * 
     * @param keys la liste des champs de la cle naturelle avec leur valeur
     * @return l'entite trouvé
     * @throws TopiaException
     */
    Entity findByPrimaryKey(Map<String, Object> keys) throws TopiaException;

    /**
     * Recherche la classe en utilisant la cle naturelle, si la cle naturelle
     * est composé de plusieurs champs alors les arguments passés doivent être
     * dans l'ordre de declaration dans le fichier de mapping
     * 
     * @param k l'objet cle naturelle de la classe
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    Entity findByPrimaryKey(Object... k) throws TopiaException;

    Entity findByProperty(String propertyName, Object value)
            throws TopiaException;

    /**
     * 
     * @param propertyName
     * @param value
     * @param others les autres proprietes doivent aller par 2 propertyName,
     *        value
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    Entity findByProperties(String propertyName, Object value,
                            Object... others) throws TopiaException;

    Entity findByProperties(Map<String, Object> properties) throws TopiaException;

    List<Entity> findAllByProperty(String propertyName, Object value)
            throws TopiaException;

    /**
     * 
     * @param propertyName
     * @param value
     * @param others les autres proprietes doivent aller par 2 propertyName,
     *        value
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    List<Entity> findAllByProperties(String propertyName, Object value,
            Object... others) throws TopiaException;

    List<Entity> findAllByProperties(Map<String, Object> properties)
            throws TopiaException;

    /*
     * Find Contains
     */
    /**
     * Cherche et renvoie la première entité trouvée dont les propriétés en clé
     * de Map contiennent toutes les valeurs de la Collection.
     * 
     * @param properties 
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    Entity findContainsProperties(Map<String, Collection> properties) throws
            TopiaException;

    /**
     * Cherche et renvoie la première entité trouvée dont la propriété
     * propertyName contient values, ainsi de suite avec others.
     * 
     * @param propertyName 
     * @param values 
     * @param others 
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    Entity findContainsProperties(String propertyName, Collection values,
            Object... others) throws TopiaException;

    /**
     * Cherche et renvoie la première entité trouvée dont les propriétés en clé
     * de Map contiennent toutes les valeurs de la Collection.
     * 
     * @param properties 
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    List<Entity> findAllContainsProperties(Map<String, Collection> properties)
            throws TopiaException;

    /**
     * Cherche et renvoie toutes les entités trouvées dont la propriété
     * propertyName contient values, ainsi de suite avec others.
     * 
     * @param propertyName the name of the property
     * @param values 
     * @param others 
     * @return l'entité trouvé
     * @throws TopiaException if any pb while getting datas
     */
    List<Entity> findAllContainsProperties(String propertyName,
            Collection values, Object... others) throws TopiaException;

    /**
     * Find usages of the given {@code entity} in the entities of the given
     * {@code type}.
     *
     * @param type the type of entity to search
     * @param entity the entity
     * @param <U> tthe type of entity to search
     * @return the list of entities of the given type which uses the given
     *         entity
     * @throws TopiaException if any problem while getting data
     * @since 2.3.0
     */
    <U extends TopiaEntity> List<U> findUsages(Class<U> type, Entity entity)
            throws TopiaException;

    /**
     * Find all usages of the given {@code entity}.
     * 
     * @param entity the entity
     * @return the dictionnary of usages of the given entities (keys are entity
     * usage container, values are the list of this type of entity to use the
     * given entity).
     * @throws TopiaException if any pb while getting data
     * @since 2.3.0
     */

    Map<Class<?>, List<? extends TopiaEntity>> findAllUsages(Entity entity)
            throws TopiaException;
    
    /**
     * Retourne les permissions a verifier pour l'acces a l'entite pour le
     * service Taas.
     * 
     * @param topiaId topiaId d'une entite
     * @param actions 
     * @return la liste des permissions
     * @throws TopiaException if any pb while getting datas
     */
    List<Permission> getRequestPermission(String topiaId, int actions)
            throws TopiaException;

    /* Adders */
    void addTopiaEntityListener(TopiaEntityListener listener);

    void addTopiaEntityVetoable(TopiaEntityVetoable vetoable);

    /* Removers */
    void removeTopiaEntityListener(TopiaEntityListener listener);

    void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable);
} //TopiaDAO

