/*
 * *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.topia.generator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelOperation;

import java.io.Serializable;
import java.util.Collection;

/**
 * Created: 14 déc. 2009
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @version $Revision: 1716 $
 *          <p/>
 *          Mise a jour: $Date: 2009-12-16 21:31:10 +0100 (mer. 16 déc. 2009) $ par :
 *          $Author: tchemit $
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.EntityImplTransformer"
 */
public class EntityImplTransformer extends ObjectModelTransformerToJava {

    /**
     * Logger
     */
    private static final Log log = LogFactory.getLog(EntityImplTransformer.class);

    @Override
    public void transformFromClass(ObjectModelClass clazz) {
        if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
            return;
        }
        // On ne génère pas le impl si l'entité a des opérations qui ne sont pas seulement pour le DAO
        if (clazz.getOperations().size() > 0 && !hasOnlyDAOOperations(clazz)) {
            return;
        }
        //De même, on ne génère pas le impl si il y a des opérations venant des
        // superclasses non implémentées
        for (ObjectModelOperation otherOp : clazz.getAllOtherOperations(false)) {
            if (otherOp.isAbstract()) {
                return;
            }
        }
        String clazzName = clazz.getName();
        String clazzFQN = clazz.getQualifiedName();
        ObjectModelClass result;
        if (isAbstract(clazz)) {
            result = createAbstractClass(clazzName + "Impl", clazz.getPackageName());
        } else {
            result = createClass(clazzName + "Impl", clazz.getPackageName());
        }
        setDocumentation(result, "Implantation des operations pour l'entité " + clazzName + ".");
        setSuperClass(result, clazzFQN + "Abstract");
        addInterface(result, Serializable.class);
        addInterface(result, clazzFQN);

    }

    protected boolean isAbstract(ObjectModelClass clazz) {
        if (clazz.isAbstract()) {
            return true;
        }

        //Une classe peut être abstraite si elle a des méthodes définies dans
        // ses superinterface et non implantées dans ses superclasses
        Collection<ObjectModelOperation> allInterfaceOperations = clazz.getAllInterfaceOperations(true);
        allInterfaceOperations.removeAll(clazz.getAllOtherOperations(true));
        for (ObjectModelOperation op : allInterfaceOperations) {
            boolean implementationFound = false;
            for (ObjectModelClass superClazz : clazz.getSuperclasses()) {
                for (ObjectModelOperation matchingOp : superClazz.getOperations(op.getName())) {
                    implementationFound = (op.equals(matchingOp) && !matchingOp.isAbstract());
                    if (implementationFound) {
                        break;
                    }
                }
                if (implementationFound) {
                    break;
                }
            }
            if (!implementationFound) {
                if (log.isDebugEnabled()) {
                    log.debug(clazz.getName() + " : abstract operation " + op);
                }
                return true;
            }
        }
        return false;
    }

    /**
     * Detect if the clazz has only operations for DAO implementation.
     * These operations are identified with the stereotype <<dao>>.
     *
     * @param clazz The ObjectModelClass with all operations.
     * @return true if there is only dao operations, false if there is no operations or some without
     *         dao stereotype.
     */
    public static boolean hasOnlyDAOOperations(ObjectModelClass clazz) {
        boolean res = true;
        Collection<ObjectModelOperation> operations = clazz.getOperations();
        if (operations.size() == 0) {
            res = false;
        }
        for (ObjectModelOperation op : operations) {
            res &= op.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DAO);
        }
        return res;
    }
}
