/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.reflect;

import jaxx.runtime.JAXXObjectDescriptor;

import java.util.Arrays;

/**
 * Mirrors the class <code>java.lang.Class</code>.  JAXX uses <code>ClassDescriptor</code> instead of <code>Class</code>
 * almost everywhere so that it can handle circular dependencies (there can't be a <code>Class</code> object for an uncompiled
 * JAXX or Java source file, and a compiler must be allow references to symbols in uncompiled source files in order to handle
 * circular dependencies).
 */
public abstract class ClassDescriptor {

    private String name;
    private String packageName;
    private String superclass;
    private String[] interfaces;
    private boolean isInterface;
    private boolean isArray;
    private String componentType;
    private JAXXObjectDescriptor jaxxObjectDescriptor;
    private ClassLoader classLoader;
    private MethodDescriptor[] methodDescriptors;
    private FieldDescriptor[] fieldDescriptors;

    public abstract MethodDescriptor getDeclaredMethodDescriptor(String name, ClassDescriptor... parameterTypes) throws NoSuchMethodException;

    public abstract FieldDescriptor getDeclaredFieldDescriptor(String name) throws NoSuchFieldException;

    ClassDescriptor(String name, String packageName, String superclass, String[] interfaces, boolean isInterface,
            boolean isArray, String componentType, JAXXObjectDescriptor jaxxObjectDescriptor,
            ClassLoader classLoader, MethodDescriptor[] methodDescriptors, FieldDescriptor[] fieldDescriptors) {
        this.name = name;
        this.packageName = packageName;
        this.superclass = superclass;
        this.interfaces = interfaces;
        this.isInterface = isInterface;
        this.isArray = isArray;
        this.componentType = componentType;
        this.jaxxObjectDescriptor = jaxxObjectDescriptor;
        this.classLoader = classLoader;
        this.methodDescriptors = methodDescriptors;
        this.fieldDescriptors = fieldDescriptors;
    }

    public String getName() {
        return name;
    }

    public String getSimpleName() {
        int dot = name.lastIndexOf(".");
        return dot == -1 ? name : name.substring(dot + 1);
    }

    public String getPackageName() {
        return packageName;
    }

    public ClassDescriptor getSuperclass() {
        try {
            return superclass != null ? ClassDescriptorLoader.getClassDescriptor(superclass, getClassLoader()) : null;
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public ClassDescriptor[] getInterfaces() {
        try {
            ClassDescriptor[] result = new ClassDescriptor[interfaces.length];
            for (int i = 0; i < result.length; i++) {
                result[i] = ClassDescriptorLoader.getClassDescriptor(interfaces[i], getClassLoader());
            }
            return result;
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public boolean isInterface() {
        return isInterface;
    }

    public boolean isArray() {
        return isArray;
    }

    public ClassDescriptor getComponentType() {
        try {
            return componentType != null ? ClassDescriptorLoader.getClassDescriptor(componentType, getClassLoader()) : null;
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public ClassLoader getClassLoader() {
        return classLoader;
    }

    public MethodDescriptor[] getMethodDescriptors() {
        return methodDescriptors;
    }

    public MethodDescriptor getMethodDescriptor(String name, ClassDescriptor... parameterTypes) throws NoSuchMethodException {
        for (MethodDescriptor methodDescriptor : methodDescriptors) {
            if (methodDescriptor.getName().equals(name) && methodDescriptor.getParameterTypes().length == parameterTypes.length && Arrays.equals(methodDescriptor.getParameterTypes(), parameterTypes)) {
                return methodDescriptor;
            }
        }
        throw new NoSuchMethodException("Could not find method " + name + "(" + Arrays.asList(parameterTypes) + ") in " + getName());
    }

    public FieldDescriptor[] getFieldDescriptors() {
        return fieldDescriptors;
    }

    public FieldDescriptor getFieldDescriptor(String name) throws NoSuchFieldException {
        for (FieldDescriptor fieldDescriptor : fieldDescriptors) {
            if (fieldDescriptor.getName().equals(name)) {
                return fieldDescriptor;
            }
        }
        throw new NoSuchFieldException("Could not find field " + name + " in " + getName());
    }

    public JAXXObjectDescriptor getJAXXObjectDescriptor() {
        return jaxxObjectDescriptor;
    }

    public boolean isAssignableFrom(ClassDescriptor descriptor) {
        while (descriptor != null) {
            if (descriptor == this) {
                return true;
            }
            for (ClassDescriptor anInterface : descriptor.getInterfaces()) {
                if (anInterface == this) {
                    return true;
                }
            }
            descriptor = descriptor.getSuperclass();
        }
        return false;
    }

    @Override
    public String toString() {
        return "ClassDescriptor[" + getName() + "]";
    }
}
