/*
 * #%L
 * Pollen :: Vote Counting
 * 
 * $Id: ServiceVoteCountingImpl.java 3122 2012-01-30 20:43:30Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1.1/pollen-votecounting/src/main/java/org/chorem/pollen/votecounting/services/ServiceVoteCountingImpl.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.services;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.votecounting.business.Choice;
import org.chorem.pollen.votecounting.business.CondorcetMethod;
import org.chorem.pollen.votecounting.business.Context;
import org.chorem.pollen.votecounting.business.Group;
import org.chorem.pollen.votecounting.business.Method;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.business.PercentageMethod;
import org.chorem.pollen.votecounting.business.StandardMethod;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.dto.VoteToChoiceDTO;
import org.chorem.pollen.votecounting.dto.VotingGroupDTO;
import org.chorem.pollen.votecounting.dto.VotingPersonDTO;
import org.chorem.pollen.votecounting.utils.Utils;

/**
 * Implémentation du service de dépouillement.
 *
 * @author fdesbois
 * @version $Id: ServiceVoteCountingImpl.java 2697 2009-08-11 09:39:09Z nrannou
 *          $
 */
public class ServiceVoteCountingImpl implements ServiceVoteCounting {
    /**
     * Identifiant du groupe courant
     */
    private String currentIdGroup;
    /**
     * Booléen déterminant si le dépouillement se fera sur les groupes
     */
    private boolean isByGroup;
    /**
     * Contexte de dépouillement
     */
    private Context context;
    /** log. */
    private static final Log log = LogFactory
            .getLog(ServiceVoteCountingImpl.class);

    /**
     * Constructeur
     */
    public ServiceVoteCountingImpl() {
        this.currentIdGroup = "";
        this.context = null;
    }

    /**
     * Service de dépouillement par votes (personnes)
     *
     * @param poll : sondage
     * @return resultat
     */
    @Override
    public VoteCountingResultDTO executeVoteCounting(PollDTO poll) {
        this.isByGroup = false;
        return this.execute(poll);
    }

    /**
     * Service de dépouillement par groupes
     *
     * @param poll : sondage
     * @return resultat
     */
    @Override
    public VoteCountingResultDTO executeGroupCounting(PollDTO poll) {
        this.isByGroup = true;
        return this.execute(poll);
    }

    /**
     * Execution du dépouillement
     *
     * @param poll : sondage
     * @return resultat
     */
    private VoteCountingResultDTO execute(PollDTO poll) {
        if (log.isInfoEnabled()) {
            log.info("Dépouillement (byGroup=" + isByGroup + ") du sondage "
                    + poll.getPollId());
        }

        // Création et remplissage du contexte
        this.createContext(poll.getVoteCounting());
        this.fillContext(poll);
        
        // Execution
        if (!context.executeCounting()) {
            return null;
        }
        List<ChoiceDTO> resChoices = context.executeStats();

        VoteCountingResultDTO result = new VoteCountingResultDTO();
        result.setNbVotes(Utils.calculateNbVotes(poll.getVotingGroups(),
                this.isByGroup));
        result.setTypeVoteCounting(poll.getVoteCounting());
        result.setByGroup(this.isByGroup);
        result.setIdPoll(poll.getPollId());
        result.setChoices(resChoices);
        return result;
    }

    /**
     * Création du contexte en fonction du type de dépouillement
     *
     * @param type : type de dépouillement
     */
    private void createContext(VoteCountingType type) {
        Method method = null;
        switch (type) {
        case NORMAL:
            method = new StandardMethod();
            break;
        case PERCENTAGE:
            method = new PercentageMethod();
            break;
        case CONDORCET:
            method = new CondorcetMethod();
            break;
        case NUMBER:
            method = new NumberMethod();
            break;
        default:
            method = new StandardMethod();
        }
        this.context = new Context(method, this.isByGroup);
    }

    /**
     * Remplissage du contexte
     *
     * @param poll : Sondage
     */
    private void fillContext(PollDTO poll) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout poll : " + poll.getPollId());
        }
        for (PollChoiceDTO choice : poll.getChoices()) {
            if (choice.isHidden()) {
                this.context.addHiddenChoice(choice);
            } else {
                this.context.addChoice(choice);
            }
        }
        for (Object o : poll.getVotingGroups()) {
            VotingGroupDTO group = (VotingGroupDTO) o;
            this.routeGroup(group);
        }
    }

    /**
     * Parcours d'un groupe et de ses votants
     *
     * @param group : groupe lié au sondage
     */
    private void routeGroup(VotingGroupDTO group) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout group : " + group.getIdGroup() + " _ weight="
                    + group.getWeight());
        }
        this.context.addGroup(group.getIdGroup(), group.getWeight());
        for (Object o : group.getVotingPersons()) {
            VotingPersonDTO person = (VotingPersonDTO) o;
            this.currentIdGroup = group.getIdGroup();
            this.routePerson(person);
        }
    }

    /**
     * Parcours d'un votant et de ses choix
     *
     * @param person : personne ayant voté
     */
    private void routePerson(VotingPersonDTO person) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout person : " + person.getVotingId() + " _ weight="
                    + person.getWeight());
        }
        for (VoteToChoiceDTO vote : person.getChoices()) {
            this.addVoteToContext(vote, person.getWeight(), person
                    .getVotingId());
        }
    }

    /**
     * Ajout du vote d'une personne au contexte pour un choix
     *
     * @param vote : vote lié au choix
     * @param weight : poids de la personne
     */
    private void addVoteToContext(VoteToChoiceDTO vote, double weight,
            String votingID) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout vote : " + vote.getValue() + " _ choice="
                    + vote.getIdChoice());
        }
        Choice choice = this.context.getChoice(vote.getIdChoice());
        Group group = choice.getGroup(this.currentIdGroup);
        group.addVote(vote.getValue(), weight, votingID);
    }

}