/*
 * #%L
 * Pollen :: Services
 * $Id: PollService.java 3771 2013-04-22 19:54:47Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.3/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollDateChoice;
import org.chorem.pollen.bean.PollImageChoice;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceDAO;
import org.chorem.pollen.business.persistence.ChoiceType;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.business.persistence.PreventRuleDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListDAO;
import org.chorem.pollen.entities.PollenBinderHelper;
import org.chorem.pollen.services.PollenServiceFunctions;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollAccountNotFound;
import org.chorem.pollen.services.exceptions.PollChoiceNotFoundException;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;
import org.nuiton.util.beans.Binder;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import static org.nuiton.i18n.I18n.l_;

public class PollService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(PollService.class);

    public static final String THUMB_PREFIX = "thumb_";

    /**
     * Retrieve a Poll for edition. It is loaded from database if {@code pollUid}
     * is defined, otherwise a new instance is returned. The {@code clone}
     * parameter will return the {@code poll} without any technical properties (topiaId)
     * and ensure business ids (accountUid, pollUid) with new values. If {@code userId}
     * matches an existing {@link UserAccount}, the resulting poll creator will
     * be attached to it.
     *
     * @param pollUid     Uid of the poll to edit, if not defined, a new poll is instanciated
     * @param userAccount Optinal User account to attach to the creator
     * @param clone       Flag to copy or not ids in case of poll cloning
     * @return the Poll ready for edition (no longer linked to the topia context)
     */
    public Poll getPollEditable(String pollUid,
                                UserAccount userAccount,
                                boolean clone) throws PollNotFoundException {
        Poll result;

        if (StringUtils.isEmpty(pollUid)) {

            // creates a new poll
            PollDAO pollDAO = getDAO(Poll.class);
            result = newInstance(pollDAO);

            PollenConfiguration configuration = getConfiguration();

            // default values from configuration
            result.setChoiceType(configuration.getDefaultChoiceType());
            result.setVoteCountingType(configuration.getDefaultVoteCountingType());
            result.setPollType(configuration.getDefaultPollType());
            result.setPollVoteVisibility(configuration.getDefaultPollVoteVisibility());
            result.setPollCommentVisibility(configuration.getDefaultPollCommentVisibility());

            // use default today as begin date (see http://chorem.org/issues/891)
            result.setBeginDate(serviceContext.getCurrentTime());

            // Initialize creator of the poll
            PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
            PollAccount creator = newInstance(pollAccountDAO);

            if (userAccount != null) {

                // Link the creator with the user
                creator.setVotingId(userAccount.getDisplayName());
                creator.setEmail(userAccount.getEmail());
                creator.setUserAccount(userAccount);
            }
            result.setCreator(creator);
        } else {

            // obtains a copy of an existing poll

            // load for sure existing poll
            Poll poll = getExistingPollByPollId(pollUid);

            PollDAO pollDAO = getDAO(Poll.class);
            result = newInstance(pollDAO);

            PollenBinderHelper.simpleCopy(poll, result, !clone);
            if (clone) {
                // reset id for clone case
                result.setPollId(null);

                // when cloning no more closed poll (see http://chorem.org/issues/892)
                result.setClosed(false);

                // remove end date (otherwise can't vote any longer)
                result.setEndDate(null);
            }

            // -- Creator -- //
            PollAccount creatorLoaded = poll.getCreator();
            PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
            PollAccount creatorEditable = newInstance(pollAccountDAO);
            result.setCreator(creatorEditable);

            PollenBinderHelper.simpleCopy(creatorLoaded, creatorEditable, !clone);
            if (clone) {
                // reset id for clone case
                creatorEditable.setAccountId(null);
            }

            if (creatorEditable.getUserAccount() == null) {
                // use the incoming userAccount
                creatorEditable.setUserAccount(userAccount);
            }

            // -- Choice -- //
            Function<Choice, Choice> choiceCreator =
                    PollenServiceFunctions.newChoiceCreator(poll.getChoiceType());
            Iterable<Choice> choices =
                    Iterables.transform(poll.getChoice(), choiceCreator);

            for (Choice choiceLoaded : choices) {
                if (clone) {
                    // reset id for clone case
                    choiceLoaded.setTopiaId(null);
                }
                result.addChoice(choiceLoaded);
            }

            // -- VotingList -- //
            VotingListDAO votingListDAO = getDAO(VotingList.class);
            PersonToListDAO personToListDAO = getDAO(PersonToList.class);
            for (VotingList votingListLoaded : poll.getVotingList()) {
                VotingList votingListEditable = newInstance(votingListDAO);
                result.addVotingList(votingListEditable);
                // Do not keep votingLists topiaId, to simplify the update will delete old votingLists and create new ones
                PollenBinderHelper.simpleCopy(
                        votingListLoaded, votingListEditable, false);

                for (PersonToList personToListLoaded : votingListLoaded.getPollAccountPersonToList()) {
                    PersonToList personToListEditable = newInstance(personToListDAO);
                    votingListEditable.addPollAccountPersonToList(personToListEditable);
                    // Do not keep personToLists topiaId, to simplify the update will delete old personToLists and create new ones
                    PollenBinderHelper.simpleCopy(
                            personToListLoaded, personToListEditable, false);

                    PollAccount personLoaded = personToListLoaded.getPollAccount();
                    PollAccount personEditable = newInstance(pollAccountDAO);
                    personToListEditable.setPollAccount(personEditable);
                    // copy the person, keeping topiaId is useless because we have the link with PersonToList
                    PollenBinderHelper.simpleCopy(
                            personLoaded, personEditable, false);
                    if (clone) {
                        // reset id for clone case
                        personEditable.setAccountId(null);
                    }
                }
            }

            // -- PreventRule -- //
            PreventRuleDAO preventRuleDAO = getDAO(PreventRule.class);
            for (PreventRule preventRuleLoaded : poll.getPreventRule()) {
                PreventRule preventRuleEditable = newInstance(preventRuleDAO);
                PollenBinderHelper.simpleCopy(
                        preventRuleLoaded, preventRuleEditable, !clone);
                result.addPreventRule(preventRuleEditable);
            }

            if (!clone) {
                // if cloned, then remove all votes (to be able to edit poll again see http://chorem.org/issues/892)
                // Load votes to have the correct size used to check if vote is started
                result.setVote(poll.getVote());
            }
        }
        return result;
    }

    public Map<String, Object> pollToMap(Poll poll, Binder<Poll, Poll> binder) {

        Map<String, Object> map = binder.obtainProperties(
                poll,
                Poll.PROPERTY_TITLE,
                Poll.PROPERTY_POLL_ID,
                Poll.PROPERTY_DESCRIPTION
        );
        Locale l = serviceContext.getLocale();
        if (poll.getBeginDate() == null) {
            map.put(Poll.PROPERTY_BEGIN_DATE, l_(l, "pollen.common.undefined"));
        } else {
            map.put(Poll.PROPERTY_BEGIN_DATE, decorateDate(poll.getBeginDate()));
        }
        if (poll.getEndDate() == null) {
            map.put(Poll.PROPERTY_END_DATE, l_(l, "pollen.common.undefined"));
        } else {
            map.put(Poll.PROPERTY_END_DATE, decorateDate(poll.getEndDate()));
        }
        map.put(Poll.TOPIA_CREATE_DATE,
                decorateDateTime(poll.getTopiaCreateDate()));

        String addingchoiceText;
        if (poll.isChoiceAddAllowed()) {

            Date beginDate = poll.getBeginChoiceDate();
            Date endDate = poll.getEndChoiceDate();

            if (beginDate == null && endDate == null) {

                // can always add choices
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesAlways");
            } else if (beginDate == null) {
                // until enddate
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesTo",
                           decorateDate(endDate));
            } else if (endDate == null) {

                // from begin date
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesFrom",
                           decorateDate(beginDate));
            } else {
                addingchoiceText =
                        l_(l, "pollen.common.addingChoicesContent",
                           decorateDate(beginDate),
                           decorateDate(endDate));
            }
        } else {
            addingchoiceText = l_(l, "pollen.common.unauthorized");
        }
        map.put("addingChoices", addingchoiceText);
        map.put("id", poll.getTopiaId());
        return map;
    }

    public Poll createPoll(Poll poll) {

        PollDAO pollDAO = getDAO(Poll.class);
        Poll result = createWithProperties(
                pollDAO,
                Poll.PROPERTY_POLL_ID, serviceContext.generateId(),
                Poll.PROPERTY_POLL_VOTE_VISIBILITY, poll.getPollVoteVisibility(),
                Poll.PROPERTY_POLL_COMMENT_VISIBILITY, poll.getPollCommentVisibility());

        // check max number choice authorized (can not be more than
        // XXX-fdesbois-2012-04-11 : don't know why the maxNbChoice must be
        // forced to the choice size, it seems useless. The case maxNbChoice = 0
        // must be available (equivalent to undefined)
//        if (poll.getMaxChoiceNb() < 1
//            || poll.getMaxChoiceNb() > poll.sizeChoice()) {
//            poll.setMaxChoiceNb(poll.sizeChoice());
//        }

        result.setAnonymousVoteAllowed(poll.isAnonymousVoteAllowed());
        result.setChoiceAddAllowed(poll.isChoiceAddAllowed());
        result.setBeginChoiceDate(poll.getBeginChoiceDate());
        result.setBeginDate(poll.getBeginDate());
        result.setChoiceType(poll.getChoiceType());
        result.setContinuousResults(poll.isContinuousResults());
        result.setDescription(poll.getDescription());
        result.setEndDate(poll.getEndDate());
        result.setEndChoiceDate(poll.getEndChoiceDate());
        result.setPollType(poll.getPollType());
        result.setMaxChoiceNb(poll.getMaxChoiceNb());
        result.setPublicResults(poll.isPublicResults());
        result.setTitle(poll.getTitle());
        result.setVoteCountingType(poll.getVoteCountingType());

        // create creator
        PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
        PollAccount creatorCreated = createWithProperties(
                pollAccountDAO,
                PollAccount.PROPERTY_ACCOUNT_ID, serviceContext.generateId());
        PollAccount creator = poll.getCreator();
        creatorCreated.setVotingId(creator.getVotingId());
        creatorCreated.setEmail(creator.getEmail());
        creatorCreated.setUserAccount(creator.getUserAccount());

        result.setCreator(creatorCreated);

        // -- Choice -- //
        for (Choice choice : poll.getChoice()) {
            saveChoice(result, choice);
        }

        // -- PreventRule -- //
        for (PreventRule choice : poll.getPreventRule()) {
            savePreventRule(result, choice);
        }

        // -- VotingList -- //
        try {
            for (VotingList votingList : poll.getVotingList()) {
                // Do not set emailService, all emails will be send during 
                // onPollCreated
                saveVotingList(result, votingList, null);
            }
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Can't save votingLists", e);
        }

        // feed notification
        // XXX-fdesbois-2012-04-12 : if the feed is the same for everybody maybe
        // not publish restricted poll that can't be accessed by everybody
        PollFeedService pollFeedService = newService(PollFeedService.class);
        pollFeedService.onPollCreated(result);

        commitTransaction("Could not create poll " + poll.getTitle());

        // email notification
        EmailService emailService = newService(EmailService.class);
        emailService.onPollCreated(result);

        return result;
    }

    public Poll updatePoll(Poll poll) throws PollNotFoundException {

        Poll pollToUpdate = getEntityById(Poll.class, poll.getTopiaId());

        if (pollToUpdate == null) {
            throw new PollNotFoundException();
        }

        boolean voteStarted = pollToUpdate.sizeVote() > 0;

        PollAccount creatorToUpdate = pollToUpdate.getCreator();

        // FIXME-fdesbois-2012-04-05 : improve binder to avoid creator copy
        PollenBinderHelper.simpleCopy(poll, pollToUpdate, false);

        // -- Creator -- //
        PollAccount creator = poll.getCreator();
        creatorToUpdate.setUserAccount(creator.getUserAccount());
        creatorToUpdate.setVotingId(creator.getVotingId());
        creatorToUpdate.setEmail(creator.getEmail());

        // -- Choices -- //
        if (!voteStarted) {

            // Save all choices from source poll
            List<Choice> choicesAdded = Lists.newLinkedList();
            for (Choice choice : poll.getChoice()) {
                Choice choiceLoaded = saveChoice(pollToUpdate, choice);
                choicesAdded.add(choiceLoaded);
            }

            // removed choice will be deleted from db by composition relation
            pollToUpdate.clearChoice();

            // re-add choices in incoming order.
            pollToUpdate.addAllChoice(choicesAdded);
        }

        // -- PreventRules -- //
        for (PreventRule preventRule : poll.getPreventRule()) {
            savePreventRule(pollToUpdate, preventRule);
        }

        // -- VotingLists -- //
        if (!voteStarted) {

            EmailService emailService = newService(EmailService.class);
            pollToUpdate.clearVotingList();
            try {
                for (VotingList votingList : poll.getVotingList()) {
                    saveVotingList(pollToUpdate, votingList, emailService);
                }
            } catch (TopiaException e) {
                throw new PollenTechnicalException("Can't save votingLists", e);
            }
        }

        commitTransaction("Could not update poll [" + poll.getTopiaId() + "]");

        return pollToUpdate;
    }

    public List<Poll> getPolls(TopiaPagerBean pager) {

        Preconditions.checkNotNull(pager);
        try {
            PollDAO dao = getDAO(Poll.class);
            List<Poll> result = dao.getPolls(pager);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain polls", e);
        }
    }

    public List<Poll> getCreatedPolls(TopiaPagerBean pager, UserAccount user) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);
        try {
            PollDAO dao = getDAO(Poll.class);
            List<Poll> result = dao.findCreatedPolls(pager, user);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain created polls", e);
        }
    }

    public List<Pair<Poll, PollAccount>> getInvitedPolls(
            TopiaPagerBean pager,
            UserAccount user) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        UserService userService = newService(UserService.class);
        UserAccount userToUse = userService.getEntityById(UserAccount.class,
                                                          user.getTopiaId());

        Preconditions.checkNotNull(userToUse);
        try {
            PollDAO pollDao = getDAO(Poll.class);
            List<Pair<Poll, PollAccount>> result =
                    pollDao.findInvitedPolls(pager, userToUse);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain invited polls", e);
        }
    }

    public List<Pair<Poll, PollAccount>> getParticipatedPolls(TopiaPagerBean pager,
                                                              UserAccount user) {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        UserService userService = newService(UserService.class);
        UserAccount userToUse =
                userService.getEntityById(UserAccount.class, user.getTopiaId());

        Preconditions.checkNotNull(userToUse);

        try {
            PollDAO pollDao = getDAO(Poll.class);
            List<Pair<Poll, PollAccount>> result =
                    pollDao.findParticipatedPolls(pager, userToUse);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not obtain running polls", e);
        }
    }

    public List<Poll> getRunningPollsWithEndTime(Date currentTime) {

        try {
            PollDAO dao = getDAO(Poll.class);
            List<Poll> results = dao.findRunningPollsWithEndTime(currentTime);
            return results;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not obtain running polls", e);
        }
    }

    public Poll getPollByPollId(String pollId) {

        Preconditions.checkNotNull(pollId);
        try {
            PollDAO dao = getDAO(Poll.class);
            Poll result = dao.findByPollId(pollId);
            return result;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not find poll with pollId '" + pollId + "'", e);
        }
    }

    public Poll getExistingPollByPollId(String pollId) throws PollNotFoundException {

        Poll result = getPollByPollId(pollId);

        if (result == null) {
            throw new PollNotFoundException();
        }
        return result;
    }

    /**
     * Retrieve a pollAccount with {@code accountId}. This pollAccount will be
     * attached to the given {@code userAccount} (only if not already attached).
     * If the {@code accountId} is undefined, a new instance of PollAccount will
     * be retrieved, otherwise a copy of the existing PollAccount is used. No
     * create or update is done here. The {@code poll} is used to retrieve a potential
     * existing vote with pollAccount linked to {@code userAccount}.
     *
     * @param accountId   Id of the existing account (optional)
     * @param userAccount UserAccount where account will be attached (optional)
     * @param poll        Poll where pollAccount will be added
     * @return the existing PollAccount or a new instance
     * @throws PollAccountNotFound if accountId is defined and doesn't match any
     *                             existing PollAccount
     */
    public PollAccount getPollAccountEditable(String accountId,
                                              UserAccount userAccount,
                                              Poll poll) throws PollAccountNotFound {

        PollAccountDAO dao = getDAO(PollAccount.class);

        PollAccount pollAccountLoaded = null;

        boolean withUserAccount = userAccount != null;

        if (StringUtils.isNotEmpty(accountId) || withUserAccount) {

            if (poll.getCreator().getAccountId().equals(accountId)) {

                // let's use the creator account
                pollAccountLoaded = poll.getCreator();

            } else if (poll.isPollFree()) {
                try {
                    pollAccountLoaded =
                            dao.findVoterPollAccount(poll.getPollId(),
                                                     accountId,
                                                     userAccount);
                } catch (TopiaException e) {
                    throw new PollenTechnicalException(e);
                }
            } else {
                // try to find pollAccount from the list of participants
                try {

                    pollAccountLoaded =
                            dao.findRestrictedPollAccount(poll.getPollId(),
                                                          accountId,
                                                          userAccount);
                } catch (TopiaException e) {
                    throw new PollenTechnicalException(e);
                }
            }
        }

        PollAccount result = null;

        if (pollAccountLoaded != null) {

            // copy the loaded pollAccount
            result = newInstance(dao);
            PollenBinderHelper.copy("", pollAccountLoaded, result, true);
            result.setUserAccount(pollAccountLoaded.getUserAccount());

            // Don't remove or update userAccount link if already set
            if (withUserAccount && result.getUserAccount() == null) {

                //TODO-tchemit-2012-08-13 Is UserAccount must be done here ? Not sure (vote time is better (and unique...))
                // link pollAccount to his userAccount
                result.setUserAccount(userAccount);
            }
        }

        if (result == null) {

            // create a new pollAccount linked to given userAccount (if not null)
            result = newInstance(dao);
            String votingId = userAccount != null ? userAccount.getDisplayName() : "";
            result.setVotingId(votingId);
            result.setUserAccount(userAccount);
        }
        return result;
    }

    public void deletePoll(String pollId) throws PollNotFoundException {

        // can not have an null nor empty pollId
        Preconditions.checkArgument(StringUtils.isNotBlank(pollId));

        Poll pollToDelete = getExistingPollByPollId(pollId);

        PollDAO dao = getDAO(Poll.class);
        delete(dao, pollToDelete);

        commitTransaction("Could not delete poll" + pollToDelete.getTitle());
    }

    public void closePoll(String pollId) throws PollNotFoundException {

        // can not have an null nor empty pollId
        Preconditions.checkArgument(StringUtils.isNotBlank(pollId));

        Poll poll = getExistingPollByPollId(pollId);

        if (!poll.isChoiceEmpty()) {
            for (Choice choice : poll.getChoice()) {
                choice.setValidate(true);
            }
        }
        Date now = serviceContext.getCurrentTime();
        if (poll.getEndDate() == null || now.before(poll.getEndDate())) {
            poll.setEndDate(now);
        }
        poll.setClosed(true);

        commitTransaction("Could not close poll " + poll.getTitle());
    }

    public void attachPoll(String pollId, UserAccount userAccount) throws PollNotFoundException {

        // can not have an null nor empty pollId
        Preconditions.checkArgument(StringUtils.isNotBlank(pollId));
        Preconditions.checkNotNull(userAccount);

        Poll poll = getExistingPollByPollId(pollId);

        // just link poll creator account to given user account
        PollAccount creator = poll.getCreator();
        creator.setUserAccount(userAccount);

        // fill votingId from user account
        if (StringUtils.isEmpty(creator.getVotingId())) {
            creator.setVotingId(userAccount.getDisplayName());
        }

        // fill email from user account
        if (StringUtils.isEmpty(creator.getEmail())) {
            creator.setEmail(userAccount.getEmail());
        }
        commitTransaction("Could not attach poll " + poll.getTitle() +
                          " to user account " + userAccount.getDisplayName());
    }

    public Choice getNewChoice(ChoiceType choiceType) {
        Choice result;
        switch (choiceType) {

            case DATE:
                result = new PollDateChoice();
                break;

            case IMAGE:
                result = new PollImageChoice();
                break;

            case TEXT:
            default:
                ChoiceDAO dao = getDAO(Choice.class);
                result = newInstance(dao);
        }
        return result;
    }

    public void addChoice(String pollId, Choice choice) throws PollNotFoundException {

        Preconditions.checkNotNull(pollId);
        Preconditions.checkNotNull(choice);

        Poll poll = getExistingPollByPollId(pollId);

        saveChoice(poll, choice);

        commitTransaction("Can't create new choice [" +
                          poll.getChoiceType() + "] for poll '" + pollId + "'");
    }

    protected Choice saveChoice(Poll poll, Choice choice) {

        ChoiceType choiceType = poll.getChoiceType();
        ChoiceDAO dao = getDAO(Choice.class);
        Choice choiceLoaded;

        boolean newChoice = choice.getTopiaId() == null;

        if (newChoice) {
            choiceLoaded = create(dao);
            poll.addChoice(choiceLoaded);

        } else {
            choiceLoaded = poll.getChoiceByTopiaId(choice.getTopiaId());
        }

        if (choiceType == ChoiceType.IMAGE) {

            PollImageChoice imageChoice = (PollImageChoice) choice;
            if (newChoice ||
                !ObjectUtils.equals(choice.getName(),
                                    choiceLoaded.getName())) {
                // copy image where it belong and generate the thumb
                // only if choice is to create or name has change (so image too...)
                try {
                    saveImages(poll, imageChoice);
                } catch (IOException e) {
                    throw new PollenTechnicalException(
                            "Could not create image choice", e);
                }
                // bind other fields
                imageChoice.toChoice(choiceLoaded);
            }

        } else if (choiceType == ChoiceType.DATE) {

            // date choice
            PollDateChoice dateChoice = (PollDateChoice) choice;
            dateChoice.toChoice(choiceLoaded);

        } else {

            // text choice
            choiceLoaded.setDescription(choice.getDescription());
            choiceLoaded.setValidate(choice.isValidate());
            choiceLoaded.setName(choice.getName());
        }

        return choiceLoaded;
    }

    public void deleteChoice(String pollId, String choiceId)
            throws PollNotFoundException, PollChoiceNotFoundException {

        Poll poll = getExistingPollByPollId(pollId);

        Choice choice = poll.getChoiceByTopiaId(choiceId);

        if (choice == null) {
            throw new PollChoiceNotFoundException();
        }

        ChoiceDAO dao = getDAO(Choice.class);

        poll.removeChoice(choice);
        delete(dao, choice);

        commitTransaction("Could not delete choice " + choice.getName());
    }

    /**
     * Obtain the location of an image choice.
     *
     * @param pollId   the id of the poll containing the image choice
     * @param choiceId the id of the choice to render
     * @return the location of image
     * @since 1.4
     */
    public File getPollChoiceImageFile(String pollId, String choiceId) {
        File imageDirectory = getConfiguration().getImageDirectory();
        File pollDirectory = new File(imageDirectory, pollId);

        File result = new File(pollDirectory, choiceId);

        return result;
    }

    /**
     * given the location of an image, gets his name as a thumb image.
     *
     * @param imagePath location of the image
     * @return location of the thumb of a given image
     * @since 1.4
     */
    public File getImageThumbFile(File imagePath) {

        File imageDirectory = imagePath.getParentFile();

        File result = new File(imageDirectory,
                               THUMB_PREFIX + imagePath.getName());

        return result;
    }

    /**
     * Sauvegarde des images d'un choix de type image.
     *
     * @param poll   le sondage contenant le choix
     * @param choice le choix à sauvegarder.
     * @throws IOException si un problème IO lors de la copie ou
     *                     génération de la miniature
     */
    public void saveImages(Poll poll, PollImageChoice choice) throws IOException {

        String pollId = poll.getPollId();
        String name = choice.getName();

        File pollChoiceImage = getPollChoiceImageFile(pollId, name);

        // copy image to correct directory

        String location = choice.getLocation();

        FileUtils.copyFile(new File(location), pollChoiceImage);

        // generate thumb
        generateThumbIfNeeded(pollChoiceImage);
    }

    public File generateThumbIfNeeded(File pollChoiceImage) throws IOException {

        File pollChoiceImageThumb = getImageThumbFile(pollChoiceImage);

        if (!pollChoiceImageThumb.exists()) {

            int width = 100;
            ImageIcon ii = new ImageIcon(pollChoiceImage.getAbsolutePath());
            Image image = ii.getImage();
            double imageRatio = (double) image.getHeight(null)
                                / (double) image.getWidth(null);
            int height = (int) (width * imageRatio);

            BufferedImage thumbImage = new BufferedImage(
                    width, height, BufferedImage.TYPE_INT_RGB);
            Graphics2D graphics2D = thumbImage.createGraphics();
            graphics2D.setRenderingHint(
                    RenderingHints.KEY_INTERPOLATION,
                    RenderingHints.VALUE_INTERPOLATION_BILINEAR);
            graphics2D.drawImage(image, 0, 0, width, height, null);

            ImageIO.write(thumbImage, "jpg", pollChoiceImageThumb);

            if (log.isDebugEnabled()) {
                log.debug("Thumbnail created: " +
                          pollChoiceImageThumb.getName() + " (size="
                          + pollChoiceImageThumb.length() + ")");
            }
        }
        return pollChoiceImageThumb;
    }

    public PersonToList getNewPersonToList(PollAccount pollAccount) {
        PersonToList result = newInstance(getDAO(PersonToList.class));
        result.setWeight(1);
        PollAccount pollAccount2 = newInstance(getDAO(PollAccount.class));
        result.setPollAccount(pollAccount2);
        if (pollAccount != null) {
//            pollAccount2.setComment(pollAccount.getComment());
            pollAccount2.setEmail(pollAccount.getEmail());
            pollAccount2.setUserAccount(pollAccount.getUserAccount());
            pollAccount2.setVotingId(pollAccount.getVotingId());
        }
        return result;
    }

    protected void saveVotingList(Poll poll,
                                  VotingList votingList,
                                  EmailService emailService) throws TopiaException {

        VotingListDAO votingListDAO = getDAO(VotingList.class);
        PersonToListDAO personToListDAO = getDAO(PersonToList.class);
        PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);

        String creatorEmail = poll.getCreator().getEmail();

        // Prepare the VotingList and add it to the poll
        VotingList result;
        if (votingList.getTopiaId() == null) {
            result = create(votingListDAO);
            poll.addVotingList(result);

        } else {
            result = poll.getVotingListByTopiaId(votingList.getTopiaId());
        }

        result.setName(votingList.getName());
        result.setWeight(votingList.getWeight());

        // Merge PersonToList
        List<PersonToList> personToListsUpdated = Lists.newArrayList();
        for (PersonToList personToList : votingList.getPollAccountPersonToList()) {

            PollAccount pollAccount = personToList.getPollAccount();

            PersonToList personToListLoaded;
            PollAccount pollAccountLoaded;
            if (personToList.getTopiaId() == null) {

                // creates a new PersonToList

                personToListLoaded = create(personToListDAO);

                // The model doesn't have any composition for this relation,
                // the link must be set in both objects
                personToListLoaded.setVotingList(result);

                // FIXME-fdesbois-2012-04-12 : find a better way to ensure accountId
                String accountId = pollAccount.getAccountId();
                String accountEmail = pollAccount.getEmail();

                if (ObjectUtils.equals(creatorEmail, pollAccount.getEmail())) {

                    // use the creator account
                    pollAccountLoaded = pollAccountDAO.findByAccountId(
                            poll.getCreator().getAccountId());
                    if (log.isInfoEnabled()) {
                        log.info(String.format(
                                "Use the creator account as restricted account [%s]", accountEmail));
                    }

                } else if (StringUtils.isBlank(accountId)) {

                    // creates a new pollAccount
                    pollAccountLoaded = createWithProperties(
                            pollAccountDAO,
                            PollAccount.PROPERTY_ACCOUNT_ID, serviceContext.generateId());

                    if (log.isInfoEnabled()) {
                        log.info(String.format(
                                "Create new account as restricted account [%s]", accountEmail));
                    }
                } else {

                    // reuse the existing account
                    pollAccountLoaded =
                            pollAccountDAO.findByAccountId(accountId);

                    if (log.isInfoEnabled()) {
                        log.info(String.format(
                                "Reuse existing account as restricted account [%s]", accountEmail));
                    }
                }

            } else {
                personToListLoaded = getEntityById(PersonToList.class,
                                                   personToList.getTopiaId());
                pollAccountLoaded = personToListLoaded.getPollAccount();
            }
            personToListsUpdated.add(personToListLoaded);

            personToListLoaded.setWeight(personToList.getWeight());
            pollAccountLoaded.setVotingId(pollAccount.getVotingId());

            boolean emailChanged = !Objects.equal(pollAccountLoaded.getEmail(),
                                                  pollAccount.getEmail());

            pollAccountLoaded.setEmail(pollAccount.getEmail());

            // Send email to the person if service is defined and email has changed
            if (emailService != null && emailChanged) {
                emailService.onRestrictedPersonAdded(poll, pollAccountLoaded);
            }

            // Update the account (otherwise a saving error will occurs if
            // already exists on delete cascade)
            PollAccount pollAccountUpdated = update(pollAccountDAO,
                                                    pollAccountLoaded);
            personToListLoaded.setPollAccount(pollAccountUpdated);
        }
        result.setPollAccountPersonToList(personToListsUpdated);
    }

    protected void savePreventRule(Poll poll, PreventRule preventRule) {

        PreventRuleDAO preventRuleDAO = getDAO(PreventRule.class);

        // For the moment only one rule by scope
        PreventRule preventRuleLoaded = poll.getPreventRuleByScope(
                preventRule.getScope());

        if (preventRuleLoaded == null) {
            preventRuleLoaded = create(preventRuleDAO);
            poll.addPreventRule(preventRuleLoaded);
            preventRuleLoaded.setScope(preventRule.getScope());
            preventRuleLoaded.setActive(true);
        }

        preventRuleLoaded.setMethod(preventRule.getMethod());
//        preventRuleLoaded.setOneTime(preventRule.isOneTime());
        preventRuleLoaded.setRepeated(preventRule.isRepeated());
        preventRuleLoaded.setSensibility(preventRule.getSensibility());
    }

    /**
     * Création d'une miniature.
     *
     * @param img         L'image à miniaturiser.
     * @param thumbCopied le chemin complet vers la miniature.
     * @param width       La largeur de la miniature.
     * @throws IOException if could not create thumb image
     */
    protected static void createThumbnail(File img,
                                          File thumbCopied,
                                          int width) throws IOException {

        ImageIcon ii = new ImageIcon(img.getAbsolutePath());
        Image image = ii.getImage();
        double imageRatio = (double) image.getHeight(null)
                            / (double) image.getWidth(null);
        int height = (int) (width * imageRatio);

        BufferedImage thumbImage = new BufferedImage(width, height,
                                                     BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics2D = thumbImage.createGraphics();
        graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                                    RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        graphics2D.drawImage(image, 0, 0, width, height, null);

        ImageIO.write(thumbImage, "jpg", thumbCopied);

        if (log.isDebugEnabled()) {
            log.debug("Thumbnail created: " + thumbCopied.getName() + " (size="
                      + thumbCopied.length() + ")");
        }
    }
}
