/*
 * #%L
 * Pollen :: Services
 * $Id: PollVoteCountingService.java 3595 2012-08-12 11:45:39Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollVoteCountingService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.votecounting.model.ChoiceScore;
import org.chorem.pollen.votecounting.model.GroupOfVoter;
import org.chorem.pollen.votecounting.model.GroupVoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteForChoice;
import org.chorem.pollen.votecounting.model.Voter;
import org.chorem.pollen.votecounting.strategy.VoteCountingStrategy;
import org.chorem.pollen.votecounting.strategy.VoteCountingStrategyProvider;

import java.util.Map;
import java.util.Set;

/**
 * New Poll vote counting service.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public class PollVoteCountingService extends PollenServiceSupport {

    /** log. */
    private static final Log log =
            LogFactory.getLog(PollVoteCountingService.class);

    /**
     * Provider of {@link VoteCountingStrategy}.
     * <p/>
     * <strong>Note:</strong> this provider should be injected via method
     * {@link #setProvider(VoteCountingStrategyProvider)}.
     */
    private VoteCountingStrategyProvider provider;

    public VoteCountingStrategyProvider getProvider() {
        if (provider == null) {
            provider = serviceContext.getVoteCountingStrategyProvider();
        }
        return provider;
    }

    public void setProvider(VoteCountingStrategyProvider provider) {
        this.provider = provider;
    }

    /**
     * Execute a vote counting for a given strategy (by his id) and the
     * given {@code voters}.
     *
     * @param strategyId the id of the vote counting strategy to use
     * @param voter      the votes to vote count
     * @return vote counting result
     */
    public VoteCountingResult voteCount(int strategyId, Set<Voter> voter) {

        Preconditions.checkNotNull(getProvider());
        Preconditions.checkNotNull(voter);

        VoteCountingStrategy strategy = getProvider().getStrategy(strategyId);
        Preconditions.checkNotNull(strategy);

        VoteCountingResult result = strategy.votecount(voter);
        return result;
    }

    /**
     * Execute a vote counting for a given strategy (by his id) and the
     * given {@code voters}.
     *
     * @param strategyId the id of the vote counting strategy to use
     * @param voter      the votes to vote count
     * @return vote counting result
     */
    public GroupVoteCountingResult voteCountByGroup(int strategyId, Set<Voter> voter) {

        Preconditions.checkNotNull(getProvider());
        Preconditions.checkNotNull(voter);

        VoteCountingStrategy strategy = getProvider().getStrategy(strategyId);
        Preconditions.checkNotNull(strategy);

        // Create a groupVoter including of the root voters
        GroupOfVoter group = GroupOfVoter.newVoter(null, 1.0, null, voter);


        Map<GroupOfVoter, VoteCountingResult> groupResults = Maps.newHashMap();
        voteCount(strategy, group, groupResults);

        // get result for main group (and remove it from groups)
        VoteCountingResult mainResult = groupResults.remove(group);

        GroupVoteCountingResult result = GroupVoteCountingResult.newResult(
                mainResult, groupResults);
        return result;
    }

    protected void voteCount(VoteCountingStrategy strategy,
                             GroupOfVoter group,
                             Map<GroupOfVoter, VoteCountingResult> results) {

        // all childs of this group
        Set<Voter> voters = group.getVoters();

        // treat before all his group childs
        for (Voter voter : voters) {
            if (voter instanceof GroupOfVoter) {

                // treat group child before all
                voteCount(strategy, (GroupOfVoter) voter, results);
            }
        }

        // once here, all childs has been treated, can votecount this group
        VoteCountingResult voteCountingResult = strategy.votecount(voters);

        // store the result for this group
        results.put(group, voteCountingResult);

        // result of the group is now the voteForChoice for it

        for (ChoiceScore choiceScore : voteCountingResult.getScores()) {
            VoteForChoice voteForChoice = VoteForChoice.newVote(
                    choiceScore.getChoiceId(),
                    //FIXME-tchemit-2012-06-26 Which is the value to set as choice for each result?
                    choiceScore.getScoreValue().doubleValue());
            group.addVoteForChoice(voteForChoice);
        }

    }
}
