/*
 * #%L
 * Pollen :: Persistence
 * $Id: PollDAOImpl.java 3761 2012-12-01 12:34:54Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.4/pollen-persistence/src/main/java/org/chorem/pollen/business/persistence/PollDAOImpl.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.business.persistence;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.tuple.Pair;
import org.chorem.pollen.PollenFunctions;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;
import org.nuiton.util.PagerBeanUtil;

import java.util.Date;
import java.util.List;
import java.util.Map;

public class PollDAOImpl<E extends Poll> extends PollDAOAbstract<E> {

    public List<E> getPolls(TopiaPagerBean pager) throws TopiaException {

        Preconditions.checkNotNull(pager);

        String countHql = "SELECT COUNT(*) FROM PollImpl";

        computeAndAddRecordsToPager(countHql, pager);

        String hql = "FROM PollImpl";

        List<E> result = findAllByQueryAndPager(hql, pager);
        return result;
    }

    public List<E> findCreatedPolls(TopiaPagerBean pager,
                                    UserAccount user) throws TopiaException {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        String countHql = "SELECT COUNT(*) FROM PollImpl " +
                          "WHERE creator.userAccount = :user";

        computeAndAddRecordsToPager(countHql, pager, "user", user);

        String hql = "FROM PollImpl WHERE creator.userAccount = :user";

        List<E> result = findAllByQueryAndPager(hql, pager, "user", user);
        return result;
    }

    public List<Pair<Poll, PollAccount>> findInvitedPolls(TopiaPagerBean pager,
                                                          UserAccount user) throws TopiaException {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        String hql = "SELECT p, l.pollAccount FROM PollImpl p " +
                     "LEFT JOIN p.votingList v " +
                     "LEFT JOIN v.pollAccountPersonToList l " +
                     "WHERE l.pollAccount.email = :email";
        List<Pair<Poll, PollAccount>> result = findAllWithPollAccounts(
                hql, pager, "email", user.getEmail());
        return result;
    }

    public List<Pair<Poll, PollAccount>> findParticipatedPolls(TopiaPagerBean pager,
                                                               UserAccount user) throws TopiaException {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        String hql = "SELECT p, v.pollAccount FROM VoteImpl v, PollImpl p " +
                     "WHERE v IN ELEMENTS(p.vote) " +
                     "AND v.pollAccount.userAccount = :user";

        List<Pair<Poll, PollAccount>> result = findAllWithPollAccounts(
                hql, pager, "user", user);
        return result;
    }

    public List<E> findRunningPollsWithEndTime(Date currentTime) throws TopiaException {

        String hql = "FROM PollImpl WHERE " +
                     "(endDate IS NOT NULL AND endDate > :currentTime) AND " +
                     "(beginDate IS NULL OR beginDate < :currentTime)";

        List<E> results = findAllByQuery(hql, "currentTime", currentTime);
        return results;
    }

    protected List<Pair<Poll, PollAccount>> findAllWithPollAccounts(String hql,
                                                                    TopiaPagerBean pager, Object... params)
            throws TopiaException {

        List<Object[]> queryResults = findAllByQuery(Object[].class, hql, params);

        Map<Poll, PollAccount> mapResult = Maps.newLinkedHashMap();
        for (Object[] row : queryResults) {
            Poll poll = (Poll) row[0];
            PollAccount pollAccount = (PollAccount) row[1];
            mapResult.put(poll, pollAccount);
        }
        long records = mapResult.size();
        pager.setRecords(records);
        PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);
        List<Map.Entry<Poll, PollAccount>> entries =
                Lists.newLinkedList(mapResult.entrySet());
        List<Pair<Poll, PollAccount>> result =
                Lists.transform(getPageFromList(entries, pager),
                                PollenFunctions.MAP_ENTRY_TO_PAIR_FUNCTION);
        return result;
    }

    public static <E> List<E> getPageFromList(List<E> elements,
                                              TopiaPagerBean pager) {

        PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);
        List<E> subList = PagerBeanUtil.getPage(elements, pager.getPageIndex(), pager.getPageSize());
        List<E> result = Lists.newArrayList(subList);

        return result;
    }

}
