/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataCommentConverter;
import org.chorem.pollen.business.dto.CommentDTO;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.CommentDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;

/**
 * Implémentation du service de gestion des commentaires.
 *
 * @author enema
 * @author rannou
 * @version $Id: ServiceCommentImpl.java 2862 2010-02-09 15:31:45Z jruchaud $
 */
public class ServiceCommentImpl implements ServiceComment {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private CommentDAO commentDAO = null;
    private DataCommentConverter converter = new DataCommentConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceCommentImpl.class);

    public ServiceCommentImpl() {
    }

    @Override
    public String createComment(CommentDTO comment) {
        TopiaContext transaction = null;
        String topiaId = "";
        try {
            transaction = rootContext.beginTransaction();

            commentDAO = PollenModelDAOHelper.getCommentDAO(transaction);

            Comment commentEntity = commentDAO.create();
            converter.setTransaction(transaction);
            converter.populateCommentEntity(comment, commentEntity);

            // FIXME do not call a Service from an other one
            ServicePollAccount spa = new ServicePollAccountImpl();
            commentEntity.setPollAccount(spa.createPollAccount(comment
                    .getPollAccountId(), null, null));

            topiaId = commentEntity.getTopiaId();
            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return "";
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<Comment> createComments(List<CommentDTO> comments,
            String pollId, TopiaContext transaction) {
        if (comments.isEmpty() || pollId.length() == 0) {
            return null;
        }

        List<Comment> result = null;
        try {
            commentDAO = PollenModelDAOHelper.getCommentDAO(transaction);
            converter.setTransaction(transaction);

            result = new ArrayList<Comment>();
            for (CommentDTO comment : comments) {
                Comment commentEntity = commentDAO.create();
                comment.setPollId(pollId);
                converter.populateCommentEntity(comment, commentEntity);
                result.add(commentEntity);
            }

            return result;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public boolean updateComment(CommentDTO comment) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            commentDAO = PollenModelDAOHelper.getCommentDAO(transaction);

            Comment commentEntity = commentDAO.findByTopiaId(comment.getId());

            converter.setTransaction(transaction);
            converter.populateCommentEntity(comment, commentEntity);

            commentDAO.update(commentEntity);
            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + comment.getId());
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public boolean deleteComment(String commentId) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            commentDAO = PollenModelDAOHelper.getCommentDAO(transaction);

            Comment commentEntity = commentDAO.findByTopiaId(commentId);

            commentDAO.delete(commentEntity);
            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity deleted: " + commentId);
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public CommentDTO findCommentById(String commentId) {
        TopiaContext transaction = null;
        CommentDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            commentDAO = PollenModelDAOHelper.getCommentDAO(transaction);

            Comment commentEntity = commentDAO.findByTopiaId(commentId);

            if (commentEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createCommentDTO(commentEntity);
            }

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<CommentDTO> selectComments(Map<String, Object> properties) {
        TopiaContext transaction = null;
        List<CommentDTO> results = null;
        List<Comment> commentEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            commentDAO = PollenModelDAOHelper.getCommentDAO(transaction);

            if (properties == null) {
                commentEntities = commentDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les commentaires ont été sélectionnés !");
                }
            } else {
                commentEntities = commentDAO.findAllByProperties(properties);
            }
            converter.setTransaction(transaction);
            results = converter.createCommentDTOs(commentEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }
}