package com.cybelia.sandra;

import com.cybelia.sandra.entities.*;
import com.cybelia.sandra.entities.trace.SuiviEtape;
import com.cybelia.sandra.entities.trace.SuiviLigneProduit;
import com.cybelia.sandra.entities.trace.SuiviLigneProduitDAO;
import com.cybelia.sandra.entities.trace.SuiviTour;
import com.cybelia.sandra.entities.trace.SuiviTourDAO;
import com.cybelia.sandra.entities.trace.SuiviUsine;
import com.cybelia.sandra.notifier.SandraNotifier;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImpl;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.taas.entities.TaasUser;
import org.nuiton.util.ApplicationConfig;

import static com.cybelia.sandra.ibu.csv.CSVReaderGeneric.DELIM;
import java.text.DecimalFormat;

/**
 * @author letellier
 */
public class SandraHelper {

    protected static final Log log = LogFactory.getLog(SandraHelper.class);

    public static final String SUIVI_TOUR_PREFIX = "_";
    public static final String SUIVI_TOUR_EXTENTION = ".txt";

    /**
     * Do the difference between the two dates in argument. The result is a number
     * of seconds between the two dates.
     *
     * @param beginDate first date
     * @param endDate second date
     * @return a number of seconds between beginDate and endDate
     */
    public static int getDifferenceInMilliseconds(Date beginDate, Date endDate) {
        long begin = beginDate.getTime();
        long end = endDate.getTime();
        return (int) Math.ceil((end - begin));
    }

    static public String[] convertToStringArray(Object[] datas) {
        String[] params = new String[datas.length];
        System.arraycopy(datas, 0, params, 0, datas.length);
        return params;
    }

    static public int pareInt(String s) {
        return s == null || "".equals(s) ? 0 : Integer.parseInt(s);
    }

    public static List<List<String>> reformatData(List<String[]> datas) {
        List<List<String>> datasResult = new ArrayList<List<String>>();
        List<String> etapeTopiaIds = new ArrayList<String>();
        List<String> eleveurTopiaIds = new ArrayList<String>();
        List<String> tourTopiaIds = new ArrayList<String>();
        List<String> camionTopiaIds = new ArrayList<String>();
        for (String[] data : datas) {
            etapeTopiaIds.add(data[0]);
            eleveurTopiaIds.add(data[1]);
            tourTopiaIds.add(data[2]);
            camionTopiaIds.add(data[3]);
        }
        datasResult.add(etapeTopiaIds);
        datasResult.add(eleveurTopiaIds);
        datasResult.add(tourTopiaIds);
        datasResult.add(camionTopiaIds);

        return datasResult;
    }
                  
    // Creation de la date -7j
    public static Date getLastWeekDate(){
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(Calendar.DATE, -7);

//        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
//        return sdf.format(cal.getTime());
        return cal.getTime();
    }

    public static String convertEntitiesToListForNotifier(Collection<? extends TopiaEntity> entities) {
        return convertToListForNotifier(extractIds(entities));
    }

    public static String convertToListForNotifier(Collection<?> list) {
        return StringUtils.join(list, "&&") + "&&";
    }

    public static String convertToSubListForNotifier(Collection<?> list) {
        return StringUtils.join(list, "##") + "##";
    }

    public static <K, V> String convertToListOfListForNotifier(ListMultimap<K, V> multimap) {
        List<String> list = new ArrayList<String>();
        for (K key : multimap.keySet()) {
            List<V> values = multimap.get(key);
            String subList = convertToSubListForNotifier(values);
            list.add(subList);
        }
        String result = convertToListForNotifier(list);
        return result;
    }

    public static List<String> extractIds(Collection<? extends TopiaEntity> entities) {
        List<String> result = new ArrayList<String>();
        for (TopiaEntity entity : entities) {
            result.add(entity.getTopiaId());
        }
        return result;
    }

    public static String convertDateToString(Date d){
        Calendar c = Calendar.getInstance();
        c.setTime(d);

        return c.get(Calendar.DAY_OF_MONTH) + "/"
                + c.get(Calendar.MONTH)
                + "/" + c.get(Calendar.YEAR)
                + " à " + c.get(Calendar.HOUR_OF_DAY)
                + " h " + c.get(Calendar.MINUTE);
    }

    public static int getWorstSecurityLevel(Eleveur breeder) {
        return getWorstInfoAccessSecurityLevel(breeder).getNiveauSecurite();
    }

    public static InfoAccess getWorstInfoAccessSecurityLevel(Eleveur breeder) {

        InfoAccess result = breeder.getAccesEleveur();
        Collection<InfoAccess> ass = breeder.getActiveAccesSilo();

        int worstSecurityLevel = result.getNiveauSecurite();

        for (InfoAccess as : ass) {
            if (as != null) {
                int siloSecurity = as.getNiveauSecurite();
                if (worstSecurityLevel < siloSecurity) {
                    worstSecurityLevel = siloSecurity;
                    result = as;
                }
            }
        }
        return result;
    }

    public static UserIndicateurs razUserIndicateurs(UserIndicateurs userIndicateurs) {

        // RAZ user indicators
        userIndicateurs.setNbMaj(0);
        userIndicateurs.setNbGpsMaj(0);
        userIndicateurs.setNbSecuMaj(0);
        userIndicateurs.setNbSynchGPRS(0);
        userIndicateurs.setNbSynchWifi(0);
        userIndicateurs.setNbSynchKo(0);
        userIndicateurs.setNbNotifs(0);
        userIndicateurs.setNbOctetsReceivedGPRS(0);
        userIndicateurs.setNbOctetsSendGPRS(0);
        userIndicateurs.setNbOctetsReceivedWifi(0);
        userIndicateurs.setNbOctetsSendWifi(0);

        return userIndicateurs;
    }

    public static String addOrder(String alias, String sortCriterion, int sortDirection) {
        return addOrder(alias, false, sortCriterion, sortDirection);
    }

    public static String addOrder(String alias, boolean isNull, String sortCriterion, int sortDirection) {
        String criteria = alias + "." + sortCriterion;
        if (!isNull) {
            return " ORDER BY " + criteria + " " + (sortDirection == 1 ? "DESC" : "ASC");
        } else {
            return " ORDER BY CASE WHEN " + criteria + " IS NULL THEN 1 ELSE 0 END, " +
                    criteria + " " + (sortDirection == 1 ? "DESC" : "ASC");
        }
    }

    public static UserIndicateurs createUserIndicateurIfDontExist(TopiaContext transaction, TaasUser user) throws TopiaException {

       // Creating user indicator if not existing
       UserIndicateursDAO userIndicateursDAO = SandraDAOHelper.getUserIndicateursDAO(transaction);

       UserIndicateurs userIndicateurs = userIndicateursDAO.findByTaasUser(user);
       if (userIndicateurs == null) {
           userIndicateurs = userIndicateursDAO.createByNaturalId(user);
           userIndicateurs = SandraHelper.razUserIndicateurs(userIndicateurs);
           userIndicateursDAO.update(userIndicateurs);
       }
       return userIndicateurs;
    }

    public static List<TaasUser> getAllCamionUser(TopiaContext transaction) throws TopiaException {
        String hql = "SELECT user FROM " + TaasUser.class.getName() + " user" +
                " JOIN user.principals AS principal WHERE" +
                    " principal.name = 'synchro' AND user.login != 'synchro'";

        List<TaasUser> users = transaction.findAll(hql);
        for (TaasUser user : users) {
            //fixme : pb de lazy...
            user.getPrincipals().size();
        }
        return users;
    }

    public static void addSuiviTourCSV(Tour tour, SuiviTour suiviTour) throws IOException {

        // do nothing if no km is found, more than 1000 or if one of kmDepart or kmArrivee is null : SDRAMTNC-283
        if (suiviTour.getKmDepart() <= 0 || suiviTour.getKmArrivee() <= 0) {
            return;
        }

        // Km total
        int kmTotal = suiviTour.getKmTotal();
        if (kmTotal <= 0 || kmTotal > 1000) {
            return;
        }

        // Get work dir
        ApplicationConfig config = SandraConfig.getConfig();
        String suiviTourPath = SandraSchedulerConfigHelper.getSuiviTourPath(config);

        // Get file
        SimpleDateFormat format = new SimpleDateFormat("yyMMddHHmmss");
        String now = format.format(new Date());
        File file = new File(suiviTourPath + File.separator + tour.getCamion().getSociete().getCode() + SUIVI_TOUR_PREFIX + now + SUIVI_TOUR_EXTENTION);

        // Creating line to add : codeSociete-codeCamion;numeroTour;dateLivraisonTour;kmTotal
        Camion camion = tour.getCamion();
        String codeSociete = camion.getSociete().getCode();
        String codeCamion = camion.getCode();

        // Tour info
        int numeroTour = tour.getNumero();
        Date dateLivraisonTour = tour.getDateLivraison();
        SimpleDateFormat dateLivraisonPatern = new SimpleDateFormat("dd/MM/yyyy");
        String dateLivraisonAsString = dateLivraisonPatern.format(dateLivraisonTour);

        Date duree = suiviTour.getDuree();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(duree);
        
        String dureeAsString = "0";
        if (duree != null) {
            float hour = calendar.get(Calendar.HOUR);
            hour += calendar.get(Calendar.MINUTE) / 60;
            dureeAsString = new DecimalFormat("00.#0").format(hour);
        }

        // Litre gasoil
        int litresGasoil = suiviTour.getLitresGasoil();

        String lineToWrite = dateLivraisonAsString + DELIM + "\"" + codeSociete + "-" + codeCamion + "\"" + DELIM + numeroTour + DELIM + kmTotal + DELIM + dureeAsString + DELIM + litresGasoil + "\n";

        FileWriter writer = null;

        try {
            // Creating new one
            if (!file.exists()) {

                writer = new FileWriter(file, true);

                // Add header : CAM_GROUP;TOU;DAT_LIV;KM
                writer.write("DAT_LIV;CAM_GROUP;TOU;KM;h_REEL;L_GO\n");
            }

            if (writer == null) {
                writer = new FileWriter(file, true);
            }

            // Add souvi tour line
            writer.write(lineToWrite);
        } finally {
            if (writer != null) {
                writer.close();
            }
        }
    }
    
    public static Collection<InfoAccess> mergeEleveurInfoAccess(Eleveur oldEleveur, Eleveur eleveur) {
        final Map<String, InfoAccess> silos = extractInfoAccess(eleveur.getAccesSilos());

        Extractor<InfoAccess, Map<String, InfoAccess>> infoAccessMerge = new Extractor<InfoAccess, Map<String, InfoAccess>>() {

            @Override
            public void extract(InfoAccess value) {
                silos.put(value.getNomAcces(), value);
            }

            @Override
            public Map<String, InfoAccess> getResult() {
                return silos;
            }
        };

        for (InfoAccess oldSilo : oldEleveur.getAccesSilos()) {
            infoAccessMerge.extract(oldSilo);
        }
        return infoAccessMerge.getResult().values();
    }
    
    public static Map<String, InfoAccess> extractInfoAccess(Collection<InfoAccess> silos) {


        SimpleMapExtractor<String, InfoAccess> infoAccessExtractor = new SimpleMapExtractor<String, InfoAccess>() {

            @Override
            protected String getKey(InfoAccess value) {
                return value.getNomAcces();
            }
        };

        if (silos != null) {
            for (InfoAccess silo : silos) {
                infoAccessExtractor.extract(silo);
            }
        }

        return infoAccessExtractor.getResult();
    }

    public static String getLogTour(Tour tour, String additionnal) {
        SimpleDateFormat format = new SimpleDateFormat("dd/MM/yyyy");

        String formatedDate = null;

        if (tour.getDateLivraison() != null) {
            formatedDate = format.format(tour.getDateLivraison());
        }
        return "tour : topiaId[" + tour.getTopiaId() + "] num[" + tour.getNumero() + "] dateLivraison[" + formatedDate + "] " + additionnal;
    }

    public static String getEleveurTitle(Eleveur eleveur) {
        return getEleveurTitle(eleveur.getRaisonSociale(), eleveur.getSociete().getCode(), eleveur.getCode(), eleveur.getVille(), eleveur.getCodePostal());
    }

    public static String getEleveurTitle(String raisonSocial, String codeSociete, String codeEleveur, String ville, String codePostal) {
        String title = "";
        String department = SandraHelper.getDepartment(codePostal);
        title += raisonSocial +
                " (" + codeSociete + "-" + codeEleveur + ") " +
                ville;
        if (department != null) {
            title += " (" + department + ")";
        }
        return title;
    }

    public static String getDepartment(Eleveur eleveur) {
        return getDepartment(eleveur.getCodePostal());
    }

    public static String getDepartment(String codePostal) {

        String department = null;
        if (codePostal != null) {
            if (codePostal.length() == 4) {
                department = "0" + codePostal.substring(0, 1);
            } else if (codePostal.length() == 5) {
                department = codePostal.substring(0, 2);
            }
        }
        return department;
    }

    public static void deleteEtape(TopiaContext transaction, Etape etape) throws TopiaException {

        Tour tour = etape.getTour();

        EtapeDAO etapeDAO = SandraDAOHelper.getEtapeDAO(transaction);
        SuiviLigneProduitDAO suiviLigneProduitDAO = SandraDAOHelper.getSuiviLigneProduitDAO(transaction);

        SuiviTour suiviTour = tour.getSuiviTour();
        if (suiviTour != null) {
            for (SuiviEtape suiviEtape : suiviTour.getSuiviEtapes()) {
                suiviEtape.setEtape(null);
            }
            for (SuiviUsine suiviUsine : suiviTour.getSuiviUsines()) {
                for (SuiviLigneProduit suiviLigneProduit : suiviUsine.getSuiviLigneProduit()) {
                    suiviLigneProduit.setLigneProduit(null);
                }
            }
            tour.setSuiviTour(null);
        }

        // remove missing
        for (LigneProduit produit : etape.getProduits()) {
            List<SuiviLigneProduit> suiviLigneProduits = suiviLigneProduitDAO.findAllByLigneProduit(produit);
            for (SuiviLigneProduit suiviLigneProduit : suiviLigneProduits) {
                suiviLigneProduit.setLigneProduit(null);
            }
        }
        tour.removeEtapes(etape);

        etapeDAO.delete(etape);

        if (suiviTour != null) {
            SuiviTourDAO suiviTourDAO = SandraDAOHelper.getSuiviTourDAO(transaction);
            suiviTourDAO.delete(suiviTour);
        }
    }

    public static void checkDuplicateBreeder(TopiaContext transaction, List<Eleveur> eleveurs, String template) throws TopiaException {

        for (Eleveur eleveur : eleveurs) {
            // call sql method
            List<String> duplicateEleveurIds = ((TopiaContextImpl) transaction).getHibernate().createSQLQuery("SELECT find_duplicate_breeder('" + eleveur.getTopiaId() + "');").list();
            if (!duplicateEleveurIds.isEmpty() && duplicateEleveurIds.get(0) != null) {

                // convert to string for notifier
                String duplicateEleveurIdsString = SandraHelper.convertToListForNotifier(duplicateEleveurIds);

                // send notification
                ApplicationConfig config = SandraConfig.getConfig();
                new SandraNotifier().notifyEvent(template,
                        SandraSchedulerConfigHelper.getSandraName(config),
                        SandraSchedulerConfigHelper.getSandraUrl(config),
                        eleveur.getTopiaId(),
                        duplicateEleveurIdsString);
            }
        }
    }

    public static void createActionSecuriteIfNeeded(TopiaContext transaction, String userLogin, Eleveur eleveur, InfoAccess infoAccess) throws TopiaException {

        // TMA 201 : https://jira.groupe-glon.fr/jira/browse/SDRAMTNC-201
        // Create default action security if securityLevel = orange, red or black

        // TMA 371 : https://jira.groupe-glon.fr/browse/SDRAMTNC-371
        // Create only for silo

        // Search if already exist
        String type = "A001";

        // Empty creation
        ActionSecuriteDAO actionSecuriteDAO = SandraDAOHelper.getActionSecuriteDAO(transaction);
        EleveurDAO eleveurDAO = SandraDAOHelper.getEleveurDAO(transaction);

        ActionSecurite actionSecurite = actionSecuriteDAO.create();
        actionSecurite.setNiveauSecurite(infoAccess.getNiveauSecurite());
        actionSecurite.setType(type);
        actionSecurite.setDateAction(new Date());
        actionSecurite.setUserCreation(userLogin);

        // only for silo
        if (!eleveur.getAccesEleveur().equals(infoAccess)) {
            actionSecurite.setInfoAccess(infoAccess);
        }

        actionSecurite.setCommentaire(infoAccess.getCommentaireSecurite());

        eleveur.addActionSecurite(actionSecurite);

        actionSecuriteDAO.update(actionSecurite);
        eleveurDAO.update(eleveur);

        log.info("Creation of default action security for eleveur " + eleveur.getNom());
    }

    protected interface Extractor<T, E> {
        public void extract(T value);

        public E getResult();
    }

    protected static class SimpleListExtractor<T> extends ListExtractor<T, T> {

        @Override
        protected T getElement(T value) {
            return value;
        }
    }

    protected static abstract class ListExtractor<T, R> implements Extractor<T, List<R>> {

        protected List<R> values;

        protected ListExtractor() {
            values = Lists.newArrayList();
        }

        public void extract(T value) {
            values.add(getElement(value));
        }

        public List<R> getResult() {
            return values;
        }

        protected abstract R getElement(T value);
    }

    protected static abstract class SimpleMapExtractor<K, T> extends MapExtractor<K, T, T> {

        protected abstract K getKey(T value);

        @Override
        protected T getElement(T value) {
            return value;
        }
    }

    protected static abstract class MapExtractor<K, T, R> implements Extractor<T, Map<K, R>> {

        protected Map<K, R> values;

        protected MapExtractor() {
            values = Maps.newHashMap();
        }

        @Override
        public void extract(T value) {
            values.put(getKey(value), getElement(value));
        }

        @Override
        public Map<K, R> getResult() {
            return values;
        }

        protected abstract K getKey(T value);

        protected abstract R getElement(T value);
    }
}
