/*
 * #%L
 * Refcomp :: services
 * 
 * $Id: ServiceFactory.java 263 2012-02-09 13:10:48Z jcouteau $
 * $HeadURL: svn+ssh://jcouteau@scm.mulcyber.toulouse.inra.fr/svnroot/refcomp/tags/refcomp-1.7/refcomp-services/src/main/java/fr/inra/refcomp/services/ServiceFactory.java $
 * %%
 * Copyright (C) 2010 Inra
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.inra.refcomp.services;

import fr.inra.refcomp.entities.AgentImpl;
import fr.inra.refcomp.entities.AgentSkillImpl;
import fr.inra.refcomp.entities.CatiImpl;
import fr.inra.refcomp.entities.DepartmentImpl;
import fr.inra.refcomp.entities.FrequencyImpl;
import fr.inra.refcomp.entities.NatureImpl;
import fr.inra.refcomp.entities.SkillImpl;
import fr.inra.refcomp.entities.UnitDeptLinkImpl;
import fr.inra.refcomp.entities.UnitImpl;
import fr.inra.refcomp.entities.UserImpl;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyClient;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceFactory;
import org.nuiton.wikitty.addons.WikittyImportExportService;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyTreeNodeImpl;
import org.nuiton.wikitty.entities.WikittyUserImpl;

import java.util.Arrays;
import java.util.List;

/**
 * User: couteau
 * Date: 26 nov. 2010
 */
public class ServiceFactory {

    /**
     * Log.
     */
    private static final Log log = LogFactory.getLog(ServiceFactory.class);

    /**
     * Wikitty service (cached version) for remote access.
     */
    protected static WikittyService wikittyService;

    /**
     * Single instance of importExport service.
     */
    protected static WikittyImportExportService importExportService;

    protected static RefcompConfiguration config;

    /**
     * Creates wikitty service (cached).
     * <p/>
     * Constructor added for UI embedded mode, with custom configuration.
     *
     * @return a unique instance of WikittyService
     */
    public static WikittyService getWikittyService() {
        if (wikittyService == null) {
            synchronized (ServiceFactory.class) {
                if(wikittyService == null) {
                    wikittyService = WikittyServiceFactory.buildWikittyService(getConfig().applicationConfig);
                }
            }
        }

        return wikittyService;
    }

    public static RefcompConfiguration getConfig(String... args) {
        if (config == null) {
            synchronized (ServiceFactory.class) {
                if (config == null) {
                    config = new RefcompConfiguration(args);
                }
            }
        }

        return config;
    }

    /**
     * Creates wikitty service (cached).
     * <p/>
     * Constructor added for UI embedded mode, with custom configuration.
     *
     * @return a unique instance of WikittyService
     */
    public static WikittyImportExportService getImportExportService(WikittyClient client) {
        if (importExportService == null) {
            synchronized (ServiceFactory.class) {
                if (importExportService == null) {
                    importExportService = new WikittyImportExportService(config.applicationConfig,
                            client.getSecurityToken(),
                            client.getWikittyService());
                }
            }
        }

        return importExportService;
    }

    public static StorageService getService(RefcompSession session) {
        StorageService service = session.getService();
        if (service == null) {
            if (log.isDebugEnabled()) {
                log.debug("Initiate service");
            }
            WikittyClient client = new WikittyClient(getConfig().applicationConfig, getWikittyService());
            updateExtensions(client);
            service = new StorageServiceImpl(client, getConfig());
            session.setService(service);
        }
        return service;
    }

    /**
     * Register all extensions.
     * <p/>
     * Already existing extensions with same version are skipped,
     * new extension are created and stored.
     *
     * @param wikittyClient wikitty client
     */
    protected static void updateExtensions(WikittyClient wikittyClient) {

        // easier if ordered
        List<WikittyExtension> extensions = Arrays.asList(
                AgentImpl.extensionAgent,
                AgentSkillImpl.extensionAgentSkill,
                CatiImpl.extensionCati,
                DepartmentImpl.extensionDepartment,
                FrequencyImpl.extensionFrequency,
                NatureImpl.extensionNature,
                SkillImpl.extensionSkill,
                UnitImpl.extensionUnit,
                UnitDeptLinkImpl.extensionUnitDeptLink,
                UserImpl.extensionUser,
                WikittyTreeNodeImpl.extensionWikittyTreeNode,
                WikittyUserImpl.extensionWikittyUser);

        wikittyClient.storeExtension(extensions);
        if (log.isDebugEnabled()){
            log.debug("Migration done");
        }
    }

}
