/*
 * #%L
 * Refcomp :: services
 * 
 * $Id: RefcompConfiguration.java 156 2011-04-14 15:23:17Z jcouteau $
 * $HeadURL: svn+ssh://jcouteau@scm.mulcyber.toulouse.inra.fr/svnroot/refcomp/tags/refcomp-1.0/refcomp-services/src/main/java/fr/inra/refcomp/services/RefcompConfiguration.java $
 * %%
 * Copyright (C) 2010 Inra
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.inra.refcomp.services;

import java.io.File;
import java.util.Locale;
import org.apache.commons.lang.UnhandledException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyConfig;
import org.nuiton.wikitty.WikittyConfigOption;

import static org.nuiton.i18n.I18n._;

/**
 * User: couteau
 * Date: 26 nov. 2010
 */
public class RefcompConfiguration extends ApplicationConfig {

    /**
     * log.
     */
    private static final Log log = LogFactory.getLog(RefcompConfiguration.class);

    /**
     * After init step.
     */
    public static final int STEP_AFTER_INIT = 0;

    /**
     * Singleton instance.
     */
    protected static RefcompConfiguration instance = null;

    public static final String DATA_DEFAULT_DIR = "${user.home}" + File.separator + ".refcomp" +
            File.separator + "refcomp-${application.version}";

    protected RefcompConfiguration() {
        super();

        // Load defaults wikitty options
        loadDefaultOptions(WikittyConfigOption.class);

        // Load defaults refcomp options
        loadDefaultOptions(RefcompOption.class);

        // Load actions
        loadActions(RefcompServiceActionDef.class);

    }

    public static synchronized RefcompConfiguration getInstance(String... args) {
        if (instance == null) {
            try {
                instance = new RefcompConfiguration();
                instance.parse(args);
                // was done in parse before (but not good for tests)
                instance.setSystemDefaultProps();
                if (log.isDebugEnabled()){
                    log.debug("Configuration initialized");
                }
            } catch (Exception ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't intitialize configuration", ex);
                }
                throw new UnhandledException(ex);
            }
        }
        return instance;
    }

    /**
     * Set {@code solr} and {@code jms} system configuration.
     * <p/>
     * This is the "only" way to configure embedded solr.
     */
    protected void setSystemDefaultProps() {
        String[] datadirs = new String[]{"solr.data.dir", "refcomp.objectStore.dir"};

        for (String datadir : datadirs) {
            String value = System.getProperty(datadir, null);
            if (value == null) {
                value = getOption(datadir);
                if (log.isInfoEnabled()) {
                    log.info("Setting system property " + datadir + " : " + value);
                }
                if (value != null) {
                    System.setProperty(datadir, value);
                    env.put(datadir, value);
                }
            }
        }
    }

    /**
     * Vradi option definition.
     */
    public static enum RefcompOption implements OptionDef {

        ADMIN_EMAIL(
                "refcomp.admin.email",
                _("refcomp.config.admin.email.description"),
                "couteau@codelutin.com", String.class, false, false),

        ADMIN_LOGIN(
                "refcomp.admin.login",
                _("refcomp.config.admin.login.description"),
                "admin", String.class, false, false),

        ADMIN_PASSWORD(
                "refcomp.admin.password",
                _("refcomp.config.admin.password.description"),
                "admin", String.class, false, false),

        CONFIG_FILE(
                CONFIG_FILE_NAME,
                _("refcomp.config.configFileName.description"),
                "refcomp.properties",
                String.class, true, true),

        APPLICATION_VERSION(
                "application.version",
                _("refcomp.config.application.version.description"),
                null, String.class, false, false),

        DATA_DIR(
                "refcomp.data.dir",
                _("refcomp.config.data.dir.description"),
                DATA_DEFAULT_DIR,
                String.class, false, false),

        LOCALE(
                "refcomp.locale",
                _("refcomp.config.ui.locale.description"),
                Locale.FRANCE.toString(), Locale.class, false, false);

        public String key;
        public String description;
        public String defaultValue;
        public Class<?> type;
        public boolean isTransient;
        public boolean isFinal;

        private RefcompOption(String key, String description, String defaultValue,
                                   Class<?> type, boolean isTransient, boolean isFinal) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this.isFinal = isFinal;
            this.isTransient = isTransient;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public Class<?> getType() {
            return type;
        }

        @Override
        public String getDescription() {
            return description;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public boolean isTransient() {
            return isTransient;
        }

        @Override
        public boolean isFinal() {
            return isFinal;
        }

        @Override
        public void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public void setTransient(boolean isTransient) {
            this.isTransient = isTransient;
        }

        @Override
        public void setFinal(boolean isFinal) {
            this.isFinal = isFinal;
        }

        private static String getDefaultDir(String dirName) {
            return DATA_DEFAULT_DIR + File.separator + dirName;
        }
    }
}
